/*
 *  Number.prototype.toString() tests.
 *
 *  The specification doesn't state an exact sequence of operations for
 *  Number.prototype.toString() (e.g. order of radix coercion vs. 'this'
 *  coercion).
 *
 *  Tests are custom because no particular precision is required.
 *
 *  Expected values are probably not 100% what we want: they're quite
 *  challenging to generate:
 *
 *    - V8 3.7.12.22 has good (and probably optimal) output for bases
 *      10, 2, 4, 8, 16, 32.  Output for other bases is insanely long.
 *
 *    - Rhino has short output for all bases, but not sure if they're
 *      100% correct (shortest, correctly rounded).  Rhino also has a
 *      weird sign bug shown below.
 *
 *  Expect strings are now from V8 for bases 10 and 2^N, and from Rhino
 *  for other bases (with signs fixed manually).
 *
 *  Example of base 15 output from V8 and Rhino:
 *
 *      // V8
 *      > Math.PI.toString(16)
 *      '3.243f6a8885a3'
 *      > Math.PI.toString(15)
 *      '3.21cd1dc46c2b7ab624ee5cd3a5322906081b7e4cc88225384e4053d70db95bb16341a265dd07b4135d0200c2bcb53d7dea9e200e326c298b6402c6494d74277952a56ac43cc56ac8a96c900676c6ddd99b6dba844a045458617d04311dc66aa92e5e7d04e05d0eb61d084e42a1484e3a0ed0e079965dba408810e47dc77679ccb2c4b46a6c324d6e278c18a62ca3d276b55bc6089e58c1c8a50c6590c4c2d756427cb1d54c5d336ad2128ebde01b3a7ab7986996de5b0c7d0e03ec2273b2647a8d30cd810ca3ea18c0943a74c05b59079b5d8026e4755e9d7697e53b34d52175eea738e62c711caa026c3a0ea7d86ca2718592d8405ee96e49ac62616b745c574beab91b3db9646611e7843184aee4e4c328e477656b2c0950362c748e61b513d5261b53c63bd74813a1d4ea45868d3db6638cecea93a65bda38d5b4172716748473a45d6c2ecc419737b0c7ccac53d2db904cc0797a6559d6ead3ca60d87c47a04e9d56ab82c926c7685193312971ea5e2aa9b28d5c85ba9cc344c0e96262211835e9b731cda1a34932a882487d7193c4b21103c61e833b1037ed0a44185e7738c551757d5367652c958e3820031ab04e027530c43e174c7bd29959c4095bb7e7db32c443cb6cd11cd282a5a37a662687cac2422be22b65da7a9aed10db8034a667c46318ca535e4049c946e480a893a73b988d85e807d19e0b2b70840e4003506a234d966ae83c455e80c5e63777a75d5551d78ab922cea57e2ab9ba6a402e4b23e2ee776'
 *
 *      // Rhino
 *      js> Math.PI.toString(15)
 *      3.21cd1dc46c2b8
 *
 *      // Duktape
 *      ((o) Duktape 0.12.99
 *      duk> Math.PI.toString(15)
 *      = 3.21cd1dc46c2b8
 *
 *  Rhino seems to have a weird sign bug:
 *
 *      Rhino 1.7 release 4 2013 08 27
 *      js> (-0.9).toString(18)
 *  (!) 0.g3ae73ae73ae8
 *      js> (-0.9).toString(10)
 *      -0.9
 *
 *  These signs have been fixed manually in the test expect output.
 */

/*---
{
    "custom": true
}
---*/

function test(this_value, args) {
    var t;

    try {
        t = Number.prototype.toString.apply(this_value, args);
        print(typeof t, t);
    } catch (e) {
        print(e.name);
    }
}

/*===
basic
string 12345
string 12345
RangeError
RangeError
RangeError
2 -1100100 -1100011 -1100010 -1100001 -1100000 -1011111 -1011110 -1011101 -1011100 -1011011 -1011010 -1011001 -1011000 -1010111 -1010110 -1010101 -1010100 -1010011 -1010010 -1010001 -1010000 -1001111 -1001110 -1001101 -1001100 -1001011 -1001010 -1001001 -1001000 -1000111 -1000110 -1000101 -1000100 -1000011 -1000010 -1000001 -1000000 -111111 -111110 -111101 -111100 -111011 -111010 -111001 -111000 -110111 -110110 -110101 -110100 -110011 -110010 -110001 -110000 -101111 -101110 -101101 -101100 -101011 -101010 -101001 -101000 -100111 -100110 -100101 -100100 -100011 -100010 -100001 -100000 -11111 -11110 -11101 -11100 -11011 -11010 -11001 -11000 -10111 -10110 -10101 -10100 -10011 -10010 -10001 -10000 -1111 -1110 -1101 -1100 -1011 -1010 -1001 -1000 -111 -110 -101 -100 -11 -10 -1 0 1 10 11 100 101 110 111 1000 1001 1010 1011 1100 1101 1110 1111 10000 10001 10010 10011 10100 10101 10110 10111 11000 11001 11010 11011 11100 11101 11110 11111 100000 100001 100010 100011 100100 100101 100110 100111 101000 101001 101010 101011 101100 101101 101110 101111 110000 110001 110010 110011 110100 110101 110110 110111 111000 111001 111010 111011 111100 111101 111110 111111 1000000 1000001 1000010 1000011 1000100 1000101 1000110 1000111 1001000 1001001 1001010 1001011 1001100 1001101 1001110 1001111 1010000 1010001 1010010 1010011 1010100 1010101 1010110 1010111 1011000 1011001 1011010 1011011 1011100 1011101 1011110 1011111 1100000 1100001 1100010 1100011 1100100
3 -10201 -10200 -10122 -10121 -10120 -10112 -10111 -10110 -10102 -10101 -10100 -10022 -10021 -10020 -10012 -10011 -10010 -10002 -10001 -10000 -2222 -2221 -2220 -2212 -2211 -2210 -2202 -2201 -2200 -2122 -2121 -2120 -2112 -2111 -2110 -2102 -2101 -2100 -2022 -2021 -2020 -2012 -2011 -2010 -2002 -2001 -2000 -1222 -1221 -1220 -1212 -1211 -1210 -1202 -1201 -1200 -1122 -1121 -1120 -1112 -1111 -1110 -1102 -1101 -1100 -1022 -1021 -1020 -1012 -1011 -1010 -1002 -1001 -1000 -222 -221 -220 -212 -211 -210 -202 -201 -200 -122 -121 -120 -112 -111 -110 -102 -101 -100 -22 -21 -20 -12 -11 -10 -2 -1 0 1 2 10 11 12 20 21 22 100 101 102 110 111 112 120 121 122 200 201 202 210 211 212 220 221 222 1000 1001 1002 1010 1011 1012 1020 1021 1022 1100 1101 1102 1110 1111 1112 1120 1121 1122 1200 1201 1202 1210 1211 1212 1220 1221 1222 2000 2001 2002 2010 2011 2012 2020 2021 2022 2100 2101 2102 2110 2111 2112 2120 2121 2122 2200 2201 2202 2210 2211 2212 2220 2221 2222 10000 10001 10002 10010 10011 10012 10020 10021 10022 10100 10101 10102 10110 10111 10112 10120 10121 10122 10200 10201
4 -1210 -1203 -1202 -1201 -1200 -1133 -1132 -1131 -1130 -1123 -1122 -1121 -1120 -1113 -1112 -1111 -1110 -1103 -1102 -1101 -1100 -1033 -1032 -1031 -1030 -1023 -1022 -1021 -1020 -1013 -1012 -1011 -1010 -1003 -1002 -1001 -1000 -333 -332 -331 -330 -323 -322 -321 -320 -313 -312 -311 -310 -303 -302 -301 -300 -233 -232 -231 -230 -223 -222 -221 -220 -213 -212 -211 -210 -203 -202 -201 -200 -133 -132 -131 -130 -123 -122 -121 -120 -113 -112 -111 -110 -103 -102 -101 -100 -33 -32 -31 -30 -23 -22 -21 -20 -13 -12 -11 -10 -3 -2 -1 0 1 2 3 10 11 12 13 20 21 22 23 30 31 32 33 100 101 102 103 110 111 112 113 120 121 122 123 130 131 132 133 200 201 202 203 210 211 212 213 220 221 222 223 230 231 232 233 300 301 302 303 310 311 312 313 320 321 322 323 330 331 332 333 1000 1001 1002 1003 1010 1011 1012 1013 1020 1021 1022 1023 1030 1031 1032 1033 1100 1101 1102 1103 1110 1111 1112 1113 1120 1121 1122 1123 1130 1131 1132 1133 1200 1201 1202 1203 1210
5 -400 -344 -343 -342 -341 -340 -334 -333 -332 -331 -330 -324 -323 -322 -321 -320 -314 -313 -312 -311 -310 -304 -303 -302 -301 -300 -244 -243 -242 -241 -240 -234 -233 -232 -231 -230 -224 -223 -222 -221 -220 -214 -213 -212 -211 -210 -204 -203 -202 -201 -200 -144 -143 -142 -141 -140 -134 -133 -132 -131 -130 -124 -123 -122 -121 -120 -114 -113 -112 -111 -110 -104 -103 -102 -101 -100 -44 -43 -42 -41 -40 -34 -33 -32 -31 -30 -24 -23 -22 -21 -20 -14 -13 -12 -11 -10 -4 -3 -2 -1 0 1 2 3 4 10 11 12 13 14 20 21 22 23 24 30 31 32 33 34 40 41 42 43 44 100 101 102 103 104 110 111 112 113 114 120 121 122 123 124 130 131 132 133 134 140 141 142 143 144 200 201 202 203 204 210 211 212 213 214 220 221 222 223 224 230 231 232 233 234 240 241 242 243 244 300 301 302 303 304 310 311 312 313 314 320 321 322 323 324 330 331 332 333 334 340 341 342 343 344 400
6 -244 -243 -242 -241 -240 -235 -234 -233 -232 -231 -230 -225 -224 -223 -222 -221 -220 -215 -214 -213 -212 -211 -210 -205 -204 -203 -202 -201 -200 -155 -154 -153 -152 -151 -150 -145 -144 -143 -142 -141 -140 -135 -134 -133 -132 -131 -130 -125 -124 -123 -122 -121 -120 -115 -114 -113 -112 -111 -110 -105 -104 -103 -102 -101 -100 -55 -54 -53 -52 -51 -50 -45 -44 -43 -42 -41 -40 -35 -34 -33 -32 -31 -30 -25 -24 -23 -22 -21 -20 -15 -14 -13 -12 -11 -10 -5 -4 -3 -2 -1 0 1 2 3 4 5 10 11 12 13 14 15 20 21 22 23 24 25 30 31 32 33 34 35 40 41 42 43 44 45 50 51 52 53 54 55 100 101 102 103 104 105 110 111 112 113 114 115 120 121 122 123 124 125 130 131 132 133 134 135 140 141 142 143 144 145 150 151 152 153 154 155 200 201 202 203 204 205 210 211 212 213 214 215 220 221 222 223 224 225 230 231 232 233 234 235 240 241 242 243 244
7 -202 -201 -200 -166 -165 -164 -163 -162 -161 -160 -156 -155 -154 -153 -152 -151 -150 -146 -145 -144 -143 -142 -141 -140 -136 -135 -134 -133 -132 -131 -130 -126 -125 -124 -123 -122 -121 -120 -116 -115 -114 -113 -112 -111 -110 -106 -105 -104 -103 -102 -101 -100 -66 -65 -64 -63 -62 -61 -60 -56 -55 -54 -53 -52 -51 -50 -46 -45 -44 -43 -42 -41 -40 -36 -35 -34 -33 -32 -31 -30 -26 -25 -24 -23 -22 -21 -20 -16 -15 -14 -13 -12 -11 -10 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 10 11 12 13 14 15 16 20 21 22 23 24 25 26 30 31 32 33 34 35 36 40 41 42 43 44 45 46 50 51 52 53 54 55 56 60 61 62 63 64 65 66 100 101 102 103 104 105 106 110 111 112 113 114 115 116 120 121 122 123 124 125 126 130 131 132 133 134 135 136 140 141 142 143 144 145 146 150 151 152 153 154 155 156 160 161 162 163 164 165 166 200 201 202
8 -144 -143 -142 -141 -140 -137 -136 -135 -134 -133 -132 -131 -130 -127 -126 -125 -124 -123 -122 -121 -120 -117 -116 -115 -114 -113 -112 -111 -110 -107 -106 -105 -104 -103 -102 -101 -100 -77 -76 -75 -74 -73 -72 -71 -70 -67 -66 -65 -64 -63 -62 -61 -60 -57 -56 -55 -54 -53 -52 -51 -50 -47 -46 -45 -44 -43 -42 -41 -40 -37 -36 -35 -34 -33 -32 -31 -30 -27 -26 -25 -24 -23 -22 -21 -20 -17 -16 -15 -14 -13 -12 -11 -10 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 10 11 12 13 14 15 16 17 20 21 22 23 24 25 26 27 30 31 32 33 34 35 36 37 40 41 42 43 44 45 46 47 50 51 52 53 54 55 56 57 60 61 62 63 64 65 66 67 70 71 72 73 74 75 76 77 100 101 102 103 104 105 106 107 110 111 112 113 114 115 116 117 120 121 122 123 124 125 126 127 130 131 132 133 134 135 136 137 140 141 142 143 144
9 -121 -120 -118 -117 -116 -115 -114 -113 -112 -111 -110 -108 -107 -106 -105 -104 -103 -102 -101 -100 -88 -87 -86 -85 -84 -83 -82 -81 -80 -78 -77 -76 -75 -74 -73 -72 -71 -70 -68 -67 -66 -65 -64 -63 -62 -61 -60 -58 -57 -56 -55 -54 -53 -52 -51 -50 -48 -47 -46 -45 -44 -43 -42 -41 -40 -38 -37 -36 -35 -34 -33 -32 -31 -30 -28 -27 -26 -25 -24 -23 -22 -21 -20 -18 -17 -16 -15 -14 -13 -12 -11 -10 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 10 11 12 13 14 15 16 17 18 20 21 22 23 24 25 26 27 28 30 31 32 33 34 35 36 37 38 40 41 42 43 44 45 46 47 48 50 51 52 53 54 55 56 57 58 60 61 62 63 64 65 66 67 68 70 71 72 73 74 75 76 77 78 80 81 82 83 84 85 86 87 88 100 101 102 103 104 105 106 107 108 110 111 112 113 114 115 116 117 118 120 121
10 -100 -99 -98 -97 -96 -95 -94 -93 -92 -91 -90 -89 -88 -87 -86 -85 -84 -83 -82 -81 -80 -79 -78 -77 -76 -75 -74 -73 -72 -71 -70 -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 31 32 33 34 35 36 37 38 39 40 41 42 43 44 45 46 47 48 49 50 51 52 53 54 55 56 57 58 59 60 61 62 63 64 65 66 67 68 69 70 71 72 73 74 75 76 77 78 79 80 81 82 83 84 85 86 87 88 89 90 91 92 93 94 95 96 97 98 99 100
11 -91 -90 -8a -89 -88 -87 -86 -85 -84 -83 -82 -81 -80 -7a -79 -78 -77 -76 -75 -74 -73 -72 -71 -70 -6a -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a 10 11 12 13 14 15 16 17 18 19 1a 20 21 22 23 24 25 26 27 28 29 2a 30 31 32 33 34 35 36 37 38 39 3a 40 41 42 43 44 45 46 47 48 49 4a 50 51 52 53 54 55 56 57 58 59 5a 60 61 62 63 64 65 66 67 68 69 6a 70 71 72 73 74 75 76 77 78 79 7a 80 81 82 83 84 85 86 87 88 89 8a 90 91
12 -84 -83 -82 -81 -80 -7b -7a -79 -78 -77 -76 -75 -74 -73 -72 -71 -70 -6b -6a -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b 10 11 12 13 14 15 16 17 18 19 1a 1b 20 21 22 23 24 25 26 27 28 29 2a 2b 30 31 32 33 34 35 36 37 38 39 3a 3b 40 41 42 43 44 45 46 47 48 49 4a 4b 50 51 52 53 54 55 56 57 58 59 5a 5b 60 61 62 63 64 65 66 67 68 69 6a 6b 70 71 72 73 74 75 76 77 78 79 7a 7b 80 81 82 83 84
13 -79 -78 -77 -76 -75 -74 -73 -72 -71 -70 -6c -6b -6a -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -5c -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 50 51 52 53 54 55 56 57 58 59 5a 5b 5c 60 61 62 63 64 65 66 67 68 69 6a 6b 6c 70 71 72 73 74 75 76 77 78 79
14 -72 -71 -70 -6d -6c -6b -6a -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -5d -5c -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 50 51 52 53 54 55 56 57 58 59 5a 5b 5c 5d 60 61 62 63 64 65 66 67 68 69 6a 6b 6c 6d 70 71 72
15 -6a -69 -68 -67 -66 -65 -64 -63 -62 -61 -60 -5e -5d -5c -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 50 51 52 53 54 55 56 57 58 59 5a 5b 5c 5d 5e 60 61 62 63 64 65 66 67 68 69 6a
16 -64 -63 -62 -61 -60 -5f -5e -5d -5c -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 50 51 52 53 54 55 56 57 58 59 5a 5b 5c 5d 5e 5f 60 61 62 63 64
17 -5f -5e -5d -5c -5b -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4g -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 4g 50 51 52 53 54 55 56 57 58 59 5a 5b 5c 5d 5e 5f
18 -5a -59 -58 -57 -56 -55 -54 -53 -52 -51 -50 -4h -4g -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 4g 4h 50 51 52 53 54 55 56 57 58 59 5a
19 -55 -54 -53 -52 -51 -50 -4i -4h -4g -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 4g 4h 4i 50 51 52 53 54 55
20 -50 -4j -4i -4h -4g -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 4g 4h 4i 4j 50
21 -4g -4f -4e -4d -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 40 41 42 43 44 45 46 47 48 49 4a 4b 4c 4d 4e 4f 4g
22 -4c -4b -4a -49 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3l -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 3l 40 41 42 43 44 45 46 47 48 49 4a 4b 4c
23 -48 -47 -46 -45 -44 -43 -42 -41 -40 -3m -3l -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 3l 3m 40 41 42 43 44 45 46 47 48
24 -44 -43 -42 -41 -40 -3n -3m -3l -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 3l 3m 3n 40 41 42 43 44
25 -40 -3o -3n -3m -3l -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 3l 3m 3n 3o 40
26 -3m -3l -3k -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j 3k 3l 3m
27 -3j -3i -3h -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g 3h 3i 3j
28 -3g -3f -3e -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d 3e 3f 3g
29 -3d -3c -3b -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 30 31 32 33 34 35 36 37 38 39 3a 3b 3c 3d
30 -3a -39 -38 -37 -36 -35 -34 -33 -32 -31 -30 -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 30 31 32 33 34 35 36 37 38 39 3a
31 -37 -36 -35 -34 -33 -32 -31 -30 -2u -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 2u 30 31 32 33 34 35 36 37
32 -34 -33 -32 -31 -30 -2v -2u -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1v -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -v -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u v 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 1v 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 2u 2v 30 31 32 33 34
33 -31 -30 -2w -2v -2u -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1w -1v -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -w -v -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u v w 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 1v 1w 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 2u 2v 2w 30 31
34 -2w -2v -2u -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1x -1w -1v -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -x -w -v -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u v w x 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 1v 1w 1x 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 2u 2v 2w
35 -2u -2t -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1y -1x -1w -1v -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -y -x -w -v -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u v w x y 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 1v 1w 1x 1y 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s 2t 2u
36 -2s -2r -2q -2p -2o -2n -2m -2l -2k -2j -2i -2h -2g -2f -2e -2d -2c -2b -2a -29 -28 -27 -26 -25 -24 -23 -22 -21 -20 -1z -1y -1x -1w -1v -1u -1t -1s -1r -1q -1p -1o -1n -1m -1l -1k -1j -1i -1h -1g -1f -1e -1d -1c -1b -1a -19 -18 -17 -16 -15 -14 -13 -12 -11 -10 -z -y -x -w -v -u -t -s -r -q -p -o -n -m -l -k -j -i -h -g -f -e -d -c -b -a -9 -8 -7 -6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6 7 8 9 a b c d e f g h i j k l m n o p q r s t u v w x y z 10 11 12 13 14 15 16 17 18 19 1a 1b 1c 1d 1e 1f 1g 1h 1i 1j 1k 1l 1m 1n 1o 1p 1q 1r 1s 1t 1u 1v 1w 1x 1y 1z 20 21 22 23 24 25 26 27 28 29 2a 2b 2c 2d 2e 2f 2g 2h 2i 2j 2k 2l 2m 2n 2o 2p 2q 2r 2s
2 -0.11100110011001100110011001100110011001100110011001101 -0.1100110011001100110011001100110011001100110011001101 -0.1011001100110011001100110011001100110011001100110011 -0.10011001100110011001100110011001100110011001100110011 -0.1 -0.01100110011001100110011001100110011001100110011001101 -0.010011001100110011001100110011001100110011001100110011 -0.001100110011001100110011001100110011001100110011001101 -0.0001100110011001100110011001100110011001100110011001101 0 0.0001100110011001100110011001100110011001100110011001101 0.001100110011001100110011001100110011001100110011001101 0.010011001100110011001100110011001100110011001100110011 0.01100110011001100110011001100110011001100110011001101 0.1 0.10011001100110011001100110011001100110011001100110011 0.1011001100110011001100110011001100110011001100110011 0.1100110011001100110011001100110011001100110011001101 0.11100110011001100110011001100110011001100110011001101
3 -0.22002200220022002200220022002201 -0.2101210121012101210121012101210122 -0.200220022002200220022002200220022 -0.1210121012101210121012101210121012 -0.1111111111111111111111111111111112 -0.1012101210121012101210121012101211 -0.02200220022002200220022002200220022 -0.0121012101210121012101210121012102 -0.0022002200220022002200220022002201 0 0.0022002200220022002200220022002201 0.0121012101210121012101210121012102 0.02200220022002200220022002200220022 0.1012101210121012101210121012101211 0.1111111111111111111111111111111112 0.1210121012101210121012101210121012 0.200220022002200220022002200220022 0.2101210121012101210121012101210122 0.22002200220022002200220022002201
4 -0.321212121212121212121212122 -0.30303030303030303030303031 -0.23030303030303030303030303 -0.212121212121212121212121212 -0.2 -0.121212121212121212121212122 -0.103030303030303030303030303 -0.030303030303030303030303031 -0.0121212121212121212121212122 0 0.0121212121212121212121212122 0.030303030303030303030303031 0.103030303030303030303030303 0.121212121212121212121212122 0.2 0.212121212121212121212121212 0.23030303030303030303030303 0.30303030303030303030303031 0.321212121212121212121212122
5 -0.42222222222222222222223 -0.4 -0.32222222222222222222222 -0.3 -0.22222222222222222222223 -0.2 -0.122222222222222222222222 -0.1 -0.022222222222222222222223 0 0.022222222222222222222223 0.1 0.122222222222222222222222 0.2 0.22222222222222222222223 0.3 0.32222222222222222222222 0.4 0.42222222222222222222223
6 -0.522222222222222222223 -0.44444444444444444445 -0.41111111111111111111 -0.333333333333333333333 -0.3 -0.222222222222222222223 -0.144444444444444444445 -0.1111111111111111111113 -0.0333333333333333333334 0 0.0333333333333333333334 0.1111111111111111111113 0.144444444444444444445 0.222222222222222222223 0.3 0.333333333333333333333 0.41111111111111111111 0.44444444444444444445 0.522222222222222222223
7 -0.6204620462046204621 -0.5412541254125412542 -0.4620462046204620462 -0.4125412541254125413 -0.3333333333333333334 -0.25412541254125412543 -0.2046204620462046205 -0.12541254125412541255 -0.04620462046204620463 0 0.04620462046204620463 0.12541254125412541255 0.2046204620462046205 0.25412541254125412543 0.3333333333333333334 0.4125412541254125413 0.4620462046204620462 0.5412541254125412542 0.6204620462046204621
8 -0.714631463146314632 -0.631463146314631464 -0.546314631463146314 -0.463146314631463146 -0.4 -0.314631463146314632 -0.231463146314631463 -0.146314631463146315 -0.0631463146314631464 0 0.0631463146314631464 0.146314631463146315 0.231463146314631463 0.314631463146314632 0.4 0.463146314631463146 0.546314631463146314 0.631463146314631464 0.714631463146314632
9 -0.8080808080808081 -0.71717171717171718 -0.62626262626262626 -0.53535353535353535 -0.44444444444444445 -0.35353535353535354 -0.262626262626262625 -0.17171717171717172 -0.08080808080808081 0 0.08080808080808081 0.17171717171717172 0.262626262626262625 0.35353535353535354 0.44444444444444445 0.53535353535353535 0.62626262626262626 0.71717171717171718 0.8080808080808081
10 -0.9 -0.8 -0.7 -0.6 -0.5 -0.4 -0.3 -0.2 -0.1 0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9
11 -0.99999999999999a -0.888888888888889 -0.7777777777777776 -0.6666666666666666 -0.5555555555555555 -0.4444444444444445 -0.3333333333333333 -0.2222222222222223 -0.11111111111111114 0 0.11111111111111114 0.2222222222222223 0.3333333333333333 0.4444444444444445 0.5555555555555555 0.6666666666666666 0.7777777777777776 0.888888888888889 0.99999999999999a
12 -0.a97249724972498 -0.972497249724973 -0.849724972497249 -0.724972497249724 -0.6 -0.4972497249724977 -0.372497249724972 -0.2497249724972499 -0.124972497249724b 0 0.124972497249724b 0.2497249724972499 0.372497249724972 0.4972497249724977 0.6 0.724972497249724 0.849724972497249 0.972497249724973 0.a97249724972498
13 -0.b913b913b913b9 -0.a527a527a527a55 -0.913b913b913b91 -0.7a527a527a527a4 -0.666666666666666 -0.527a527a527a529 -0.3b913b913b913b9 -0.27a527a527a527b -0.13b913b913b913c 0 0.13b913b913b913c 0.27a527a527a527b 0.3b913b913b913b9 0.527a527a527a529 0.666666666666666 0.7a527a527a527a4 0.913b913b913b91 0.a527a527a527a55 0.b913b913b913b9
14 -0.c8585858585859 -0.b2b2b2b2b2b2b3 -0.9b2b2b2b2b2b2b -0.85858585858585 -0.7 -0.585858585858589 -0.42b2b2b2b2b2b29 -0.2b2b2b2b2b2b2b5 -0.158585858585859 0 0.158585858585859 0.2b2b2b2b2b2b2b5 0.42b2b2b2b2b2b29 0.585858585858589 0.7 0.85858585858585 0.9b2b2b2b2b2b2b 0.b2b2b2b2b2b2b3 0.c8585858585859
15 -0.d7777777777778 -0.c -0.a7777777777776 -0.9 -0.77777777777777 -0.6 -0.47777777777777 -0.3 -0.17777777777777a 0 0.17777777777777a 0.3 0.47777777777777 0.6 0.77777777777777 0.9 0.a7777777777776 0.c 0.d7777777777778
16 -0.e6666666666668 -0.ccccccccccccd -0.b333333333333 -0.99999999999998 -0.8 -0.66666666666668 -0.4ccccccccccccc -0.33333333333334 -0.1999999999999a 0 0.1999999999999a 0.33333333333334 0.4ccccccccccccc 0.66666666666668 0.8 0.99999999999998 0.b333333333333 0.ccccccccccccd 0.e6666666666668
17 -0.f51bf51bf51bg -0.da36da36da36e -0.bf51bf51bf51b -0.a36da36da36da -0.8888888888889 -0.6da36da36da37 -0.51bf51bf51bf5 -0.36da36da36da39 -0.1bf51bf51bf51d 0 0.1bf51bf51bf51d 0.36da36da36da39 0.51bf51bf51bf5 0.6da36da36da37 0.8888888888889 0.a36da36da36da 0.bf51bf51bf51b 0.da36da36da36e 0.f51bf51bf51bg
18 -0.g3ae73ae73ae8 -0.e73ae73ae73af -0.cae73ae73ae73 -0.ae73ae73ae73a -0.9 -0.73ae73ae73ae8 -0.573ae73ae73ae -0.3ae73ae73ae74 -0.1e73ae73ae73b 0 0.1e73ae73ae73b 0.3ae73ae73ae74 0.573ae73ae73ae 0.73ae73ae73ae8 0.9 0.ae73ae73ae73a 0.cae73ae73ae73 0.e73ae73ae73af 0.g3ae73ae73ae8
19 -0.h1h1h1h1h1h2 -0.f3f3f3f3f3f4 -0.d5d5d5d5d5d5b -0.b7b7b7b7b7b7a -0.9999999999999 -0.7b7b7b7b7b7b9 -0.5d5d5d5d5d5d5 -0.3f3f3f3f3f3f4 -0.1h1h1h1h1h1h2 0 0.1h1h1h1h1h1h2 0.3f3f3f3f3f3f4 0.5d5d5d5d5d5d5 0.7b7b7b7b7b7b9 0.9999999999999 0.b7b7b7b7b7b7a 0.d5d5d5d5d5d5b 0.f3f3f3f3f3f4 0.h1h1h1h1h1h2
20 -0.i -0.g -0.e -0.c -0.a -0.8 -0.6 -0.4 -0.2 0 0.2 0.4 0.6 0.8 0.a 0.c 0.e 0.g 0.i
21 -0.iiiiiiiiiiij -0.gggggggggggh -0.eeeeeeeeeeee -0.cccccccccccc9 -0.aaaaaaaaaaaaa -0.888888888888c -0.6666666666665 -0.4444444444446 -0.2222222222223 0 0.2222222222223 0.4444444444446 0.6666666666665 0.888888888888c 0.aaaaaaaaaaaaa 0.cccccccccccc9 0.eeeeeeeeeeee 0.gggggggggggh 0.iiiiiiiiiiij
22 -0.jhd48hd48hd5 -0.hd48hd48hd49 -0.f8hd48hd48hd -0.d48hd48hd48h -0.b -0.8hd48hd48hd5 -0.6d48hd48hd49 -0.48hd48hd48hd8 -0.248hd48hd48hf 0 0.248hd48hd48hf 0.48hd48hd48hd8 0.6d48hd48hd49 0.8hd48hd48hd5 0.b 0.d48hd48hd48h 0.f8hd48hd48hd 0.hd48hd48hd49 0.jhd48hd48hd5
23 -0.kg26kg26kg27 -0.i94di94di94f -0.g26kg26kg26k -0.di94di94di94 -0.bbbbbbbbbbbb -0.94di94di94dj -0.6kg26kg26kg2 -0.4di94di94di9a -0.26kg26kg26kg5 0 0.26kg26kg26kg5 0.4di94di94di9a 0.6kg26kg26kg2 0.94di94di94dj 0.bbbbbbbbbbbb 0.di94di94di94 0.g26kg26kg26k 0.i94di94di94f 0.kg26kg26kg27
24 -0.le9e9e9e9e9f -0.j4j4j4j4j4j6 -0.gj4j4j4j4j4i -0.e9e9e9e9e9e9 -0.c -0.9e9e9e9e9e9f -0.74j4j4j4j4j4 -0.4j4j4j4j4j4k -0.29e9e9e9e9ea 0 0.29e9e9e9e9ea 0.4j4j4j4j4j4k 0.74j4j4j4j4j4 0.9e9e9e9e9e9f 0.c 0.e9e9e9e9e9e9 0.gj4j4j4j4j4i 0.j4j4j4j4j4j6 0.le9e9e9e9e9f
25 -0.mcccccccccce -0.k -0.hcccccccccca -0.f -0.cccccccccccc -0.a -0.7ccccccccccc -0.5 -0.2ccccccccccd 0 0.2ccccccccccd 0.5 0.7ccccccccccc 0.a 0.cccccccccccc 0.f 0.hcccccccccca 0.k 0.mcccccccccce
26 -0.naaaaaaaaaad -0.kkkkkkkkkkl -0.i5555555555 -0.fffffffffffd -0.d -0.aaaaaaaaaaad -0.7kkkkkkkkkkk -0.555555555556 -0.2ffffffffffg 0 0.2ffffffffffg 0.555555555556 0.7kkkkkkkkkkk 0.aaaaaaaaaaad 0.d 0.fffffffffffd 0.i5555555555 0.kkkkkkkkkkl 0.naaaaaaaaaad
27 -0.o82io82io83 -0.lg5alg5alg5h -0.io82io82io8 -0.g5alg5alg5ai -0.dddddddddddd -0.alg5alg5alg9 -0.82io82io82in -0.5alg5alg5ali -0.2io82io82io9 0 0.2io82io82io9 0.5alg5alg5ali 0.82io82io82in 0.alg5alg5alg9 0.dddddddddddd 0.g5alg5alg5ai 0.io82io82io8 0.lg5alg5alg5h 0.o82io82io83
28 -0.p5gmb5gmb5h -0.mb5gmb5gmb6 -0.jgmb5gmb5gm -0.gmb5gmb5gmb -0.e -0.b5gmb5gmb5h -0.8b5gmb5gmb5e -0.5gmb5gmb5gme -0.2mb5gmb5gmb7 0 0.2mb5gmb5gmb7 0.5gmb5gmb5gme 0.8b5gmb5gmb5e 0.b5gmb5gmb5h 0.e 0.gmb5gmb5gmb 0.jgmb5gmb5gm 0.mb5gmb5gmb6 0.p5gmb5gmb5h
29 -0.q2q2q2q2q2q -0.n5n5n5n5n5o -0.k8k8k8k8k8k -0.hbhbhbhbhbh -0.eeeeeeeeeef -0.bhbhbhbhbhc -0.8k8k8k8k8k8g -0.5n5n5n5n5n6 -0.2q2q2q2q2q3 0 0.2q2q2q2q2q3 0.5n5n5n5n5n6 0.8k8k8k8k8k8g 0.bhbhbhbhbhc 0.eeeeeeeeeef 0.hbhbhbhbhbh 0.k8k8k8k8k8k 0.n5n5n5n5n5o 0.q2q2q2q2q2q
30 -0.r -0.o -0.l -0.i -0.f -0.c -0.9 -0.6 -0.3 0 0.3 0.6 0.9 0.c 0.f 0.i 0.l 0.o 0.r
31 -0.rrrrrrrrrrs -0.ooooooooooq -0.lllllllllll -0.iiiiiiiiiii -0.fffffffffff -0.ccccccccccd -0.99999999999 -0.66666666666f -0.333333333337 0 0.333333333337 0.66666666666f 0.99999999999 0.ccccccccccd 0.fffffffffff 0.iiiiiiiiiii 0.lllllllllll 0.ooooooooooq 0.rrrrrrrrrrs
32 -0.spj6cpj6cpk -0.pj6cpj6cpj8 -0.mcpj6cpj6co -0.j6cpj6cpj6c -0.g -0.cpj6cpj6cpk -0.9j6cpj6cpj6 -0.6cpj6cpj6cq -0.36cpj6cpj6d 0 0.36cpj6cpj6d 0.6cpj6cpj6cq 0.9j6cpj6cpj6 0.cpj6cpj6cpk 0.g 0.j6cpj6cpj6c 0.mcpj6cpj6co 0.pj6cpj6cpj8 0.spj6cpj6cpk
33 -0.tn39tn39tn4 -0.qd6jqd6jqd9 -0.n39tn39tn38 -0.jqd6jqd6jqc -0.ggggggggggg -0.d6jqd6jqd6l -0.9tn39tn39tn -0.6jqd6jqd6jr -0.39tn39tn39u 0 0.39tn39tn39u 0.6jqd6jqd6jr 0.9tn39tn39tn 0.d6jqd6jqd6l 0.ggggggggggg 0.jqd6jqd6jqc 0.n39tn39tn38 0.qd6jqd6jqd9 0.tn39tn39tn4
34 -0.ukdkdkdkdkf -0.r6r6r6r6r7 -0.nr6r6r6r6r -0.kdkdkdkdkdj -0.h -0.dkdkdkdkdkf -0.a6r6r6r6r6q -0.6r6r6r6r6r8 -0.3dkdkdkdkdl 0 0.3dkdkdkdkdl 0.6r6r6r6r6r8 0.a6r6r6r6r6q 0.dkdkdkdkdkf 0.h 0.kdkdkdkdkdj 0.nr6r6r6r6r 0.r6r6r6r6r7 0.ukdkdkdkdkf
35 -0.vhhhhhhhhhk -0.s -0.ohhhhhhhhhd -0.l -0.hhhhhhhhhhh -0.e -0.ahhhhhhhhhg -0.7 -0.3hhhhhhhhhi 0 0.3hhhhhhhhhi 0.7 0.ahhhhhhhhhg 0.e 0.hhhhhhhhhhh 0.l 0.ohhhhhhhhhd 0.s 0.vhhhhhhhhhk
36 -0.weeeeeeeeeh -0.ssssssssst -0.p777777777 -0.llllllllllj -0.i -0.eeeeeeeeeeh -0.asssssssssr -0.77777777779 -0.3lllllllllm 0 0.3lllllllllm 0.77777777779 0.asssssssssr 0.eeeeeeeeeeh 0.i 0.llllllllllj 0.p777777777 0.ssssssssst 0.weeeeeeeeeh
===*/

print('basic');

function basicTest() {
    var frac_values;

    // radix not present or undefined -> radix 10

    test(new Number(12345), []);
    test(new Number(12345), [ undefined ]);

    // in contrast, radix null will coerce to 0, which will cause a RangeError

    test(new Number(12345), [ null ]);

    // radix is ToInteger() coerced (not ToUint32() coerced), so these will
    // cause a RangeError

    test(new Number(12345), [ -256*256*256*256 + 10 ]);
    test(new Number(12345), [ 256*256*256*256 + 10 ]);

    // test all valid radix values for a reasonable integer range

    for (r = 2; r <= 36; r++) {
        tmp = [];
        for (i = -100; i <= 100; i++) {
            tmp.push(new Number(i).toString(r));
        }
        print(r, tmp.join(' '));
    }

    // test all valid radix values for a set of fractional values
    // (note that exact results are NOT required by the spec)

    frac_values = [ -0.9, -0.8, -0.7, -0.6, -0.5, -0.4, -0.3, -0.2, -0.1, 0.0,
                    0.1, 0.2, 0.3, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9 ];
    for (r = 2; r <= 36; r++) {
        tmp = [];
        for (i = 0; i < frac_values.length; i++) {
            tmp.push(new Number(frac_values[i]).toString(r));
        }
        print(r, tmp.join(' '));
    }
}

try {
    basicTest();
} catch (e) {
    print(e);
}

/*===
coercion
TypeError
TypeError
TypeError
string 7b
TypeError
TypeError
TypeError
TypeError
string 7b
string 12345
string 12345
RangeError
RangeError
RangeError
string 7189
string 7189
string 7189
RangeError
RangeError
string 11000000111001
string 11000000111001
string 9ix
string 9ix
RangeError
RangeError
radix valueOf
string 5808
string 5808
===*/

print('coercion');

function coercionTest() {
    // this coercion; only plain number and Number objects allowed

    test(undefined, [ 16 ]);
    test(true, [ 16 ]);
    test(false, [ 16 ]);
    test(123, [ 16 ]);
    test('123', [ 16 ]);
    test([123], [ 16 ]);
    test([1,2], [ 16 ]);
    test({ foo: 1, bar: 2 }, [ 16 ]);
    test(new Number(123), [ 16 ]);

    // radix coercion

    test(new Number(12345), []);             // -> treat as radix 10 (explicitly required)
    test(new Number(12345), [ undefined ]);  // -> treat as radix 10 (explicitly required)
    test(new Number(12345), [ null ]);       // ToInteger(null) = 0 -> RangeError
    test(new Number(12345), [ true ]);       // ToInteger(true) = 1 -> RangeError
    test(new Number(12345), [ false ]);      // ToInteger(false) = 0 -> RangeError
    test(new Number(12345), [ 12 ]);         // ToInteger(12) = 12 -> OK
    test(new Number(12345), [ '12' ]);       // ToInteger('12') = 12 -> OK
    test(new Number(12345), [ [12] ]);       // ToInteger([12]) -> ToInteger("12") = 12 -> OK (!)
    test(new Number(12345), [ [1,2] ]);      // ToInteger([1,2]) -> ToInteger("1,2") = 0 -> RangeError
    test(new Number(12345), [ { foo: 1, bar: 2 } ]);  // ToInteger(...) -> 0 -> RangeError

    // radix coercion uses ToInteger()

    test(new Number(12345), [ 2.1 ]);   // -> 2 = valid
    test(new Number(12345), [ 2 ]);

    test(new Number(12345), [ 36.9 ]);  // -> 36 = valid
    test(new Number(12345), [ 36 ]);

    test(new Number(12345), [ -256*256*256*256 + 10 ]);  // invalid (not same as 10)
    test(new Number(12345), [ 256*256*256*256 + 10 ]);   // invalid (not same as 10)

    // radix coercion, side effects

    test(new Number(12345), [ {
        toString: function() { print('radix toString'); return 16; },
        valueOf: function() { print('radix valueOf'); return 13.9; }  // -> 13
    } ]);
    test(new Number(12345), [ 13 ]);
}

try {
    coercionTest();
} catch (e) {
    print(e);
}
