/*###############################################################################
# Linux Management Providers (LMP), OS provider package
# Copyright (C) 2007 Ilsoo Byun, ETRI <widepis@etri.re.kr ,widepis@empal.com>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by 2009 Khahramon NURIDDINOV, TUIT <qahramon0786@gmail.com>
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
###############################################################################*/

#include "OpenDRIM_OperatingSystemAccess.h"
#include <sys/resource.h>

static string _CSCreationClassName = "OpenDRIM_ComputerSystem";
static string _CSName;
static string _Name;
static string _Version;
static unsigned long _MaxNumberOfProcesses;
static unsigned long _MaxProcessMemorySize;
static string _ElementName;
static unsigned long _DateOfInstallation;
static unsigned long _NumberOfLogicalCPU;

static short int _CurrentTimeZone;

int OS_OpenDRIM_OperatingSystem_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getOSName(_Name, errorMessage));
	CF_assert(CF_getSystemName(_CSName, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getVersion(_Version, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getMaxNumberOfProcesses(_MaxNumberOfProcesses, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getMaxProcessMemorySize(_MaxProcessMemorySize, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getElementName(_ElementName, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getDateOfInstallation(_DateOfInstallation, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getNumberOfLogicalCPU(_NumberOfLogicalCPU, errorMessage));
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_unload(string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_OperatingSystem>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	OpenDRIM_OperatingSystem instance;
	instance.setCSCreationClassName(_CSCreationClassName);
	instance.setCSName(_CSName);
	instance.setName(_Name);
	instance.setCreationClassName(OpenDRIM_OperatingSystem_classnames[0]);
	if (discriminant == "ei") {
		CF_assert(OS_OpenDRIM_OperatingSystem_getOperatingSystemInfo(instance, errorMessage));
	}
	result.push_back(instance);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_OperatingSystem& instance, const char** properties, string& errorMessage) {
	_E_;
	string CSCreationClassName, CSName, Name, CreationClassName;
	instance.getCreationClassName(CreationClassName);
	instance.getCSName(CSName);
	instance.getName(Name);
	instance.getCSCreationClassName(CSCreationClassName);
	if (!CF_strCmpNoCase(OpenDRIM_OperatingSystem_classnames[0], CreationClassName) || !CF_strCmpNoCase(CSName, _CSName) || !CF_strCmpNoCase(Name, _Name) || !CF_strCmpNoCase(CSCreationClassName, _CSCreationClassName))
		return NOT_FOUND;
	CF_assert(OS_OpenDRIM_OperatingSystem_getOperatingSystemInfo(instance, errorMessage));
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& newInstance, const OpenDRIM_OperatingSystem& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int OS_OpenDRIM_OperatingSystem_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int OS_OpenDRIM_OperatingSystem_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int OS_OpenDRIM_OperatingSystem_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& instance, unsigned int& returnValue, const OpenDRIM_OperatingSystem_RequestStateChange_In& in, OpenDRIM_OperatingSystem_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return NOT_SUPPORTED;
}

int OS_OpenDRIM_OperatingSystem_Shutdown(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	string stdOut, stdErr;
	int errorCode = CF_runCommand("/sbin/shutdown 3 &", stdOut, stdErr, errorMessage);
	if (stdErr.find("shutdown: already running.") != string::npos) {
		returnValue = 2;
		return OK;
	}
	if (errorCode!=OK) {
		returnValue = 3;
		return OK;
	}
	returnValue = 0;
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_Reboot(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_OperatingSystem& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	string stdOut, stdErr;
	int errorCode = CF_runCommand("/sbin/shutdown -r 3 &", stdOut, stdErr, errorMessage);
	if (stdErr.find("shutdown: already running.") != string::npos) {
		returnValue = 2;
		return OK;
	}
	if (errorCode!=OK) {
		returnValue = 3;
		return OK;
	}
	returnValue = 0;
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_populate(OpenDRIM_OperatingSystem& instance, string& errorMessage) {
	_E_;
	// TODO
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getOperatingSystemInfo(OpenDRIM_OperatingSystem& instance, string& errorMessage) {
	_E_;
	instance.setVersion(_Version);
	instance.setOSType(36);
	_CurrentTimeZone = CF_getCurrentTimeZone();
	instance.setCurrentTimeZone(_CurrentTimeZone);
	instance.setLocalDateTime(CF_toLocalTime(CF_getUTCTime(), _CurrentTimeZone));
	string LastBootUpTime;
	CF_assert(OS_OpenDRIM_OperatingSystem_getLastBootUpTime(LastBootUpTime, errorMessage));
	instance.setLastBootUpTime(LastBootUpTime);
	instance.setMaxNumberOfProcesses(_MaxNumberOfProcesses);
	unsigned long NumberOfUsers, NumberOfProcesses;
	CF_assert(OS_OpenDRIM_OperatingSystem_getNumberOfUsers(NumberOfUsers, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_getNumberOfProcesses(NumberOfProcesses, errorMessage));
	instance.setNumberOfLicensedUsers(0);
	instance.setNumberOfUsers(NumberOfUsers);
	instance.setNumberOfProcesses(NumberOfProcesses);
	instance.setMaxNumberOfProcesses(_MaxNumberOfProcesses);
	instance.setMaxProcessMemorySize(_MaxProcessMemorySize);
	instance.setElementName(_ElementName);
	unsigned long MaxProcessPerUser;
	OS_OpenDRIM_OperatingSystem_getMaxProcessPerUser(MaxProcessPerUser, errorMessage);
	instance.setMaxProcessesPerUser(MaxProcessPerUser);
	instance.setInstallDate(CF_toLocalTime(_DateOfInstallation, _CurrentTimeZone));
	bool shutting_down, stressed, rebooting;
	CF_assert(OS_OpenDRIM_OperatingSystem_isShuttingDown(shutting_down, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_isRebooting(rebooting, errorMessage));
	CF_assert(OS_OpenDRIM_OperatingSystem_isStressed(stressed, errorMessage));
	vector<unsigned short> OperationalStatus;
	instance.setEnabledState(2); // Enabled
	instance.setRequestedState(2); // Enabled
	instance.setPrimaryStatus(1); // OK
	instance.setDetailedStatus(1); // No Additional Information
	instance.setHealthState(5); // OK
	if (shutting_down) {
		instance.setEnabledState(4); // Shutting down
		instance.setRequestedState(4); // Shutdown
		OperationalStatus.push_back(9); // Stopping
	}
	if (rebooting)
		instance.setRequestedState(10); // Reboot
	if (stressed) {
		instance.setPrimaryStatus(2); // Degraded
		instance.setDetailedStatus(2); // Stressed
		OperationalStatus.push_back(3); // Degraded
		OperationalStatus.push_back(4); // Stressed
		instance.setHealthState(10); // Degraded/Warning
	} else
		OperationalStatus.push_back(2); // OK
	instance.setOperationalStatus(OperationalStatus);
	CF_assert(OS_OpenDRIM_OperatingSystem_getMemoryInfo(instance, errorMessage));
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getVersion(string& Version, string& errorMessage) {
	_E_;
	CF_assert(CF_runCommandFL("uname -r", Version, errorMessage));
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getLastBootUpTime(string& LastBootUpTime, string& errorMessage) {
	_E_;
	string who;
	CF_assert(CF_runCommandFL("LANG=en_US.UTF-8 /usr/bin/who -b", who, errorMessage));
	vector<string> who_line;
	CF_splitTextBySpace(who_line, CF_trimText(who));
	if (who_line.size() != 4 || who_line[2].size() != 10 || who_line[3].size() != 5) {
		errorMessage = "Wrong format: /usr/bin/who -b output";
		return FAILED;
	}
	LastBootUpTime = who_line[2].substr(0, 4) + who_line[2].substr(5, 2) + who_line[2].substr(8, 2) + who_line[3].substr(0, 2) + who_line[3].substr(3, 2) + "00.000000";
	CF_addTimeZone(LastBootUpTime, _CurrentTimeZone);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getNumberOfUsers(unsigned long& NumberOfUsers, string& errorMessage) {
	_E_;
	NumberOfUsers=0;
	string passwd;
	CF_assert(CF_readTextFile("/etc/passwd", passwd, errorMessage));
	vector<string> passwd_lines;
	CF_splitText(passwd_lines, passwd, '\n');
	for (size_t i=0; i<passwd_lines.size(); i++) {
		if (passwd_lines[i].size()==0)
			continue;
		vector<string> passwd_line;
		CF_splitText(passwd_line, passwd_lines[i], ':');
		if (passwd_line.size()!=7) {
			errorMessage="Wrong format (at line "+CF_intToStr((unsigned long) i+1)+"): /etc/passwd";
			return FAILED;
		}
		if (passwd_line[6] != "/sbin/nologin" && CF_strToUL(passwd_line[2]) >= 500)
			NumberOfUsers++;
	}
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getNumberOfProcesses(unsigned long& NumberOfProcesses, string& errorMessage) {
	_E_;
	string ps;
	CF_assert(CF_runCommandFL("/bin/ps -e | /usr/bin/wc -l", ps, errorMessage));
	NumberOfProcesses = CF_strToUL(ps);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getMaxNumberOfProcesses(unsigned long& MaxNumberOfProcesses, string& errorMessage) {
	_E_;
	string pid_max;
	CF_assert(CF_readTextFileFL("/proc/sys/kernel/pid_max", pid_max, errorMessage));
	MaxNumberOfProcesses = CF_strToUL(pid_max);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getMaxProcessMemorySize(unsigned long& MaxProcessMemorySize, string& errorMessage) {
	_E_;
	struct rlimit rlim;
	getrlimit(RLIMIT_AS, &rlim);
	MaxProcessMemorySize = rlim.rlim_max;
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getMaxProcessPerUser(unsigned long& MaxProcessPerUser, string& errorMessage) {
	_E_;
	string ulimit;
	CF_assert(CF_runCommandFL("ulimit -u", ulimit, errorMessage));
	MaxProcessPerUser = CF_strToUL(ulimit);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getElementName(string& ElementName, string& errorMessage) {
	_E_;
	string release_file;
	if (CF_runCommandFL("ls /etc | grep -E -e \".+-release$\"", release_file, errorMessage)!=OK || release_file.size()==0) {
		ElementName="Unknown";
		return OK;
	} else {
		CF_assert(CF_readTextFileFL("/etc/"+release_file, ElementName, errorMessage));
	}
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getDateOfInstallation(unsigned long& DateOfInstallation, string& errorMessage) {
	_E_;
	string find;
	CF_assert(CF_runCommandFL("/usr/bin/find /etc/ -maxdepth 1 -type f -name *-release*", find, errorMessage));
	if (find.size()<=13) {
		DateOfInstallation = 0;
		return OK;
	}
	string rpm;
	CF_assert(CF_runCommandFL("/bin/rpm -q --queryformat '%{INSTALLTIME:datetime}' "+find.substr(5, string::npos), rpm, errorMessage));
	if (rpm.find("not installed")!=string::npos) {
		DateOfInstallation = 0;
		return OK;
	}
	DateOfInstallation = CF_strToUL(rpm);
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_isShuttingDown(bool& shutting_down, string& errorMessage) {
	_E_;
	string ps;
	shutting_down=false;
	CF_assert(CF_runCommandFL("/bin/ps -C shutdown,halt,poweroff --no-heading -o cmd", ps, errorMessage));
	if (ps.size()>0) {
		shutting_down=true;
		if (ps.find("shutdown")!=string::npos && ps.find("-r")!=string::npos)
			shutting_down=false;
	}
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_isRebooting(bool& rebooting, string& errorMessage) {
	_E_;
	string ps;
	rebooting=false;
	CF_assert(CF_runCommandFL("/bin/ps -C shutdown,reboot --no-heading -o cmd", ps, errorMessage));
	if (ps.size()>0) {
		rebooting=true;
		if (ps.find("shutdown")!=string::npos && ps.find("-r")==string::npos)
			rebooting=false;
	}
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_isStressed(bool& stressed, string& errorMessage) {
	_E_;
	stressed=false;
	string loadavg;
	CF_assert(CF_readTextFileFL("/proc/loadavg", loadavg, errorMessage));
	vector<string> loadavg_elements;
	CF_splitTextBySpace(loadavg_elements, loadavg);
	if (loadavg_elements.size() < 3) {
		errorMessage = "Wrong format: /proc/loadavg";
		return FAILED;	
	}
	if (atof(loadavg_elements[2].c_str()) > _NumberOfLogicalCPU)
		stressed=true;
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getNumberOfLogicalCPU(unsigned long& NumberOfLogicalCPU, string& errorMessage) {
	_E_;
	vector<string> stat_lines;
	CF_assert(CF_runCommandToLines("cat /proc/stat | grep cpu", stat_lines, 0, errorMessage));
	NumberOfLogicalCPU=stat_lines.size()-2;
	_L_;
	return OK;
}

int OS_OpenDRIM_OperatingSystem_getMemoryInfo(OpenDRIM_OperatingSystem& instance, string& errorMessage) {
	_E_;
	vector<string> meminfo;
	CF_assert(CF_readTextFileToLines("/proc/meminfo", meminfo, 0, errorMessage));
	unsigned long long TotalSwapSpaceSize, FreePhysicalMemory, TotalVisibleMemorySize, FreeSpaceInPagingFiles;
	bool TotalSwapSpaceSize_isNULL = true;
	bool FreePhysicalMemory_isNULL = true;
	bool TotalVisibleMemorySize_isNULL = true;
	bool FreeSpaceInPagingFiles_isNULL = true;
	for (size_t i=0; i<meminfo.size(); i++) {
		vector<string> meminfo_line;
		CF_splitTextBySpace(meminfo_line, meminfo[i]);
		if (meminfo_line.size()<2)
			continue;
		if (meminfo_line[0]=="MemTotal:") {
			TotalVisibleMemorySize = CF_strToULL(meminfo_line[1])*(1024/1000);
			TotalVisibleMemorySize_isNULL = false;
		}
		else if (meminfo_line[0]=="MemFree:") {
			FreePhysicalMemory = CF_strToULL(meminfo_line[1])*(1024/1000);
			FreePhysicalMemory_isNULL = false;
		}
		else if (meminfo_line[0]=="SwapTotal:") {
			TotalSwapSpaceSize = CF_strToULL(meminfo_line[1])*(1024/1000);
			TotalSwapSpaceSize_isNULL = false;
		}
		else if (meminfo_line[0]=="SwapFree:") {
			FreeSpaceInPagingFiles = CF_strToULL(meminfo_line[1])*(1024/1000);
			FreeSpaceInPagingFiles_isNULL = false;
		}
	}
	if (!TotalVisibleMemorySize_isNULL)
		instance.setTotalVisibleMemorySize(TotalVisibleMemorySize);
	if (!FreePhysicalMemory_isNULL)
		instance.setFreePhysicalMemory(FreePhysicalMemory);
	if (!TotalSwapSpaceSize_isNULL)
		instance.setTotalSwapSpaceSize(TotalSwapSpaceSize);
	if (!FreeSpaceInPagingFiles_isNULL)
		instance.setFreeSpaceInPagingFiles(FreeSpaceInPagingFiles);
	if (!TotalVisibleMemorySize_isNULL && !TotalSwapSpaceSize_isNULL)
		instance.setTotalVirtualMemorySize(TotalSwapSpaceSize+TotalVisibleMemorySize);
	if (!FreePhysicalMemory_isNULL && !FreeSpaceInPagingFiles_isNULL)
		instance.setFreeVirtualMemory(FreePhysicalMemory + FreeSpaceInPagingFiles);
	if (!TotalSwapSpaceSize_isNULL && !FreeSpaceInPagingFiles_isNULL)
		instance.setSizeStoredInPagingFiles(TotalSwapSpaceSize-FreeSpaceInPagingFiles);
	_L_;
	return OK;
}

