var count = (function toString(BigNumber) {
    var start = +new Date(),
        log,
        error,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error; 
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function T(expected, value) {
        assert(expected, new BigNumber(value).toString())
    }

    log('\n Testing toString...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : 1E9
    });

    // ---------------------------------------------------------------------- v8

    T('NaN', NaN);
    T('Infinity', 1/0);
    T('-Infinity', -1/0);
    T('0', 0);
    T('9', 9);
    T('90', 90);
    T('90.12', 90.12);
    T('0.1', 0.1);
    T('0.01', 0.01);
    T('0.0123', 0.0123);
    T('111111111111111111111',   '111111111111111111111');
    T('1111111111111111111111',  '1111111111111111111111');
    T('11111111111111111111111', '11111111111111111111111');
    T('0.00001', 0.00001);
    T('0.000001', 0.000001);

    BigNumber.config({EXPONENTIAL_AT : 0});

    T('1e-7', 0.0000001);
    T('1.2e-7', 0.00000012);
    T('1.23e-7', 0.000000123);
    T('1e-8', 0.00000001);
    T('1.2e-8', 0.000000012);
    T('1.23e-8', 0.0000000123);
    T('-1e-7', -0.0000001);
    T('-1.2e-7', -0.00000012);
    T('-1.23e-7', -0.000000123);
    T('-1e-8', -0.00000001);
    T('-1.2e-8', -0.000000012);
    T('-1.23e-8', -0.0000000123);

    BigNumber.config({EXPONENTIAL_AT : 60});

    T('0', -0);
    T('-9', -9);
    T('-90', -90);
    T('-90.12', -90.12);
    T('-0.1', -0.1);
    T('-0.01', -0.01);
    T('-0.0123', -0.0123);
    T('-111111111111111111111',  '-111111111111111111111');
    T('-1111111111111111111111', '-1111111111111111111111');
    T('-11111111111111111111111', '-11111111111111111111111');
    T('-0.00001', -0.00001);
    T('-0.000001', -0.000001);

    // ------------------------------------------------------------------ v8 end

    // 1000 toString to exponentials
    BigNumber.config({EXPONENTIAL_AT : 0});

    T('5.73447902457635174479825134e+14', '573447902457635.174479825134');
    T('1.07688e+1', '10.7688');
    T('3.171194102379077141557759899307946350455841e+27', '3171194102379077141557759899.307946350455841');
    T('4.924353466898191177698653319742594890634579e+37', '49243534668981911776986533197425948906.34579');
    T('6.85558243926569397328633907445409866949445343654692955e+18', '6855582439265693973.28633907445409866949445343654692955');
    T('1e+0', '1');
    T('2.1320000803e+7', '21320000.803');
    T('5.0878741e+4', '50878.741');
    T('5.1932898288391e+8', '519328982.88391');
    T('5.690616778176956027307884725933410349604387915634524e+49', '56906167781769560273078847259334103496043879156345.24');
    T('3.25099780528575457273246693147645e+15', '3250997805285754.572732466931476450');
    T('3e+0', '3');
    T('2.5811494197573291905990947355226e+13', '25811494197573.291905990947355226');
    T('5.60372259169833930471746454125e+13', '56037225916983.3930471746454125');
    T('1.2615810663732236602461593613783e+7', '12615810.663732236602461593613783');
    T('1.4654366449266911507705477499035407722184965108377032e+11', '146543664492.66911507705477499035407722184965108377032');
    T('6.4986735507448912857131832908423940757e+38', '649867355074489128571318329084239407570.0');
    T('3.6146989180120676857245474944e+3', '3614.6989180120676857245474944');
    T('9.928654762302286149994896625074e+4', '99286.54762302286149994896625074');
    T('3.46424170787806074650506079e+3', '3464.24170787806074650506079');
    T('1.25934313355319666474752550204680303068865719647e+33', '1259343133553196664747525502046803.03068865719647');
    T('1.23014105337660651106978059198916100450966081493207e+27', '1230141053376606511069780591.98916100450966081493207');
    T('1.386164712267169624993434287237e+23', '138616471226716962499343.4287237');
    T('2.66076369930322488334961932e+3', '2660.76369930322488334961932');
    T('9.37582568e+4', '93758.2568');
    T('1.39853642894726883996875746770529e+28', '13985364289472688399687574677.0529');
    T('3.19099e+5', '319099.0');
    T('3.04557106798789396303998723e+19', '30455710679878939630.3998723');
    T('1.3024612569115368830867934222004329653604418e+9', '1302461256.9115368830867934222004329653604418');
    T('2.358787483447172786e+5', '235878.7483447172786');
    T('5.10614446965318674547416709785208086304398889160563e+28', '51061444696531867454741670978.5208086304398889160563');
    T('1.46685947134456101512731611558e+23', '146685947134456101512731.6115580');
    T('3.69960105771344554151928256518906564810300119e+25', '36996010577134455415192825.6518906564810300119');
    T('2.68683153074628e+10', '26868315307.4628');
    T('2.35656504568492312232737219553793859212e+15', '2356565045684923.12232737219553793859212');
    T('7.753292442361215e+14', '775329244236121.5');
    T('1.56e+0', '1.56');
    T('1.272818730367215461852227991200704e+21', '1272818730367215461852.227991200704');
    T('1.13900700292988027871648046839423153789e+7', '11390070.0292988027871648046839423153789');
    T('3.3431e+0', '3.3431');
    T('1.4546654966819402705e+14', '145466549668194.02705');
    T('3.05345735395805567424714891401667575466462830113819e+48', '3053457353958055674247148914016675754664628301138.19');
    T('5.1218945854639324441304933666460587e+2', '512.18945854639324441304933666460587');
    T('9.95299900896e+5', '995299.9008960');
    T('1.21564537151562431991786620635e+0', '1.21564537151562431991786620635');
    T('4.016e+1', '40.16');
    T('1.86570326e+7', '18657032.6');
    T('1.3381001727e+5', '133810.01727');
    T('2.639841700687441886800225725227e+12', '2639841700687.441886800225725227');
    T('2.45e+0', '2.45');
    T('2.8945e+2', '289.45');
    T('1.23e+0', '1.23');
    T('1.559806666149836070330006415033e+24', '1559806666149836070330006.415033');
    T('3.14984566145310751826289711761375061645611777700983e+3', '3149.84566145310751826289711761375061645611777700983');
    T('3.0940691333892283249774116223987e+5', '309406.91333892283249774116223987');
    T('6.572766274013360381079275191108732606370177179594282e+5', '657276.6274013360381079275191108732606370177179594282');
    T('1.470126973337024e+6', '1470126.973337024');
    T('5.6499e+2', '564.99');
    T('2.8416297367859233303497847667971781197616950846e+28', '28416297367859233303497847667.971781197616950846');
    T('2.1364951568189836563102481625533538320051163977e+41', '213649515681898365631024816255335383200511.63977');
    T('8.76108618687537137080904679797e+19', '87610861868753713708.0904679797');
    T('6.27683573474251182408654509953633505286e+6', '6276835.73474251182408654509953633505286');
    T('8.91411e+0', '8.91411');
    T('9.034542832410912578330021146413119399e+28', '90345428324109125783300211464.13119399');
    T('7.026094393430852002585511641212897686956090955e+39', '7026094393430852002585511641212897686956.090955');
    T('1.8812221093491505758565988678062e+11', '188122210934.91505758565988678062');
    T('9.435538492497050138580201734902181057468044875e+43', '94355384924970501385802017349021810574680448.75');
    T('5.36793419620790391866461e+20', '536793419620790391866.461');
    T('2.315089265590404012562599592854156357726817712e+26', '231508926559040401256259959.2854156357726817712');
    T('7.499170741828885273030006066196546588710962e+17', '749917074182888527.3030006066196546588710962');
    T('3.3962128305986e+5', '339621.28305986');
    T('8.17980456510031304e+9', '8179804565.10031304');
    T('4.394575876858124185382e+13', '43945758768581.24185382');
    T('7.881617323629751701107428e+9', '7881617323.629751701107428');
    T('4.89e+0', '4.89');
    T('9.85209894663520857685703881781194082356123765e+39', '9852098946635208576857038817811940823561.23765');
    T('6.849329685e+5', '684932.9685');
    T('2.8262252277815736355279617243060700437627773361e+7', '28262252.277815736355279617243060700437627773361');
    T('1.503736721902e+9', '1503736721.902');
    T('2.65213505469049775997232325076980590625670234690917845e+41', '265213505469049775997232325076980590625670.234690917845');
    T('4.23752645959719196604760963802412828187442060555521e+2', '423.752645959719196604760963802412828187442060555521');
    T('9.023159535576504097005203913521871601640521009e+36', '9023159535576504097005203913521871601.640521009');
    T('4.69339457186380276410136272120035011198438772754725e+14', '469339457186380.276410136272120035011198438772754725');
    T('1.2819429130391792511503973184804508867728894e+6', '1281942.9130391792511503973184804508867728894');
    T('1.9778e+3', '1977.8');
    T('2.456680359828937628024631306792185367572610021e+43', '24566803598289376280246313067921853675726100.21');
    T('5.25389225018085571689046925802871155628e+1', '52.5389225018085571689046925802871155628');
    T('1.733700532107e+8', '173370053.2107');
    T('1.9561099921e+5', '195610.99921');
    T('3.3409e+2', '334.09');
    T('6.09858715556186e+0', '6.09858715556186');
    T('3.20634106832106387482375790792609337383007782520694e+24', '3206341068321063874823757.90792609337383007782520694');
    T('1.46347126003930100207988814e+20', '146347126003930100207.98881400');
    T('2.717780449744210117995586561524987067807146882e+43', '27177804497442101179955865615249870678071468.82');
    T('2.86757572635270377540170639151e+22', '28675757263527037754017.0639151');
    T('3.3652257e+4', '33652.2570');
    T('1.161511687e+2', '116.1511687');
    T('3.38745832e+4', '33874.5832');
    T('1.2392653444609372603987061e+0', '1.2392653444609372603987061');
    T('3.2863728114574403555642096129516601553115371e+19', '32863728114574403555.642096129516601553115371');
    T('1.08862827358094757110520739146065028362703e+30', '1088628273580947571105207391460.65028362703');
    T('1.3488325541508602487577920722101277063863557818e+14', '134883255415086.02487577920722101277063863557818');
    T('1.96013732736436392e+13', '19601373273643.6392');
    T('4.798185890466e+2', '479.8185890466');
    T('1.696622337138949329874242519485119916519994150606e+39', '1696622337138949329874242519485119916519.994150606');
    T('5.50000572984970761183142593570950897913860587074643e+13', '55000057298497.0761183142593570950897913860587074643');
    T('4.9e+1', '49');
    T('2.353405108244768666141e+9', '2353405108.2447686661410');
    T('1.237978927714857736527530290155529e+0', '1.237978927714857736527530290155529');
    T('5.54113012e+1', '55.411301200');
    T('1.639709023131e+11', '163970902313.1');
    T('6.771810210029143571341e+20', '677181021002914357134.1');
    T('3.3659315728946599002939955548637694037e+13', '33659315728946.599002939955548637694037');
    T('5.448485697437811e+14', '544848569743781.1');
    T('2.93450799e+0', '2.93450799');
    T('4e+0', '4');
    T('5.751e+0', '5.751');
    T('7.61737213065110411e+15', '7617372130651104.110');
    T('4.958075942313945962737499249994583773691625306e+36', '4958075942313945962737499249994583773.691625306');
    T('1.358378342e+8', '135837834.20');
    T('2.048052995889503e+13', '20480529958895.03');
    T('6.79076388323848653822758720940258493627501e+10', '67907638832.38486538227587209402584936275010');
    T('8.67210047206219429882716380856729877038463e+37', '86721004720621942988271638085672987703.8463');
    T('3.5856103826876870700739098996695255188137082852345623e+4', '35856.103826876870700739098996695255188137082852345623');
    T('1.90496213987796617338452467459794976917964545462658e+15', '1904962139877966.17338452467459794976917964545462658');
    T('9.63174575036950144769050367165741e+0', '9.63174575036950144769050367165741');
    T('2.0768e+0', '2.0768');
    T('1.456779888489820409831915e+18', '1456779888489820409.831915');
    T('3.17858204994255447e+11', '317858204994.255447');
    T('1e+0', '1');
    T('1.9096483361096593551745691e+3', '1909.64833610965935517456910');
    T('1.06782878549e+8', '106782878.5490');
    T('1.939e+2', '193.9');
    T('1.35467074874780607458660332165839e+30', '1354670748747806074586603321658.39');
    T('1.3477262091e+0', '1.3477262091');
    T('1.310111103174762e+6', '1310111.103174762');
    T('1.3244499e+5', '132444.99');
    T('6.328608643684925104652431120058467829483124155656848e+5', '632860.86436849251046524311200584678294831241556568480');
    T('1.323139209814658927313821930784171e+3', '1323.1392098146589273138219307841710');
    T('4.47797097399588837e+9', '4477970973.995888370');
    T('1.443414e+4', '14434.14');
    T('1.913095284076879e+12', '1913095284076.879');
    T('2.51586474e+0', '2.51586474');
    T('2.9838935573505166030799e+9', '2983893557.3505166030799');
    T('1.573964479962393720481520537831844728449363575e+7', '15739644.79962393720481520537831844728449363575');
    T('2.496378e+2', '249.6378');
    T('1.484378e+0', '1.484378');
    T('5.21963089810195859130639819168579622e+24', '5219630898101958591306398.19168579622');
    T('1.8688440918945220036354844302195749309220888e+26', '186884409189452200363548443.02195749309220888');
    T('1.41126696817890014e+9', '1411266968.17890014');
    T('1.293182137273764465174297382284293177743591308e+15', '1293182137273764.465174297382284293177743591308');
    T('1.27466702071873897014853508266774e+7', '12746670.2071873897014853508266774');
    T('2.556582211914618427489836174964272160475e+25', '25565822119146184274898361.74964272160475');
    T('7.4140139575523848563560806694561950037847e+26', '741401395755238485635608066.94561950037847');
    T('1.9191049758936644006660416648962209488e+24', '1919104975893664400666041.66489622094880');
    T('2.31987505124515790876372807554587171781e+6', '2319875.05124515790876372807554587171781');
    T('1.15157815434449204928983e+10', '11515781543.4449204928983');
    T('3e+0', '3');
    T('2.917143980403974745273363447e+0', '2.917143980403974745273363447');
    T('4.191515340588519451276037876335997588593916166223e+8', '419151534.0588519451276037876335997588593916166223');
    T('1.5893308320910765122063972428358196502e+36', '1589330832091076512206397242835819650.2');
    T('1.564871863334923e+9', '1564871863.334923');
    T('3.4443994941943243107512237423766787557718741016702e+7', '34443994.941943243107512237423766787557718741016702');
    T('2.082688546929e+2', '208.2688546929');
    T('2.9839569879261869701414e+14', '298395698792618.69701414');
    T('1.18e+0', '1.18');
    T('4.649099416086369391962725101452597532757447601493062e+15', '4649099416086369.391962725101452597532757447601493062');
    T('5.1e+1', '51.0');
    T('1.350769554904381289856048945926109233407469025099095e+35', '135076955490438128985604894592610923.3407469025099095');
    T('1.28e+0', '1.28');
    T('1.686998748044557678256738078e+11', '168699874804.4557678256738078');
    T('2.725058665831622653286e+15', '2725058665831622.653286');
    T('3.4955955888709903930650733178004053283303e+27', '3495595588870990393065073317.8004053283303');
    T('2.71792743712e+3', '2717.92743712');
    T('3.765499741127994356229674741445362678695091917e+28', '37654997411279943562296747414.453626786950919170');
    T('8.48094736645795433828545542478643430012741252784e+2', '848.0947366457954338285455424786434300127412527840');
    T('3.252259575976e+8', '325225957.5976');
    T('3.6168564360780068610016611e+13', '36168564360780.0686100166110');
    T('1.3606305591180828419826623450927437361732403593e+14', '136063055911808.28419826623450927437361732403593');
    T('3.770936274989628272472264790077542e+21', '3770936274989628272472.264790077542');
    T('7.2511003873049904808574871389e+2', '725.11003873049904808574871389');
    T('6.62710314240271241900085261e+21', '6627103142402712419000.85261');
    T('7.5663149e+5', '756631.49');
    T('1.2062059e+1', '12.0620590');
    T('1.316934076085443594506478729911134464048168888698e+42', '1316934076085443594506478729911134464048168.888698');
    T('1.202953883353175026243055281903766950596e+16', '12029538833531750.26243055281903766950596');
    T('5.15639495337093127431811770279915335410152998e+4', '51563.9495337093127431811770279915335410152998');
    T('1.8e+1', '18');
    T('2.9510339604105305444519680156145845428324011e+35', '295103396041053054445196801561458454.28324011');
    T('9.991e+0', '9.991');
    T('2.32664625710064626302675792506518986748623800519362825e+45', '2326646257100646263026757925065189867486238005.19362825');
    T('1.15155883503801901e+9', '1151558835.03801901');
    T('2.3143e+1', '23.143');
    T('2.260422939172634138149555333552819e+1', '22.60422939172634138149555333552819');
    T('1.77998238e+7', '17799823.8');
    T('3.70098549253330116100410970723706977e+9', '3700985492.53330116100410970723706977');
    T('9.05841068683497571038524585222884373e+29', '905841068683497571038524585222.884373');
    T('2.104525700707071438146923949322482428742115356534e+28', '21045257007070714381469239493.22482428742115356534');
    T('7.024685053227353728838050757013069476546171879163e+24', '7024685053227353728838050.757013069476546171879163');
    T('4.46125872195266e+10', '44612587219.5266');
    T('1.5784e+0', '1.5784');
    T('7.2863751084180259357725026103712116366349e+23', '728637510841802593577250.26103712116366349');
    T('3.8320639752835419309258317842615e+19', '38320639752835419309.258317842615');
    T('5.022803549739540204835955185781e+8', '502280354.9739540204835955185781');
    T('1.52375875990883568458394783820252654691082e+8', '152375875.990883568458394783820252654691082');
    T('6.3061015767881141717464891e+15', '6306101576788114.1717464891');
    T('9.71891517062070497407760297561206525471485e+27', '9718915170620704974077602975.61206525471485');
    T('1.734234585265201766712574126945236309508118634425837e+0', '1.734234585265201766712574126945236309508118634425837');
    T('8.5765119797384477995e+4', '85765.119797384477995');
    T('5.13287988277059925228486890851140970386425187274e+8', '513287988.2770599252284868908511409703864251872740');
    T('1.658649165479737672749860660622867246548e+15', '1658649165479737.672749860660622867246548');
    T('3.4292698700438468428249935453830307216008845443e+3', '3429.2698700438468428249935453830307216008845443');
    T('6.4741508681889057488168545664375944171354681e+23', '647415086818890574881685.45664375944171354681');
    T('1.6942797476135028024654429183339522530812281e+31', '16942797476135028024654429183339.522530812281');
    T('4.2751009447385333857937408636958425779826e+4', '42751.009447385333857937408636958425779826');
    T('8.171245315549501039363411728044907e+8', '817124531.5549501039363411728044907');
    T('2.63956757213151457264967446772106e+14', '263956757213151.457264967446772106');
    T('5.613736432461e+9', '5613736432.461');
    T('4.78814e+3', '4788.14');
    T('4e+0', '4');
    T('4.37173137694e+3', '4371.73137694');
    T('2.2510403748584e+3', '2251.0403748584');
    T('5.06031783713235771e+6', '5060317.83713235771');
    T('1.33505152823751921282350949461e+18', '1335051528237519212.82350949461');
    T('7.600277071910098554342026310498032154036e+17', '760027707191009855.4342026310498032154036');
    T('1.876325989370064932735123485260012703584e+17', '187632598937006493.27351234852600127035840');
    T('3.87899618892199653560586133205982498365799449e+27', '3878996188921996535605861332.05982498365799449');
    T('6.2615173898555524447350478811537912100608066376e+42', '6261517389855552444735047881153791210060806.6376');
    T('9.413311009646174266728687126e+7', '94133110.09646174266728687126');
    T('1.2598016462494915983919933836115811e+13', '12598016462494.915983919933836115811');
    T('7.9083335654128601227093e+3', '7908.3335654128601227093');
    T('1.1816683721169472338636e+3', '1181.6683721169472338636');
    T('2.75369819574089239438702064921012202e+10', '27536981957.4089239438702064921012202');
    T('8.23458259735325932383e+10', '82345825973.5325932383');
    T('9.79385e+1', '97.9385');
    T('1e+0', '1');
    T('1.054786033040309868462e+15', '1054786033040309.868462');
    T('1.39529250356370349e+10', '13952925035.6370349');
    T('3.287519212843337e+8', '328751921.2843337');
    T('2.931680954526563928e+2', '293.1680954526563928');
    T('8.34563827640030036035405e+22', '83456382764003003603540.5');
    T('3.205638487e+0', '3.205638487');
    T('8.347784116251624503148e+14', '834778411625162.4503148');
    T('8.0108273810982547100003003762e+5', '801082.73810982547100003003762');
    T('5.41874345499627242543460793e+17', '541874345499627242.543460793');
    T('1.015683737878128e+2', '101.56837378781280');
    T('2.978259746276619202256565093853146061976952e+15', '2978259746276619.202256565093853146061976952');
    T('9.698293249507073809003898936542180557347147e+18', '9698293249507073809.003898936542180557347147');
    T('5.73284757676406662905522205317450204e+31', '57328475767640666290552220531745.0204');
    T('1.09314948671137385444979381052e+26', '109314948671137385444979381.052');
    T('6.657213e+1', '66.57213');
    T('5.442667e+0', '5.442667');
    T('1.497094368112928522082161622117e+17', '149709436811292852.2082161622117');
    T('8.86015267575800230944962406406137570452e+28', '88601526757580023094496240640.6137570452');
    T('8.7435115902725784785213467357972025071925716822259e+38', '874351159027257847852134673579720250719.25716822259');
    T('3.758112653964719298306554363807932e+15', '3758112653964719.298306554363807932');
    T('3.083180493316158e+7', '30831804.93316158');
    T('6.6753482e+1', '66.753482');
    T('1.511686617338690801597376192985164124335711475e+36', '1511686617338690801597376192985164124.335711475');
    T('2.561637088433158358807278932196e+1', '25.61637088433158358807278932196');
    T('9.7542804053855981164860021065e+18', '9754280405385598116.4860021065');
    T('3.862183052662673704009308528932528900914181684535e+42', '3862183052662673704009308528932528900914181.684535');
    T('1.735e+0', '1.735');
    T('1.68459549759090470803341044524240531008376812e+5', '168459.549759090470803341044524240531008376812');
    T('5.7719373345538594618757e+18', '5771937334553859461.8757');
    T('3.6601007679275500170303e+18', '3660100767927550017.0303');
    T('8.33735150265611827e+5', '833735.150265611827');
    T('1.01871087828059263083e+9', '1018710878.28059263083');
    T('1.730763e+3', '1730.763');
    T('2.0215513523e+6', '2021551.3523');
    T('3.169677580608477341e+6', '3169677.580608477341');
    T('1.36121484062885200254669e+8', '136121484.062885200254669');
    T('1.864593865698e+11', '186459386569.8');
    T('3.862360154857656116978304538802550298055142749413e+37', '38623601548576561169783045388025502980.55142749413');
    T('6.344348248e+6', '6344348.248');
    T('8.67366546264e+2', '867.366546264');
    T('4.29318860328279804558512081958216849305249e+4', '42931.8860328279804558512081958216849305249');
    T('4.5767492969936218926613265e+5', '457674.92969936218926613265');
    T('9.965617933646541e+10', '99656179336.46541');
    T('7.664031929445646163953676864248019955392457435e+19', '76640319294456461639.53676864248019955392457435');
    T('1.7557601024183018471184139748851879933153788645e+26', '175576010241830184711841397.48851879933153788645');
    T('3.59958e+1', '35.9958');
    T('3.128356271716333383085523918507809249225814121e+6', '3128356.271716333383085523918507809249225814121');
    T('1.9075527480204423875673047345572289921671553543e+7', '19075527.480204423875673047345572289921671553543');
    T('4.1705118500557863903425612212241828e+28', '41705118500557863903425612212.241828');
    T('1.18254857176614357421149713005254203068839516120630636e+10', '11825485717.6614357421149713005254203068839516120630636');
    T('7.02259913337121409861924646773988297e+3', '7022.599133371214098619246467739882970');
    T('1.146737732876241016454570289690559455455987995125423e+18', '1146737732876241016.454570289690559455455987995125423');
    T('1.355148008848587e+12', '1355148008848.587');
    T('1.015453056007471546100232031232797626130792432862e+30', '1015453056007471546100232031232.797626130792432862');
    T('2.8538363119129962635325743474766089129608347262536e+29', '285383631191299626353257434747.660891296083472625360');
    T('4.2227811926947467182312527e+0', '4.2227811926947467182312527');
    T('6.62781300059578283068526676555702438949382809379e+11', '662781300059.578283068526676555702438949382809379');
    T('7.132543524075010321e+11', '713254352407.5010321');
    T('2.6891265922739405441935129011630558514535121113e+0', '2.6891265922739405441935129011630558514535121113');
    T('2.6553300042224839703e+7', '26553300.042224839703');
    T('1.326288516626807519348228004e+12', '1326288516626.807519348228004');
    T('0e+0', '0');
    T('6.02519526915223e+9', '6025195269.15223');
    T('3.624787029215766960931977e+8', '362478702.9215766960931977');
    T('2.2216785095018655042906319049394e+10', '22216785095.018655042906319049394');
    T('1.42167287636895617482014385649865679412e+27', '1421672876368956174820143856.49865679412');
    T('1.41165522278933908893e+0', '1.41165522278933908893');
    T('9.6100608486394273371410715209128478362564609e+29', '961006084863942733714107152091.28478362564609');
    T('6.628723659367639746219006678948604246e+24', '6628723659367639746219006.678948604246');
    T('2.7977956855247594e+5', '279779.56855247594');
    T('2.031549e+5', '203154.9');
    T('1.255931797e+8', '125593179.7');
    T('5.6205236102596850984322041097e+16', '56205236102596850.9843220410970');
    T('2.03142821417788742960492e+19', '20314282141778874296.0492');
    T('1.91781433973790573018323517395847264353e+35', '191781433973790573018323517395847264.35300');
    T('6.8634738969456114664748639548209064365603770815e+10', '68634738969.456114664748639548209064365603770815');
    T('2.4116354386327449757134801702e+5', '241163.54386327449757134801702');
    T('3.36834833123607650204419305291838348810644243e+42', '3368348331236076502044193052918383488106442.43');
    T('1.2854387e+1', '12.854387');
    T('1.71911767411984485012054040057806822618e+28', '17191176741198448501205404005.7806822618');
    T('5.974376678071807862297048158076713536908142505051762e+0', '5.974376678071807862297048158076713536908142505051762');
    T('1.75040573368172e+0', '1.75040573368172');
    T('7.59296739794577295967023894e+19', '75929673979457729596.7023894');
    T('5.9387271258615e+0', '5.9387271258615');
    T('1.41e+1', '14.1');
    T('6.520718977022294517603438651524e+13', '65207189770222.94517603438651524');
    T('4.0504464748488527e+13', '40504464748488.527');
    T('4.9339025e+4', '49339.025');
    T('6.5436938338545276642922e+5', '654369.38338545276642922');
    T('6.614070881259392274974020452736433936484946441479e+14', '661407088125939.2274974020452736433936484946441479');
    T('3.6562357737624542715177216071738074863798786e+36', '3656235773762454271517721607173807486.3798786');
    T('4.483580186204558328366014e+16', '44835801862045583.28366014');
    T('8.786e+1', '87.86');
    T('3.2320103419855255742057273942480043987e+6', '3232010.3419855255742057273942480043987');
    T('4.46656766000098808044857967014414e+16', '44665676600009880.8044857967014414');
    T('1.02292056093528342365e+13', '10229205609352.8342365');
    T('8.57520635819763350659219863901837233e+21', '8575206358197633506592.19863901837233');
    T('1.245913688722124452316952395765916246e+21', '1245913688722124452316.9523957659162460');
    T('8.8964578294856908705314945178917896807516003057782324e+46', '88964578294856908705314945178917896807516003057.782324');
    T('8.7449512208056038986439889808e+7', '87449512.208056038986439889808');
    T('2.3e+2', '230.0');
    T('1.5579884689150031101693496143656840264980386818704e+23', '155798846891500311016934.961436568402649803868187040');
    T('8.5051864582679908692564183699252612773409509082078e+37', '85051864582679908692564183699252612773.409509082078');
    T('8.8798692074080075787375315447486123809923959833e+26', '887986920740800757873753154.47486123809923959833');
    T('6.77345401358539738139850599762127933983100324110805558e+4', '67734.5401358539738139850599762127933983100324110805558');
    T('2.347749726836225014840282995092832754e+20', '234774972683622501484.0282995092832754');
    T('7.429558958688e+0', '7.429558958688');
    T('1.7882214813873e+0', '1.7882214813873');
    T('4.193192196e+6', '4193192.196');
    T('4.749767618513507376588211479214898450387625965e+9', '4749767618.513507376588211479214898450387625965');
    T('1.977391043687779e+5', '197739.1043687779');
    T('2.6484576511218782036901501152986249531234497168718e+40', '26484576511218782036901501152986249531234.497168718');
    T('1.920439145880135238583976259936021e+3', '1920.439145880135238583976259936021');
    T('1.51897109592862249460404e+8', '151897109.592862249460404');
    T('5.0235407304542145031825034116450744076803325e+15', '5023540730454214.5031825034116450744076803325');
    T('7.6404110248649153016903650163544816581006668605e+23', '764041102486491530169036.50163544816581006668605');
    T('3.7874929502138e+0', '3.7874929502138');
    T('1.41787251451628994917275472e+24', '1417872514516289949172754.72');
    T('4.2104854597998748621925225253125873386e+31', '42104854597998748621925225253125.87338600');
    T('1.26567160108986726661104934542275915987889e+7', '12656716.0108986726661104934542275915987889');
    T('2.8968894398808777e+4', '28968.894398808777');
    T('2.8297292e+2', '282.97292');
    T('2.603742338988784570262735209585265906786049e+12', '2603742338988.784570262735209585265906786049');
    T('4.62131e+2', '462.131');
    T('2.22735863520665290933246331848e+24', '2227358635206652909332463.31848');
    T('4.53886361459e+10', '45388636145.9');
    T('6.6860727859954777667223002708162369e+32', '668607278599547776672230027081623.69');
    T('2.8062722e+0', '2.8062722');
    T('6.0842062525063451016806e+10', '60842062525.063451016806');
    T('6.8357371699714755863090081855483865237790749503e+22', '68357371699714755863090.081855483865237790749503');
    T('3.35802101243809011e+10', '33580210124.3809011');
    T('8.43490001143656777280840623008615269579365614931775e+0', '8.43490001143656777280840623008615269579365614931775');
    T('1.9235808728891520564223264242417980260344158021186e+8', '192358087.28891520564223264242417980260344158021186');
    T('1.41994286356037056979411e+12', '1419942863560.37056979411');
    T('1.78557003827723798504978687588959254956675353035516771e+37', '17855700382772379850497868758895925495.6675353035516771');
    T('1.2918168334241255620676751949906289e+11', '129181683342.41255620676751949906289');
    T('3.9318765130442377729598885e+1', '39.318765130442377729598885');
    T('3.88694369953891826304770521327308322619393047272853992e+29', '388694369953891826304770521327.308322619393047272853992');
    T('1.0841221160385599396e+8', '108412211.60385599396');
    T('2.927924105979441123308231e+10', '29279241059.79441123308231');
    T('9.793009327446332791991220439204599563053527e+23', '979300932744633279199122.0439204599563053527');
    T('9.423287306701960190921439346729208723159342075073e+5', '942328.7306701960190921439346729208723159342075073');
    T('9.32095532579704529e+14', '932095532579704.529');
    T('4e+0', '4');
    T('6.53989976127167541385228442576702092890124e+35', '653989976127167541385228442576702092.89012400');
    T('1.19250623478e+7', '11925062.3478');
    T('1.865458672e+0', '1.865458672');
    T('3.316183057661896194198642778725380227773158022030233e+12', '3316183057661.896194198642778725380227773158022030233');
    T('8.5601184555567625475596670332086089588e+15', '8560118455556762.5475596670332086089588');
    T('1.80118254923064703459780878790624318318152203e+37', '18011825492306470345978087879062431831.8152203');
    T('8.31702685620470833413491412813875698623351991849704e+45', '8317026856204708334134914128138756986233519918.49704');
    T('4.158534334648275764e+11', '415853433464.8275764');
    T('2.19723302477e+0', '2.19723302477');
    T('2.029177448999173372119989892440779088e+22', '20291774489991733721199.89892440779088');
    T('4.645682747987148208758271e+14', '464568274798714.8208758271');
    T('2.6112628e+6', '2611262.8');
    T('1.27260908637e+0', '1.27260908637');
    T('2.386528806365172395257137e+5', '238652.88063651723952571370');
    T('2.64450278721870797393766005966148238494e+8', '264450278.721870797393766005966148238494');
    T('1.988688294222795007e+6', '1988688.294222795007');
    T('7.078660611250891118657496594735238821341408353e+35', '707866061125089111865749659473523882.1341408353');
    T('0e+0', '0');
    T('1.096715852238427070115466117322983113926e+36', '1096715852238427070115466117322983113.926');
    T('8.3033913e+4', '83033.913');
    T('4.015204714209691387124917361818013823115690954e+5', '401520.4714209691387124917361818013823115690954');
    T('1.18391666839775944e+9', '1183916668.39775944');
    T('2.2995421984176008066735e+7', '22995421.984176008066735');
    T('9.03e+0', '9.03');
    T('5.332359881257917566876361208399e+0', '5.3323598812579175668763612083990');
    T('7.102133e+3', '7102.133');
    T('1.1483640369724895747740049e+19', '11483640369724895747.740049');
    T('2.86532834287791967206284176556306849118939923884e+42', '2865328342877919672062841765563068491189399.23884');
    T('9.076417888165894607318040129e+17', '907641788816589460.7318040129');
    T('7.3646447223309353153091289569473837758243e+14', '736464472233093.53153091289569473837758243');
    T('2.181883e+4', '21818.83');
    T('3.386300476012467818859167684043147659263155e+9', '3386300476.012467818859167684043147659263155');
    T('2.51484028084025506424805997802620148071482e+13', '25148402808402.55064248059978026201480714820');
    T('9.4435575887407391033782269887533990447300682186075521e+24', '9443557588740739103378226.9887533990447300682186075521');
    T('1.6886715415033e+8', '168867154.15033');
    T('2.5034373528604861134460194872790430617196344388725e+11', '250343735286.04861134460194872790430617196344388725');
    T('2.196175657558897384981748559894e+29', '219617565755889738498174855989.4');
    T('9.76199813364186117286131817353e+8', '976199813.364186117286131817353');
    T('2.19717393409454748689882068559316753929782544245361e+9', '2197173934.09454748689882068559316753929782544245361');
    T('3.458346873044019889e+14', '345834687304401.98890');
    T('1.7224191355912435440309996131683233957399227e+29', '172241913559124354403099961316.83233957399227');
    T('2.9118304597001059668e+13', '29118304597001.0596680');
    T('8.089765204288655830674576596713309516483099e+31', '80897652042886558306745765967133.09516483099');
    T('4.238609505643148138368920789804547924618071e+16', '42386095056431481.38368920789804547924618071');
    T('6.632956460328161742904032219604e+9', '6632956460.328161742904032219604');
    T('5.149280532588990421575423306958842719315081438104696e+2', '514.9280532588990421575423306958842719315081438104696');
    T('2.07166412218470429605806831507e+24', '2071664122184704296058068.31507');
    T('3.074159e+0', '3.074159');
    T('8.881411022677226686076012925366e+8', '888141102.2677226686076012925366');
    T('1.4327673900218857772756403845335511690820534667844161e+25', '14327673900218857772756403.845335511690820534667844161');
    T('8.11258981514701594464134633523542919936061665628845e+9', '8112589815.14701594464134633523542919936061665628845');
    T('5.96685124808828989098666066158574653883332e+34', '59668512480882898909866606615857465.3883332');
    T('5.57183574e+6', '5571835.74');
    T('6.8501903843096e+0', '6.8501903843096');
    T('1.5942819569037073307050366953094e+20', '159428195690370733070.50366953094');
    T('2.27330909197482e+2', '227.330909197482');
    T('2.982752994e+1', '29.82752994');
    T('2.6621448010147457294e+3', '2662.14480101474572940');
    T('1.086669601366824519299e+5', '108666.9601366824519299');
    T('1.18736e+1', '11.8736');
    T('2.529e+2', '252.9');
    T('9.53927547001352070346930665292474419009985922637627e+0', '9.53927547001352070346930665292474419009985922637627');
    T('1.41547817128845761243461584e+17', '141547817128845761.243461584');
    T('4.913427130278990825751597676478e+5', '491342.7130278990825751597676478');
    T('1.92247886737227536336e+7', '19224788.67372275363360');
    T('3.1497590158477647299507840130228337905235603e+22', '31497590158477647299507.840130228337905235603');
    T('1.68055184413854558094765106936369527289778050531295766e+40', '16805518441385455809476510693636952728977.8050531295766');
    T('1.8255321351888780569641286767450275327e+7', '18255321.351888780569641286767450275327');
    T('1.56201751119240301681993262341142444952941e+6', '1562017.51119240301681993262341142444952941');
    T('5.122704714291e+9', '5122704714.291');
    T('3.82474822197843e+12', '3824748221978.43');
    T('1.525347830613268824053492683517588e+9', '1525347830.613268824053492683517588');
    T('6.145278e+4', '61452.78');
    T('4.99832566753824717449034e+0', '4.99832566753824717449034');
    T('1.308323895602314056642638704891037710277185276986927e+40', '13083238956023140566426387048910377102771.85276986927');
    T('5.18915522348e+4', '51891.5522348');
    T('1.4209994e+6', '1420999.4');
    T('1.503265451278288e+7', '15032654.51278288');
    T('3.152117986796142139453802611094289123987929469e+9', '3152117986.796142139453802611094289123987929469');
    T('2.978007679998703216707397018589782695e+13', '29780076799987.03216707397018589782695');
    T('5.8512931044e+7', '58512931.044');
    T('3.295254020832544012761679098484189727227833e+41', '329525402083254401276167909848418972722783.3');
    T('8.72742250969672731429969186796754051463901e+14', '872742250969672.731429969186796754051463901');
    T('2.9072795594202554837396927026502083307998e+35', '290727955942025548373969270265020833.07998');
    T('2.34518496092800925568908809696371538989759e+27', '2345184960928009255689088096.96371538989759');
    T('4.02951776768e+4', '40295.17767680');
    T('1.3689991541714377699994556138826283759e+29', '136899915417143776999945561388.26283759');
    T('5.375319663285375407847169258e+20', '537531966328537540784.71692580');
    T('2.4838816191142589e+0', '2.4838816191142589');
    T('1e+1', '10');
    T('1.32118e+2', '132.118');
    T('3.167368488391197862244748797056311701e+1', '31.6736848839119786224474879705631170100');
    T('1.2923240005675836581430542022e+7', '12923240.005675836581430542022');
    T('2.45198799455822803503200760843340001104401687715e+1', '24.5198799455822803503200760843340001104401687715');
    T('5.017419011122159795604649952660143561601682847055e+9', '5017419011.122159795604649952660143561601682847055');
    T('1.528293223258e+9', '1528293223.258');
    T('3.36806306935615523082781617061106400210297708457466e+40', '33680630693561552308278161706110640021029.7708457466');
    T('2.24276e+4', '22427.6');
    T('1.452908e+5', '145290.8');
    T('1.730864210471288302007789478210233615399858e+21', '1730864210471288302007.789478210233615399858');
    T('1.98572237681530700377986658329680798e+32', '198572237681530700377986658329680.798');
    T('5.3625630032708e+2', '536.25630032708');
    T('1.279811707773498944767714003821019551877654490239992e+40', '12798117077734989447677140038210195518776.54490239992');
    T('1.13478473099428793759e+1', '11.3478473099428793759');
    T('3.8084703153428358911632007387325702215541573567757e+45', '3808470315342835891163200738732570221554157356.7757');
    T('1.63132182288401410656e+16', '16313218228840141.0656');
    T('1.2883403439965987126193143646e+8', '128834034.39965987126193143646');
    T('1.1337344058722593784502983693368481614e+2', '113.3734405872259378450298369336848161400');
    T('1.3810837466649418967673665311541682533151e+37', '13810837466649418967673665311541682533.151');
    T('3e+0', '3');
    T('1.77729190576542704583828487792283968438443988677e+27', '1777291905765427045838284877.92283968438443988677');
    T('1.75045497942020277400523024600228150483494020715374e+39', '1750454979420202774005230246002281504834.94020715374');
    T('9.4295017753469131679077273e+2', '942.95017753469131679077273');
    T('3.854570457401828825080672674176371617252885568525e+42', '3854570457401828825080672674176371617252885.568525');
    T('6.976781259116059962789065637569e+19', '69767812591160599627.89065637569');
    T('1.983953867384017752e+8', '198395386.7384017752');
    T('8.18621623940664383e+3', '8186.21623940664383');
    T('4.678566e+5', '467856.6');
    T('4.37454863655836062626765123683424e+5', '437454.863655836062626765123683424');
    T('7.6e+1', '76');
    T('1.356170189773197190024227714709209e+20', '135617018977319719002.42277147092090');
    T('2.524007970392697273821209419290910149e+32', '252400797039269727382120941929091.0149');
    T('2.2996e+1', '22.996');
    T('9.19536554437673037098591509995026767872538589562761e+45', '9195365544376730370985915099950267678725385895.62761');
    T('7.52981388027044914e+7', '75298138.8027044914');
    T('6.0233392042283e+4', '60233.3920422830');
    T('1.197371690422415645601906387900091876e+8', '119737169.0422415645601906387900091876');
    T('2.5e+0', '2.50');
    T('3.0513e+3', '3051.3');
    T('3.4378929513006354001139704e+1', '34.378929513006354001139704');
    T('2.7688798583863475634e+16', '27688798583863475.634');
    T('3.70733017769177e+3', '3707.33017769177');
    T('1.183552401830090152504539986275312716505e+23', '118355240183009015250453.9986275312716505');
    T('5.5532923555574e+0', '5.5532923555574');
    T('1.007309381e+2', '100.7309381');
    T('4.26e+0', '4.26');
    T('3.49082770706667347770412719905221035269e+36', '3490827707066673477704127199052210352.69');
    T('1.09667668559108738350612984427121459528827e+35', '109667668559108738350612984427121459.528827');
    T('6.7117017423837651934e+2', '671.17017423837651934');
    T('3.5513336510667288609743743e+5', '355133.36510667288609743743');
    T('2.733911637796059758003823848720999245e+12', '2733911637796.059758003823848720999245');
    T('4.7727335155560138656406364912e+21', '4772733515556013865640.6364912');
    T('7.1707742358130425165e+15', '7170774235813042.5165');
    T('1.2373e+0', '1.2373');
    T('1.95864917284678233803922075321436e+16', '19586491728467823.3803922075321436');
    T('1.39511686402859313308703279767446898138923566399e+34', '13951168640285931330870327976744689.8138923566399');
    T('5.45166767236394220015660379974682e+18', '5451667672363942200.15660379974682');
    T('5.7e+1', '57');
    T('2.008185112618419540343890697023120161916064586e+15', '2008185112618419.5403438906970231201619160645860');
    T('1.83547377703310193838625323613810597146659646688e+38', '183547377703310193838625323613810597146.659646688');
    T('1.86963380149e+6', '1869633.80149');
    T('9.0643926657221127599225e+0', '9.0643926657221127599225');
    T('3.387626848249516099010371969816649097243645e+35', '338762684824951609901037196981664909.7243645');
    T('2.38574e+4', '23857.4');
    T('4.854485927348228918413753679671472155071e+35', '485448592734822891841375367967147215.5071');
    T('1.1145412703080371165560398973334199e+32', '111454127030803711655603989733341.99');
    T('1.272285349955021228893214839127637825683e+4', '12722.85349955021228893214839127637825683');
    T('9.4315857521037576638711244118020078159357613e+2', '943.15857521037576638711244118020078159357613');
    T('2.130358009562539933673297313517219743395888826178226e+42', '2130358009562539933673297313517219743395888.826178226');
    T('1.788350524694006198162174980034358528e+19', '17883505246940061981.621749800343585280');
    T('1.26860370623505030126846090707878e+22', '12686037062350503012684.6090707878');
    T('2e+0', '2');
    T('7.97066672594e+7', '79706667.2594');
    T('2.0152e+2', '201.52');
    T('5.00637717193322215204e+6', '5006377.17193322215204');
    T('3.01590120278994390749132552334544583e+0', '3.01590120278994390749132552334544583');
    T('3.717693087339892916526272044454e+24', '3717693087339892916526272.044454');
    T('2.54829661524758873955898224553317416437e+3', '2548.29661524758873955898224553317416437');
    T('7.9072523913538861636033504135822e+1', '79.072523913538861636033504135822');
    T('1.866503037113138887457365815904225e+28', '18665030371131388874573658159.04225');
    T('1.3950429e+5', '139504.29');
    T('1.0207356655e+5', '102073.56655');
    T('3.2360853090691879699503619670118780802618636754e+9', '3236085309.0691879699503619670118780802618636754');
    T('5.47452697575793862266267385061152164e+19', '54745269757579386226.6267385061152164');
    T('6.1014855783682768e+15', '6101485578368276.8');
    T('5.31612911388139343797172e+0', '5.31612911388139343797172');
    T('5.91082171916830390736766017658603922e+26', '591082171916830390736766017.658603922');
    T('2.8172659641631766440010830065e+19', '28172659641631766440.010830065');
    T('4.1202100041496082281283466e+5', '412021.00041496082281283466');
    T('1.976856495373097710107538467325977284465274416e+19', '19768564953730977101.07538467325977284465274416');
    T('1.171504106314324317761655700166834e+32', '117150410631432431776165570016683.4');
    T('1.027537448230010431061775478e+10', '10275374482.30010431061775478');
    T('2.341237435879754661702897098867289936e+3', '2341.237435879754661702897098867289936');
    T('1.0588321785973619575e+15', '1058832178597361.9575');
    T('4.2335454826279360873889814771443479909805203261e+39', '4233545482627936087388981477144347990980.5203261');
    T('4.9e+1', '49');
    T('1.4101138066556191e+0', '1.4101138066556191');
    T('1.3899701029842433287999040786946262e+11', '138997010298.42433287999040786946262');
    T('2.14e+1', '21.4');
    T('2.575344258e+6', '2575344.258');
    T('1.3691520780202966734596879557205614425e+23', '136915207802029667345968.79557205614425');
    T('1.079727307987010933702e+19', '10797273079870109337.02');
    T('7.0961299409455830259875773863405485e+3', '7096.1299409455830259875773863405485');
    T('6.071949209634075943049707175059387500694121405338866e+45', '6071949209634075943049707175059387500694121405.338866');
    T('9.688112361029101083207689619288615989e+17', '968811236102910108.3207689619288615989');
    T('2.80923460488873864236890941794609081552183193e+41', '280923460488873864236890941794609081552183.193');
    T('1.83294035069640418182259900657124e+15', '1832940350696404.181822599006571240');
    T('1.330407118626933239746930889885854606102e+7', '13304071.18626933239746930889885854606102');
    T('8.614745245083701401256805923197041044646493828616208e+22', '86147452450837014012568.05923197041044646493828616208');
    T('1.021162949761147474655069846e+15', '1021162949761147.474655069846');
    T('2.22124197e+6', '2221241.97');
    T('1.9269783432665189582619e+12', '1926978343266.5189582619');
    T('2.4458042e+3', '2445.8042');
    T('1.24442787448470619843211196913235e+6', '1244427.87448470619843211196913235');
    T('6.5795027200660039322532e+7', '65795027.200660039322532');
    T('2.124371034489911604376156663456697624832e+36', '2124371034489911604376156663456697624.832');
    T('4.7518554e+3', '4751.8554');
    T('5.577243873330495803e+5', '557724.3873330495803');
    T('1.4411990735775980636079874248393636e+28', '14411990735775980636079874248.393636');
    T('3.532984961283621176064142e+22', '35329849612836211760641.42');
    T('6.41099968454591750260016248126e+14', '641099968454591.750260016248126');
    T('1.876802099566385814736323517234163782171196608110171e+38', '187680209956638581473632351723416378217.1196608110171');
    T('2.712536235933694424272454724983924750851e+6', '2712536.235933694424272454724983924750851');
    T('1.819734474513556013e+8', '181973447.4513556013');
    T('3.0320484961616175305658133728e+14', '303204849616161.75305658133728');
    T('1.9739814652160018980576469874802639653674373e+38', '197398146521600189805764698748026396536.74373');
    T('6.0551922025938302e+10', '60551922025.938302');
    T('3.323596297e+8', '332359629.7');
    T('5.7458544961785021027335254905977754345607e+38', '574585449617850210273352549059777543456.07');
    T('3.43869507567190104395123310443283792e+2', '343.869507567190104395123310443283792');
    T('3.964855738823341944355183015e+26', '396485573882334194435518301.5');
    T('4.75631864588339215e+12', '4756318645883.39215');
    T('3.1e+1', '31');
    T('3.750923398533815812530175e+17', '375092339853381581.2530175');
    T('1.25649942741256e+2', '125.649942741256');
    T('5.4854980319307537893979503491381982942042465844747093e+45', '5485498031930753789397950349138198294204246584.4747093');
    T('1.2940661791311373e+14', '129406617913113.73');
    T('2.90385571775480031587051447549374193061077e+10', '29038557177.5480031587051447549374193061077');
    T('9.1040729817423482924856979976434906087350486636207e+3', '9104.0729817423482924856979976434906087350486636207');
    T('1.57834489e+4', '15783.4489');
    T('2.319e+2', '231.9');
    T('5.522835788261589143141105742945478589362e+16', '55228357882615891.43141105742945478589362');
    T('8.7417851321101080306727e+15', '8741785132110108.0306727');
    T('9.96e+0', '9.96');
    T('1.343046943929639e+14', '134304694392963.9');
    T('2.21577825581227700070639623743724540952019026803e+0', '2.21577825581227700070639623743724540952019026803');
    T('5.64651860171575748936148456309414e+12', '5646518601715.75748936148456309414');
    T('2.71790519140131463205484467292975145e+33', '2717905191401314632054844672929751.450');
    T('2.46128778261687122384096383441327463273586448997789422e+8', '246128778.261687122384096383441327463273586448997789422');
    T('1.51e+0', '1.510');
    T('1.34417538839677e+6', '1344175.38839677');
    T('1.1967386179518011766805527782003e+0', '1.1967386179518011766805527782003');
    T('3.1240079864434111687577043352352970435153e+8', '312400798.64434111687577043352352970435153');
    T('3.125225939e+3', '3125.2259390');
    T('2.6130649e+4', '26130.649');
    T('6.3203802234999588e+14', '632038022349995.88');
    T('1.300191903723243249151204409158028743867e+5', '130019.1903723243249151204409158028743867');
    T('1.5457655328224382807201057293747835e+28', '15457655328224382807201057293.747835');
    T('2.09985076272906724e+4', '20998.50762729067240');
    T('2.2982784786e+6', '2298278.4786');
    T('2.73013672764e+2', '273.013672764');
    T('7.15593e+3', '7155.93');
    T('1.05530065668194919764462354032281e+20', '105530065668194919764.462354032281');
    T('7.067238e+5', '706723.8');
    T('7.101666481976457556024951459e+20', '710166648197645755602.49514590');
    T('1.98927384634532163e+13', '19892738463453.2163');
    T('9.29554688795640972981057180708607820032179861e+37', '92955468879564097298105718070860782003.2179861');
    T('6.357e+2', '635.7');
    T('3.840652373682260866164588121591718905795553491603e+8', '384065237.3682260866164588121591718905795553491603');
    T('9.648833805019588496096876819771292813546e+29', '964883380501958849609687681977.1292813546');
    T('1.01396650233101861694092927e+7', '10139665.0233101861694092927');
    T('1.1003348820123809214388166064277331414286102008e+3', '1100.3348820123809214388166064277331414286102008');
    T('2.36753216678948394164594831443879485957e+22', '23675321667894839416459.4831443879485957');
    T('4.793420545453058844039438100377715257448e+3', '4793.420545453058844039438100377715257448');
    T('2.8594271318287473088601837035675887941e+16', '28594271318287473.08860183703567588794100');
    T('1.6206370599062305392884719395678561e+31', '16206370599062305392884719395678.561');
    T('8.37074216450166132786697142264216365e+10', '83707421645.0166132786697142264216365');
    T('5.72070708e+6', '5720707.08');
    T('3.44337507384153235321301316699932527121036236885e+7', '34433750.7384153235321301316699932527121036236885');
    T('5.1e+1', '51');
    T('2.1015822918024812834120182488777123107013633378167186e+42', '2101582291802481283412018248877712310701363.3378167186');
    T('1.6372374238765434156758128e+15', '1637237423876543.4156758128');
    T('3.3419685364707547838979720464860359274118133481936e+14', '334196853647075.47838979720464860359274118133481936');
    T('2.8068807310597786792e+3', '2806.8807310597786792');
    T('4.845691098565776387667111068909874902599225606796909e+22', '48456910985657763876671.11068909874902599225606796909');
    T('4.63492511826172e+7', '46349251.1826172');
    T('8.25120492e+0', '8.25120492');
    T('2.5871391624943067662520803430464711250480700861462709e+13', '25871391624943.067662520803430464711250480700861462709');
    T('6.862831873483024223892481752e+21', '6862831873483024223892.4817520');
    T('4.270995713570409133606448268988455e+6', '4270995.713570409133606448268988455');
    T('3.21689066758930720484611614541085683569991722e+1', '32.1689066758930720484611614541085683569991722');
    T('4.2006919228915e+6', '4200691.9228915');
    T('2.111647930356982460265657710379876346e+34', '21116479303569824602656577103798763.46');
    T('4.87551174669e+5', '487551.174669');
    T('1.5091257783488078659575278297164846061663847e+26', '150912577834880786595752782.97164846061663847');
    T('1.237701316659693172445623860334e+1', '12.37701316659693172445623860334');
    T('5.3635904e+4', '53635.904');
    T('1.5977967842379292197961047064611986628e+22', '15977967842379292197961.047064611986628');
    T('6.834482866e+0', '6.8344828660');
    T('3.9109e+3', '3910.9');
    T('2.05461117535930705870869638710707996175293568e+5', '205461.117535930705870869638710707996175293568');
    T('1.2070553345143881176541757184e+18', '1207055334514388117.6541757184');
    T('9.04753411072173104408842163075173270844e+2', '904.753411072173104408842163075173270844');
    T('3.639101145152501183326810611536036936191190074e+10', '36391011451.525011833268106115360369361911900740');
    T('1.994669617516385887919849012130839536463335664239342e+50', '199466961751638588791984901213083953646333566423934.2');
    T('9.7856070170986006874778e+3', '9785.6070170986006874778');
    T('6.1871253720389630778712195614969724340856119729e+35', '618712537203896307787121956149697243.40856119729');
    T('1.208e+1', '12.08');
    T('1.198080560663539944762503564196454e+22', '11980805606635399447625.03564196454');
    T('2.5328238533704762297890589157e+22', '25328238533704762297890.5891570');
    T('5.552e+2', '555.2');
    T('2.01773928527273821442e+12', '2017739285272.738214420');
    T('7.1036358731897125539910962e+19', '71036358731897125539.910962');
    T('3.01168351390557005880750701501256409e+28', '30116835139055700588075070150.1256409');
    T('8.873816195265370349563768092711378467e+0', '8.873816195265370349563768092711378467');
    T('5.31989959593655063323434328335073160690174981170061e+33', '5319899595936550633234343283350731.60690174981170061');
    T('1.170061672875437e+9', '1170061672.875437');
    T('2.631e+0', '2.631');
    T('6.992802197718685e+6', '6992802.197718685');
    T('6.4873437617951852808027501577341488220457588954e+3', '6487.3437617951852808027501577341488220457588954');
    T('4.1846611240468792953070738254513e+26', '418466112404687929530707382.54513');
    T('9.65827406708491826272070488217543356756570950537e+5', '965827.406708491826272070488217543356756570950537');
    T('2.680770183874090982680392187869e+13', '26807701838740.909826803921878690');
    T('6.7565808330233111784457537072905e+7', '67565808.330233111784457537072905');
    T('2.38408874905822388315809187608e+17', '238408874905822388.315809187608');
    T('2.447793231118383053396286312152e+27', '2447793231118383053396286312.152');
    T('4.412e+0', '4.412');
    T('6.9222338533800975058347547493064339259e+10', '69222338533.800975058347547493064339259');
    T('1.72125107835943e+12', '1721251078359.4300');
    T('6.607121e+4', '66071.21');
    T('7.03864465898721081635229140790621289565513e+24', '7038644658987210816352291.40790621289565513');
    T('5.7060362541741681216530676958032280095247281014555554e+31', '57060362541741681216530676958032.280095247281014555554');
    T('2.7482039096119173840576395905241256e+28', '27482039096119173840576395905.241256');
    T('6.4362e+2', '643.62');
    T('1.72406543968787e+12', '1724065439687.87');
    T('3.7500096080724676535769717e+8', '375000960.80724676535769717');
    T('3.3903249492445e+4', '33903.249492445');
    T('3.985710962147295694116665218405340482945689e+6', '3985710.962147295694116665218405340482945689');
    T('5.9597233999e+2', '595.972339990');
    T('5.065730949256446196418568178865493e+9', '5065730949.256446196418568178865493');
    T('1.6265740670254208186e+10', '16265740670.254208186000');
    T('2.471885430096e+2', '247.1885430096');
    T('9.764510566491700832916679297457385114302061622402e+27', '9764510566491700832916679297.457385114302061622402');
    T('1.100445271296546289546e+20', '110044527129654628954.6');
    T('8.561528590365492629836625093938e+28', '85615285903654926298366250939.38');
    T('7.31603872157247934437172663482574448376720106618697e+0', '7.31603872157247934437172663482574448376720106618697');
    T('2.75299872462131924258265574524762989786191459317e+7', '27529987.2462131924258265574524762989786191459317');
    T('6.7303660756133592428732533499288191492268777722116295e+8', '673036607.56133592428732533499288191492268777722116295');
    T('5.44277296e+3', '5442.77296');
    T('1.727400264734055421671805513613990565159952e+17', '172740026473405542.16718055136139905651599520');
    T('1.410636835060372309578742278281108273215901435e+42', '1410636835060372309578742278281108273215901.435');
    T('4.92125270107294889416151811122942461554e+6', '4921252.70107294889416151811122942461554');
    T('8.263574210550744338917479e+4', '82635.74210550744338917479');
    T('3.301295895927052871052e+18', '3301295895927052871.052');
    T('2.8345346500631739e+9', '2834534650.0631739');
    T('2.18804350994050349e+5', '218804.350994050349');
    T('3.383e+2', '338.30');
    T('7.581461444218101096345865e+3', '7581.461444218101096345865');
    T('1.584289425751937525462e+15', '1584289425751937.525462');
    T('1.202636592001646399174824119297846262e+35', '120263659200164639917482411929784626.2');
    T('1.72579256998649794e+15', '1725792569986497.94');
    T('1.109e+0', '1.109');
    T('6.40345690864260982946869972537190219e+32', '640345690864260982946869972537190.219');
    T('7.5479229773185178963182361773e+8', '754792297.73185178963182361773');
    T('7.78534635586647281e+15', '7785346355866472.81');
    T('4.5177508859669314863310078725e+4', '45177.508859669314863310078725');
    T('8.488973363671775235458481e+7', '84889733.63671775235458481');
    T('4e+0', '4');
    T('2.160257894346090658681225233774014977725364e+11', '216025789434.6090658681225233774014977725364');
    T('1.607e+1', '16.07');
    T('2.0206643483788145768022873735045719015739162865885754e+19', '20206643483788145768.022873735045719015739162865885754');
    T('1e+0', '1');
    T('6.086027486877e+4', '60860.27486877');
    T('6.49464610960956342354353921e+18', '6494646109609563423.5435392100');
    T('9.043087147437339975894215149e+5', '904308.7147437339975894215149');
    T('9.1415795e+4', '91415.795');
    T('1.27325e+3', '1273.25');
    T('1.1457620373836988522455309158001309e+23', '114576203738369885224553.09158001309');
    T('3.5258358268876488e+3', '3525.8358268876488');
    T('5.903854119252757788054640537784867975213379818757043e+12', '5903854119252.757788054640537784867975213379818757043');
    T('2.85971614953528217489857558639088593436284e+10', '28597161495.352821748985755863908859343628400');
    T('6.32147605459582535823869726726906892032424730199715e+12', '6321476054595.82535823869726726906892032424730199715');
    T('1.66883e+3', '1668.83');
    T('8.77804543701627148053590314981878177298945834797e+34', '87780454370162714805359031498187817.7298945834797');
    T('1.12758900853879813587345336618836e+24', '1127589008538798135873453.366188360');
    T('2.650452875488521637e+15', '2650452875488521.637');
    T('9.81e+1', '98.1');
    T('9.4897220794023859948771e+7', '94897220.794023859948771');
    T('8.2072647415486658112286228544358927e+27', '8207264741548665811228622854.4358927');
    T('1.2676e+1', '12.676');
    T('2.6048528862310648794425199635976449765151357e+40', '26048528862310648794425199635976449765151.357');
    T('1e+0', '1');
    T('2.2e+1', '22');
    T('1.3672902326585293252722797324062e+4', '13672.902326585293252722797324062');
    T('8.2594112194437e+4', '82594.112194437');
    T('0e+0', '0');
    T('1.21895563609362455802176301e+3', '1218.95563609362455802176301');
    T('3.7463929411575183690242615828695666e+19', '37463929411575183690.242615828695666');
    T('3.28403622335683751121762619033739817557846187135037e+37', '32840362233568375112176261903373981755.7846187135037');
    T('7.0543560548522121622953708263633228563e+22', '70543560548522121622953.708263633228563');
    T('1.739819e+3', '1739.819');
    T('2.15324060127001207725970506357881e+19', '21532406012700120772.5970506357881');
    T('2.373532121822929762731612214095513784533409e+29', '237353212182292976273161221409.5513784533409');
    T('4.5883026736677354807679611737881799804e+16', '45883026736677354.807679611737881799804');
    T('2.5996714820346689325468319633061e+21', '2599671482034668932546.8319633061');
    T('8.22641928e+6', '8226419.280');
    T('2.56006014528479284199702229871263269e+20', '256006014528479284199.702229871263269');
    T('4.301260132991159779386275268219519443685e+24', '4301260132991159779386275.268219519443685');
    T('1.052721790360165649330888881e+22', '10527217903601656493308.88881');
    T('6.85257703973809064426443728e+0', '6.85257703973809064426443728');
    T('1.341206836e+5', '134120.6836');
    T('1.293696083809745995580141432072678134217648017629e+25', '12936960838097459955801414.32072678134217648017629');
    T('9.81886611183e+9', '9818866111.83');
    T('1.3e+1', '13');
    T('2.185212134168411755342025405260683400574952243371e+1', '21.8521213416841175534202540526068340057495224337100');
    T('5.09812942277266e+1', '50.9812942277266');
    T('1.15841228150473459450904593187073359993e+37', '11584122815047345945090459318707335999.3');
    T('2.946408e+1', '29.46408');
    T('7.8843253757257e+6', '7884325.3757257');
    T('4.149829532631829e+7', '41498295.32631829');
    T('9.76844406944663415436782518894675931581135161845733e+46', '97684440694466341543678251889467593158113516184.5733');
    T('1.320634109357604978481e+7', '13206341.09357604978481');
    T('1.2300117044692162331376535732386421937e+8', '123001170.44692162331376535732386421937');
    T('1.79343822239530391558796001578394154846951511735e+42', '1793438222395303915587960015783941548469515.11735');
    T('3.46227335968923941657647562338569e+7', '34622733.5968923941657647562338569');
    T('3.6081901133629252234652167e+18', '3608190113362925223.4652167');
    T('3.41769614577210353834283168068494e+24', '3417696145772103538342831.68068494');
    T('1.036693895e+8', '103669389.5');
    T('9.840862048026534392868878603161623504069221701e+27', '9840862048026534392868878603.161623504069221701');
    T('2.56437211238605e+10', '25643721123.86050');
    T('2.645333616435501e+6', '2645333.616435501');
    T('3.75834254646606787747061360998e+1', '37.5834254646606787747061360998');
    T('1.21582101247e+6', '1215821.01247');
    T('5.43e+1', '54.3');
    T('3.1461380403028457753654142032015e+27', '3146138040302845775365414203.2015');
    T('2.73039e+4', '27303.9');
    T('3.349112077000398203735762417e+25', '33491120770003982037357624.170');
    T('2.293912475527946909960963698602754526495697363e+31', '22939124755279469099609636986027.54526495697363');
    T('7.800578368e+8', '780057836.8');
    T('3.503304265046835170500513083432983735273e+28', '35033042650468351705005130834.32983735273');
    T('6.521027589563589728e+9', '6521027589.563589728');
    T('1.26604818273232e+3', '1266.04818273232');
    T('4.5844253800756959854340115e+7', '45844253.800756959854340115');
    T('2.5103887704609158215979351198183e+20', '251038877046091582159.79351198183');
    T('6.5170765018089001398157674630438543e+17', '651707650180890013.98157674630438543');
    T('7.85679659655762637941070216699747e+18', '7856796596557626379.41070216699747');
    T('6.55113755834849587145e+18', '6551137558348495871.45');
    T('1.37856413555592382324487860882977704999616e+32', '137856413555592382324487860882977.704999616');
    T('7.51530486314140193e+5', '751530.486314140193');
    T('1.3712642461229590011e+7', '13712642.4612295900110');
    T('8.945222111405724e+2', '894.5222111405724');
    T('1.74501389497524149414213254563953197394499747444317e+44', '174501389497524149414213254563953197394499747.444317');
    T('7.1583294041845987824307132e+11', '715832940418.45987824307132');
    T('1.282007923703783860923747442697572540049e+13', '12820079237037.83860923747442697572540049');
    T('5.660625174793381639446229222e+11', '566062517479.3381639446229222');
    T('2.094745267e+4', '20947.45267');
    T('8.4497877437844686621097450218313191175e+13', '84497877437844.6866210974502183131911750');
    T('1.7e+1', '17');
    T('1.707217105197425488000493702652714920318758323999364e+23', '170721710519742548800049.3702652714920318758323999364');
    T('2.5487434814078948112667918801256335353406914111636153e+36', '2548743481407894811266791880125633535.3406914111636153');
    T('7.975944255792483246376368330364e+8', '797594425.5792483246376368330364');
    T('1.1038710051127692465453332862048e+20', '110387100511276924654.53332862048');
    T('2.0214122542287381656860062564183697682e+13', '20214122542287.381656860062564183697682');
    T('7.853012025112e+4', '78530.12025112');
    T('7.97061651146928e+3', '7970.61651146928');
    T('1.5712956919727392305361179349388e+10', '15712956919.727392305361179349388');
    T('8.2480059e+3', '8248.0059');
    T('2.71929146422591231877279340940781443961397491e+19', '27192914642259123187.7279340940781443961397491');
    T('2.131e+2', '213.10');
    T('5.53443299017925367e+6', '5534432.99017925367');
    T('9.0599636453e+8', '905996364.53');
    T('2.86841011915001378943e+0', '2.86841011915001378943');
    T('3.50691916034642635201767965866239485795145676493e+28', '35069191603464263520176796586.6239485795145676493');
    T('1.20268235416561427028396813141142129291e+18', '1202682354165614270.283968131411421292910');
    T('4.8878729e+4', '48878.7290');
    T('9.04344910891e+4', '90434.4910891');
    T('3.87232843764031e+2', '387.232843764031');
    T('2.246212201353343e+11', '224621220135.33430');
    T('4.92916234816086643529027167741689023e+4', '49291.6234816086643529027167741689023');
    T('2.1773818234639052045922630496e+22', '21773818234639052045922.630496');
    T('4.78705362683815125848e+0', '4.787053626838151258480');
    T('2.700762078698436846755198719005e+28', '27007620786984368467551987190.05');
    T('3.04897901998664513240359358574664525651334171e+36', '3048979019986645132403593585746645256.51334171');
    T('3.807300704307638993582e+18', '3807300704307638993.582');
    T('7.9846840795076707340124614425632613353204e+3', '7984.6840795076707340124614425632613353204');
    T('6.69918558928e+4', '66991.8558928');
    T('5.11e+1', '51.1');
    T('1.8e+1', '18');
    T('2.629676971e+2', '262.9676971');
    T('6.8402048967767212280354493672372347369800788279670097e+39', '6840204896776721228035449367237234736980.0788279670097');
    T('4.684145127602661593941009299096573092581e+21', '4684145127602661593941.009299096573092581');
    T('1.3010133600355313564757759338788842e+18', '1301013360035531356.4757759338788842');
    T('1.58527974113934732993372979240170059e+30', '1585279741139347329933729792401.70059');
    T('1.8249338722142728063286e+2', '182.49338722142728063286');
    T('1.5941e+0', '1.59410');
    T('2.0337546838170082473539926326926478252219475e+29', '203375468381700824735399263269.26478252219475');
    T('4.626173e+3', '4626.173');
    T('6.04857e+0', '6.04857');
    T('9.55039030589038069713466331e+0', '9.55039030589038069713466331');
    T('3.13977842e+6', '3139778.42');
    T('4.4046554138751e+10', '44046554138.751');
    T('2.27133088062335544441002965096e+25', '22713308806233554444100296.5096');
    T('1.72895143e+4', '17289.5143');
    T('2.59665963383099309886519729836757e+20', '259665963383099309886.5197298367570');
    T('5.42804375404301025317704270939778493719619031067122726e+24', '5428043754043010253177042.70939778493719619031067122726');
    T('4.0526054186532354119711729303068171063825508e+2', '405.26054186532354119711729303068171063825508');
    T('1.26960267394273418410687782475849e+9', '1269602673.94273418410687782475849');
    T('5.657206212494798707700546288522044895183104747814298e+7', '56572062.12494798707700546288522044895183104747814298');
    T('4.80834664047405196104320584899449259286e+21', '4808346640474051961043.20584899449259286');
    T('5.6639294733687553228095e+12', '5663929473368.7553228095');
    T('3.08469899797006019074738182718775120203832280411e+44', '308469899797006019074738182718775120203832280.4110');
    T('6.7324018330891115163882850963905830707247414517739e+20', '673240183308911151638.82850963905830707247414517739');
    T('7.8e+1', '78');
    T('3.7511576e+4', '37511.576');
    T('4.9744737445922007872559411117007e+20', '497447374459220078725.59411117007');
    T('1.4264855114407053894401398660016825255242638071603e+2', '142.64855114407053894401398660016825255242638071603');
    T('1.7972e+1', '17.972');
    T('1.08223075909551421423442320791403363148864e+12', '1082230759095.51421423442320791403363148864');
    T('1.27992032328728568e+16', '12799203232872856.8');
    T('3.23410998756876789482263723951851692122375679e+9', '3234109987.56876789482263723951851692122375679');
    T('8.309058187826413886051933894555524364e+5', '830905.8187826413886051933894555524364');
    T('5.9126904485324084868266487306831291316268437628598e+40', '59126904485324084868266487306831291316268.437628598');
    T('3.579918283412251816470339246643e+16', '35799182834122518.16470339246643');
    T('8.292403288e+1', '82.92403288');
    T('7.39431e+2', '739.431');
    T('3.880259e+2', '388.0259');
    T('8.356612898392420429137009991722851e+18', '8356612898392420429.137009991722851');
    T('9.45261313e+6', '9452613.13');
    T('3.3127850701711548546e+12', '3312785070171.1548546');
    T('7.883085682938156731330453228115e+0', '7.883085682938156731330453228115');
    T('7.8019561298380079335847245119645e+18', '7801956129838007933.5847245119645');
    T('1.162986351331516344668182075630641387e+5', '116298.63513315163446681820756306413870');
    T('1.28721291829552168e+9', '1287212918.29552168');
    T('6.1228832015e+5', '612288.32015');
    T('7.093619563532718270358988194610140484e+4', '70936.19563532718270358988194610140484');
    T('2.301304548870898223195731e+19', '23013045488708982231.95731');
    T('7.8447233503822258690539142204913891e+33', '7844723350382225869053914220491389.1');
    T('1.0090353662689425563590122399240645713114036952885e+26', '100903536626894255635901223.99240645713114036952885');
    T('2.6e+1', '26');
    T('2.015599e+5', '201559.9');
    T('3.1885351403871521971462599368214857856505297484e+33', '3188535140387152197146259936821485.78565052974840');
    T('3.6914534995202995808748962e+15', '3691453499520299.5808748962');
    T('1.2184696326401127654851812133097e+7', '12184696.326401127654851812133097');
    T('1.320231321916575882279815969200269836e+33', '1320231321916575882279815969200269.836');
    T('6.9618612891393774e+15', '6961861289139377.4');
    T('1.413827890310425450368036252350406468114295965313e+11', '141382789031.0425450368036252350406468114295965313');
    T('7.2883e+0', '7.2883');
    T('4.39128379250644938491985214e+17', '439128379250644938.491985214');
    T('4.08749008923147560205987e+12', '4087490089231.47560205987');
    T('1.2963755286285698528172931273502071438656e+36', '1296375528628569852817293127350207143.8656');
    T('1.748593351151123699414e+1', '17.48593351151123699414');
    T('9.734881140813067568986787719510123722831153669e+35', '973488114081306756898678771951012372.2831153669');
    T('2.32721108706388120897763e+11', '232721108706.388120897763');
    T('2.362273992041095475822632722018109548e+22', '23622739920410954758226.327220181095480');
    T('9.127306952822127691560865985866475e+18', '9127306952822127691.560865985866475');
    T('2.4e+1', '24');
    T('9.33725589231422920537799340863789387113892e+31', '93372558923142292053779934086378.9387113892');
    T('2.8683466985317806508353476e+6', '2868346.6985317806508353476');
    T('1.96167065035164685305509057685021959865228913867e+40', '19616706503516468530550905768502195986522.8913867');
    T('2.1513e+3', '2151.3');
    T('3.65316922559843117188159177065637844e+19', '36531692255984311718.8159177065637844');
    T('5.87971958699076454900383524e+21', '5879719586990764549003.835240');
    T('1.29414973913652368531805363331786105e+14', '129414973913652.368531805363331786105');
    T('1.68007422270052611738598e+22', '16800742227005261173859.8');
    T('2.9465419082779898714713969973e+4', '29465.4190827798987147139699730');
    T('2.865456717594117060964538744726484002e+28', '28654567175941170609645387447.26484002');
    T('1.2781496e+0', '1.2781496');
    T('5.395903069e+4', '53959.03069');
    T('2.084656696443025813e+12', '2084656696443.025813');
    T('7.54671420639507228408101673007042651462774881888e+46', '75467142063950722840810167300704265146277488188.8');
    T('1.436685520447262044600029e+14', '143668552044726.2044600029');
    T('6.9e+1', '69');
    T('8.921e+2', '892.10');
    T('2.51196408e+4', '25119.6408');
    T('4.2502325992166027236666111862782e+15', '4250232599216602.7236666111862782');
    T('1.48181688637265610577148846528720697801886e+17', '148181688637265610.577148846528720697801886');
    T('6.578849006577789439801702e+0', '6.578849006577789439801702');
    T('4.346905625146927213132990652e+22', '43469056251469272131329.90652');
    T('5.5037893e+0', '5.5037893');
    T('2.47731675267438120023934691987e+15', '2477316752674381.20023934691987');
    T('3.37487366652276710575333974697197457e+9', '3374873666.52276710575333974697197457');
    T('1.0385387059229329627650733271e+0', '1.0385387059229329627650733271');
    T('9.83004944893186643985967600066862369294062e+17', '983004944893186643.985967600066862369294062');
    T('4.24581658127100743607231740518633563667839856744e+26', '424581658127100743607231740.518633563667839856744');
    T('3.14222253457223322124561584676953981561133e+2', '314.222253457223322124561584676953981561133');
    T('1.1553833141891499623566322265502695096447390786748e+48', '1155383314189149962356632226550269509644739078674.8');
    T('5.2811e+1', '52.811');
    T('9.925202445922857021945001443270353221818473047344e+2', '992.5202445922857021945001443270353221818473047344');
    T('1.5758941e+4', '15758.9410');
    T('6.6630010328564980059488916358767e+30', '6663001032856498005948891635876.7');
    T('1.49898377473153728100588907982263779724221092732531e+44', '149898377473153728100588907982263779724221092.732531');
    T('4.175238908185616536855e+20', '417523890818561653685.5');
    T('1.192838736272799853174021036238e+21', '1192838736272799853174.021036238');
    T('1.145038e+3', '1145.038');
    T('4.0973786626728889384598402998014750474268e+9', '4097378662.6728889384598402998014750474268');
    T('5.5038104e+4', '55038.104');
    T('6.83895535917805849194871290958068199407518e+2', '683.895535917805849194871290958068199407518');
    T('2.9716066182e+0', '2.9716066182');
    T('1e+0', '1');
    T('1.78063428481384259205331358231117935e+0', '1.780634284813842592053313582311179350');
    T('6.277714976103425712837719e+22', '62777149761034257128377.1900');
    T('1.37376909692642287134486582232200547809845780076e+26', '137376909692642287134486582.232200547809845780076');
    T('7.0255659498942180908195e+16', '70255659498942180.908195');
    T('1.36758412477e+6', '1367584.12477');
    T('2.8993016541323392639291954727329719281958174e+23', '289930165413233926392919.54727329719281958174');
    T('2.44e+0', '2.44');
    T('5.39870374073212675286058196342904027304008232e+40', '53987037407321267528605819634290402730400.8232');
    T('6.4507160654825e+9', '6450716065.4825');
    T('1.21664e+3', '1216.64');
    T('3.13108416362400780440861428855476299376486503e+0', '3.13108416362400780440861428855476299376486503');
    T('7.2960499529336221198242592384915903149558006256202995e+17', '729604995293362211.98242592384915903149558006256202995');
    T('5.67239328846178836850536139916737284448973e+0', '5.67239328846178836850536139916737284448973');
    T('7.20020305957519743064e+3', '7200.203059575197430640');
    T('1.85115423780064073715032545790701546649748120114e+27', '1851154237800640737150325457.90701546649748120114');
    T('1.25021250836778893386687012660759710902e+21', '1250212508367788933866.87012660759710902');
    T('2.3323707491301665555664068537483355865980611e+25', '23323707491301665555664068.5374833558659806110');
    T('2.5088131581298507401113299236e+4', '25088.131581298507401113299236');
    T('9.612326850563943155774866e+17', '961232685056394315.5774866');
    T('1.54114517176248297154289225338049499367447824e+22', '15411451717624829715428.9225338049499367447824');
    T('5.22e+0', '5.22');
    T('4.04698305476309533783897e+21', '4046983054763095337838.97');
    T('2.620876536774240989563272117908814902188002596311e+24', '2620876536774240989563272.117908814902188002596311');
    T('1.7290754650750439926458970782158e+10', '17290754650.750439926458970782158');
    T('8.570789332248e+6', '8570789.332248');
    T('1.21e+1', '12.1');
    T('9.749134061639126502181192178140679940393318673720443e+45', '9749134061639126502181192178140679940393318673.720443');
    T('1.26878e+5', '126878.0');
    T('1.2391599841950849289559651456348e+9', '1239159984.1950849289559651456348');
    T('1.72220118427662724614289256133342842086e+22', '17222011842766272461428.9256133342842086');
    T('1.512063585971680294584184272035496e+15', '1512063585971680.294584184272035496');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;