//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/Data/DataUtil.h
//! @brief     Defines namespace DataUtil.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_DEVICE_DATA_DATAUTIL_H
#define BORNAGAIN_DEVICE_DATA_DATAUTIL_H

#include <memory>
#include <string>
#include <vector>

class Datafield;

namespace DataUtil::Data {

//! Changes the direction of rows or columns
std::vector<std::vector<double>> invertAxis(int axis,
                                            const std::vector<std::vector<double>>& original);

//! Transposes the matrix
std::vector<std::vector<double>> transpose(const std::vector<std::vector<double>>& original);

//! Creates a vector of vectors of double (2D Array) from Datafield.
std::vector<std::vector<double>> create2DArrayfromDatafield(const Datafield& data);

//! Reads 1D array of doubles to Python, for use in persistence test
Datafield* importArrayToDatafield(const std::vector<double>& vec);
//! Reads 2D array of doubles to Python, for use in persistence test
Datafield* importArrayToDatafield(const std::vector<std::vector<double>>& vec);

//! Returns vector of peak center coordinates, for peaks in given histogram.
std::vector<std::pair<double, double>> FindPeaks(const Datafield& data, double sigma = 2,
                                                 const std::string& option = {},
                                                 double threshold = 0.05);

#ifndef SWIG

//! Returns new object with input data rotated by
//! n*90 deg counterclockwise (n > 0) or clockwise (n < 0)
//! Axes are swapped if the data is effectively rotated by 90 or 270 degrees
//! Applicable to 2D arrays only
std::unique_ptr<Datafield> createRearrangedDataSet(const Datafield& data, int n);

//! Creates Datafield from a 2D Array.
std::unique_ptr<Datafield> vecvecToDatafield(const std::vector<std::vector<double>>& array_2d);

//! Creates Fourier Transform (Datafield format) of intensity map (Datafield format).
std::unique_ptr<Datafield> createFFT(const Datafield& data);

#endif // SWIG

} // namespace DataUtil::Data

#endif // BORNAGAIN_DEVICE_DATA_DATAUTIL_H
