//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Model/JobModel.h
//! @brief     Defines class JobModel
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_MODEL_JOBMODEL_H
#define BORNAGAIN_GUI_MODEL_MODEL_JOBMODEL_H

#include "Base/Types/OwningVector.h"
#include "GUI/Model/Job/JobItem.h"
#include "GUI/Model/Model/DataFilesCleaner.h"
#include "GUI/Model/Model/JobQueueData.h"

class InstrumentItem;
class SampleItem;
class RealItem;
class DataItem;
class SimulationOptionsItem;

class JobModel : public QObject {
    Q_OBJECT

public:
    explicit JobModel(QObject* parent = nullptr);

    JobItem* jobItemForIdentifier(const QString& identifier);

    JobItem* createJobItem();
    JobItem* addJobItem(const SampleItem* sampleItem, const InstrumentItem* instrumentItem,
                        const RealItem* realItem, const SimulationOptionsItem& optionItem);

    QVector<JobItem*> jobItems() const;

    void restore(JobItem* jobItem);

    bool hasUnfinishedJobs();

    void clear();
    QVector<DataItem*> dataItems() const;

    void writeTo(QXmlStreamWriter* w) const;
    void readFrom(QXmlStreamReader* r);

    void writeDataFiles(const QString& projectDir);
    void readDataFiles(const QString& projectDir, MessageService* messageService);

    void runJob(JobItem* jobItem);
    void cancelJob(JobItem* jobItem);
    void removeJob(JobItem* jobItem);

    int selectedIndex() const { return m_selectedIndex; }
    void setSelectedIndex(int index) { m_selectedIndex = index; }

signals:
    void focusRequest(JobItem* item);
    void globalProgress(int);
    void jobAdded();

private:
    //! generates numbered job name with new/unused number
    QString generateJobName();

    DataFilesCleaner dataFilesCleaner;
    std::unique_ptr<JobQueueData> m_queue_data;
    OwningVector<JobItem> m_jobItems;
    int m_selectedIndex = -1;
};

#endif // BORNAGAIN_GUI_MODEL_MODEL_JOBMODEL_H
