//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/PlotSpecular/SpecularDataPropertyWidget.cpp
//! @brief     Implements class SpecularDataPropertyWidget
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/PlotSpecular/SpecularDataPropertyWidget.h"
#include "GUI/Model/Axis/AmplitudeAxisItem.h"
#include "GUI/Model/Axis/BasicAxisItem.h"
#include "GUI/Model/Data/ComboProperty.h"
#include "GUI/Model/Data/SpecularDataItem.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/Numeric/DoubleSpinBox.h"
#include "GUI/View/Numeric/NumWidgetUtil.h"
#include "GUI/View/Tool/LayoutUtil.h"
#include <QAction>
#include <QCheckBox>
#include <QFormLayout>
#include <QLineEdit>

SpecularDataPropertyWidget::SpecularDataPropertyWidget(QWidget* parent)
    : DataPropertyWidget(parent)
{
}

void SpecularDataPropertyWidget::createPanelElements()
{
    if (allSpecularDataItems().size() == 0)
        return;

    for (auto item : allSpecularDataItems())
        disconnect(item, nullptr, this, nullptr);

    GUI::Util::Layout::clearLayout(m_mainLayout);
    m_updaters.clear();

    m_mainLayout->addRow("Axes units:",
                         GUI::Util::createUnitsComboBox(
                             axesUnitsList(),
                             [this] { return currentSpecularDataItem()->currentAxesUnits(); },
                             [this](const QString& newVal) {
                                 for (auto item : allSpecularDataItems())
                                     item->setCurrentAxesUnits(newVal);
                                 emit axesRangeResetRequested();
                                 gProjectDocument.value()->setModified();
                             },
                             &m_updaters));

    // -- x-axis
    auto* xGroup = new QGroupBox("X axis", this);
    auto* xFormLayout = new QFormLayout(xGroup);
    xFormLayout->setContentsMargins(0, 0, 0, 0);
    xFormLayout->setSpacing(5);

    xFormLayout->addRow("Min:",
                        GUI::Util::createDoubleSpinbox(
                            [this] { return currentSpecularDataItem()->xAxisItem()->min(); },
                            [this](double newValue) {
                                for (auto item : allSpecularDataItems())
                                    item->xAxisItem()->setMin(newValue);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    xFormLayout->addRow("Max:",
                        GUI::Util::createDoubleSpinbox(
                            [this] { return currentSpecularDataItem()->xAxisItem()->max(); },
                            [this](double newValue) {
                                for (auto item : allSpecularDataItems())
                                    item->xAxisItem()->setMax(newValue);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    xFormLayout->addRow("Title:",
                        GUI::Util::createTextEdit(
                            [this] { return currentSpecularDataItem()->xAxisItem()->title(); },
                            [this](QString newText) {
                                for (auto item : allSpecularDataItems())
                                    item->xAxisItem()->setTitle(newText);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    m_mainLayout->addRow(xGroup);

    // -- y-axis
    auto* yGroup = new QGroupBox("Y axis", this);
    auto* yFormLayout = new QFormLayout(yGroup);
    yFormLayout->setContentsMargins(0, 0, 0, 0);
    yFormLayout->setSpacing(5);

    yFormLayout->addRow("Min:",
                        GUI::Util::createDoubleSpinbox(
                            [this] { return currentSpecularDataItem()->yAxisItem()->min(); },
                            [this](double newValue) {
                                for (auto item : mainSpecularDataItems())
                                    item->yAxisItem()->setMin(newValue);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    yFormLayout->addRow("Max:",
                        GUI::Util::createDoubleSpinbox(
                            [this] { return currentSpecularDataItem()->yAxisItem()->max(); },
                            [this](double newValue) {
                                for (auto item : mainSpecularDataItems())
                                    item->yAxisItem()->setMax(newValue);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    yFormLayout->addRow("Title:",
                        GUI::Util::createTextEdit(
                            [this] { return currentSpecularDataItem()->yAxisItem()->title(); },
                            [this](QString newText) {
                                for (auto item : mainSpecularDataItems())
                                    item->yAxisItem()->setTitle(newText);
                                gProjectDocument.value()->setModified();
                            },
                            &m_updaters));

    yFormLayout->addRow(GUI::Util::createCheckBox(
        "log10", [this] { return currentSpecularDataItem()->yAxisItem()->isLogScale(); },
        [this](bool b) {
            for (auto item : allSpecularDataItems())
                item->yAxisItem()->setLogScale(b);
            gProjectDocument.value()->setModified();
        },
        &m_updaters));

    m_mainLayout->addRow(yGroup);

    updateUIValues();

    // react on external changes (e.g. zooming in customplot shall update the axis values)
    connect(currentSpecularDataItem(), &DataItem::itemAxesRangeChanged, this,
            &SpecularDataPropertyWidget::updateUIValues, Qt::UniqueConnection);

    // update coordinates on axes units change
    for (auto item : allSpecularDataItems())
        connect(item, &DataItem::axesUnitsChanged, this,
                &SpecularDataPropertyWidget::updateItemCoords, Qt::UniqueConnection);
}
