(* :Name: CorrelationPlot *)

(* :Title: CorrelationPlot *)

(* :Author: Tom Wickham-Jones*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(*:Summary:
	This package produces correlation plots.
*)

(* :History:
	New version winter 1994 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Warnings:
*)

If[ Context[ CorrelationPlot] =!= "ExtendGraphics`CorrelationPlot`",
		Remove[ CorrelationPlot]]

BeginPackage[ "ExtendGraphics`CorrelationPlot`",
				"ExtendGraphics`Ticks`",
				"Utilities`FilterOptions`"]

CorrelationPlot::usage =
	"CorrelationPlot[ data], where data is a vector of n-dimensional
	 points, produces an array of n by n plots showing the i'th component of
	 data plotted against the j'th.   The plots can be labelled with
	 the Labels options.  Axes will be drawn if Axes -> True is set."

Labels::usage =
	"Labels is an option of CorrelationPlot that specifies the labels
	 to be plotted.  The setting of the Labels option must be a list
	 of length equal to the dimensionality of the data."
	 
TickFont::usage =
	"TickFont is an option of CorrelationPlot that specifies the font
	 to be used for the ticks.  The font must be of the form
	 {\"name\", size}."
	 
Begin["`Private`"]

If[ Not[ $VersionNumber > 2.2],
		Offset[ off_, pos_] := Scaled[ off/100, pos]]

If[ Not[ $VersionNumber > 2.2],
	Unprotect[ Text] ;
	(Text[ x__, opts__?OptionQ] := 
		Block[{font},
			font = DefaultFont /. {opts} ;
			If[ font === DefaultFont,
				Text[x], 
				Text[ FontForm[ First[{x}], font], Sequence @@ Rest[{x}]]]
			] );
	Protect[ Text]]



Options[ CorrelationPlot] =
	Join[
		{
		Labels -> {},
		TickFont -> {"Courier", 7}
		}, Options[ GraphicsArray]]
		

Clear[ CorrelationPlot]

GetFont[ {name_String, size_}, def_] :=
	{ name, CheckSize[ size]}

GetFont[ name_String, def_] :=
	If[ Length[ def] === 2,
			{name, CheckSize[ Last[ def]]},
			{name, 6}]

GetFont[ size_, def_] :=
	If[ Length[ def] === 2,
			{First[ def], CheckSize[size]},
			{"Courier", CheckSize[ size]}]
	
CheckSize[ x_] := If[ Positive[ x], x, 6, 6]


CorrelationPlot[ data_ /; MatrixQ[ data, NumberQ], opts___] :=
  Block[{ i, j, res, len = Length[ First[ data]],txt,ext,axes,
          axesx, axesy, txtx, txty, font, rng, rngx, rngy},
		   
  	txt = Labels /. {opts} /. Options[ CorrelationPlot] ;
  	axes = Axes /. {opts} /. Options[ CorrelationPlot] ;
  	font = TickFont /. {opts} /. Options[ CorrelationPlot] ;
	font = GetFont[ font, $DefaultFont] ;

	If[ Length[ txt] === Length[ First[ data]],
			ext = True
	,
			txt = Table[ {}, {Length[ First[ data]]}] ;
			ext = False] ;

    res = Table[ Map[ {Part[ #, i], Part[#, j]}&, data],
        {j, len}, {i, len}] ;
    res = Reverse[ res] ;
    res = Map[ 
			ListPlot[ #, 
				Frame -> True,
				Axes -> False, 
        		FrameTicks -> None,
        		AspectRatio -> 1, 
        		DisplayFunction -> Identity]&,
        	res, {2}] ;

	If[ ext || axes,
			rngx = Map[ First[FullOptions[#, PlotRange]]&, 
						First[ res]] ;
			rngy = Map[ Part[FullOptions[#, PlotRange],2]&, 
						First[ Transpose[res]]] ;
			If[ ext,
					txtx = TextCalcX[ txt, rngx] ;
					txty = TextCalcY[ txt, rngy] ;
					oppx = { -0.1,1.0};
					oppy = { -1.0, 0.1};
			,
					txtx = txt ;
					txty = txt ;
					oppx = {0, 1.1};
					oppy = {0, 1.1};
				] ;	
			
			If[ axes && ext,
					oppx = {-1.1, 1.4};
					oppy = {-1, 1.1};
				] ;

			If[ axes,
					axesx = Map[ AxesCalcX[ #, font]&, rngx] ;
					axesy = Map[ AxesCalcY[ #, font]&, rngy] ;
					txtx = Transpose[ {txtx, axesx}] ; 
					txty = Transpose[ {txty, axesy}] ;
				] ;
			
			txtx = Transpose[ {txtx, rngx}] ;
			txty = Transpose[ {txty, rngy}] ;
			
			txtx = Map[ Graphics[#[[1]], 
						AspectRatio -> 1,
						PlotRange -> {#[[2]], oppx}]&, txtx] ;
			txty = Map[ Graphics[#[[1]], 
						AspectRatio -> 1,
						PlotRange -> {oppy, #[[2]]}]&, txty] ;
	
			res = Append[ res, txtx] ;
			txty = Reverse[ txty] ;
			txty = Append[ txty, Graphics[{},AspectRatio ->1]] ;
			res = Apply[ Prepend, Transpose[ {res, txty}], {1}]
		] ;

    Show[ GraphicsArray[ res], 
			Axes -> False,
			FilterOptions[GraphicsArray, opts]]
  ]

TextCalcX[ txt_, pos_] := 
	Map[ 
	  Text[#[[1]], {Apply[ Plus, #[[2]]] / 2., 0},{0,-1}]&, 
	  Transpose[ {txt, pos}]]

TextCalcY[ txt_, pos_] :=
	Map[ 
	  Text[#[[1]], {0,Apply[ Plus, #[[2]]] / 2.},{1,0}]&, 
	  Transpose[ {txt, pos}]]


AxesCalcX[ {min_, max_}, font_] :=
  Block[{ticks},
	ticks = TickPosition[ min, max, 8] ;
	ticks = Map[ 
				{
				Text[#, Offset[ {0, -2}, {#,1}], {0,1}, DefaultFont -> font],
				Line[ { {#,1}, Offset[ {0, 1}, {#,1}]}]
				}&, ticks] ;
	{
	Thickness[0.0001],
	Line[ {{min, 1}, {max, 1}}],
	ticks
	}
	]

AxesCalcY[ {min_, max_}, font_] :=
  Block[{ticks},
	ticks = TickPosition[ min, max, 8] ;
	ticks = Map[ 
				{
				Text[#, Offset[ {-2,0}, {1, #}], {1,0}, DefaultFont -> font],
				Line[ { {1,#}, Offset[ {1,0}, {1,#}]}]
				}&, ticks] ;
	{
	Thickness[0.0001],
	Line[ {{1, min}, {1, max}}],
	ticks
	}
	]


End[]

EndPackage[]


(*:Examples:

<<ExtendGraphics`CorrelationPlot`


data = Table[ {x, Random[], - 2 x^2, x^3}, {x,-1,2,.05}];

CorrelationPlot[ data]


CorrelationPlot[ data, 
		Labels -> {x,Random,-x^2,x^3}]

CorrelationPlot[ data,
	Axes -> True]

CorrelationPlot[ data,
	Axes -> True,
	TickFont -> 4]

CorrelationPlot[ data,
	Axes -> True, 
	Labels -> {x,Random,-x^2,x^3}]

*)



