/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_graftpoints.c
   \brief Implements access to graftpoints file

   File implements function that are wrappers around graftpoints file
   descriptor: creating (with random file name), writing and closing.

   Graftpoints file is used to tell mkisofs which files to use to
   create ISO image, and to create desired directory structure in ISO file.
*/

#define _BSD_SOURCE /* fsync() */

#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h> /* current time helps creating seed for srand() */
#include <errno.h>

/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "cdw_graftpoints.h"
#include "cdw_string.h"
#include "cdw_file.h"
#include "cdw_fs.h"
#include "cdw_debug.h"
#include "cdw_file_manager.h"



static cdw_rv_t cdw_graftpoints_open_file(void);
static cdw_rv_t cdw_graftpoints_close_file(void);
static cdw_rv_t cdw_graftpoints_write_to_file(const char *left, const char *right);
static void set_string_random(char *buffer);

/* value assigned to graftpoints_file when file is closed (it either wasn't
   open yet or was closed by cdw graftpoints functions) */
#define CDW_FILE_CLOSED -2

/** \brief File descriptor of current graftpoints file */
static int graftpoints_file = CDW_FILE_CLOSED;

/** \brief Full path to current file name of graftpoints file */
static char *graftpoints_fullpath = (char *) NULL;

/** \brief Length of part of graftpoints file name that is randomly generated */
#define RANDOM_STRING_LEN 10 /* length does not include trailing '\0' */


/**
   \brief Put content of list of files to temporary file

   Put content of list of files (that are going to be written to iso image)
   to temporary file that will be used by program creating iso image.

   Graft points (see 'man mkisofs') are created here: this function simply
   takes name of last item in full path and puts the name on left side of
   '=' sign. On the right side of the sign a fullpath is printed.

   Use this function only when needed, that is: only when you want to do
   sth with selected files (before writing them to CD or to image).
   Make sure that there are any selected files before calling the function.

   \return CDW_ERROR when some error occurs
   \return CDW_OK on success
*/
cdw_rv_t cdw_graftpoints_create_file(void)
{
	cdw_dll_item_t *f = cdw_file_manager_get_list_of_selected_files(); /* head */
	cdw_assert (f != (cdw_dll_item_t *) NULL, "ERROR: called the function when list of selected files is empty\n")

	cdw_rv_t crv = cdw_graftpoints_open_file();
	if (crv != CDW_OK) {
		/* no need to call cdw_graftpoints_delete_file(),
		   open_file() does not leave mess in case of errors */
		cdw_vdm ("ERROR: failed to open graftpoints file\n");
		return CDW_ERROR;
	}

	size_t i = 0;
	for (; f; f = f->next) {
		cdw_file_t *file = (cdw_file_t *) f->data;
		cdw_assert (file->fullpath != (char *) NULL, "ERROR: file has no fullpath\n");
		cdw_assert (strlen(file->fullpath) != 0, "ERROR: file full path length == 0\n");
		cdw_assert (file->name_start > 0, "ERROR: file has no name\n");
		cdw_assert (strlen(file->fullpath + file->name_start) != 0, "ERROR: file name length == 0\n");

		if (!strcmp(file->fullpath, "/")) {
			crv = cdw_graftpoints_write_to_file("", file->fullpath);
		} else {
			crv = cdw_graftpoints_write_to_file(file->fullpath + file->name_start, file->fullpath);
		}

		i++;

		if (crv == CDW_ERROR) {
			cdw_vdm ("ERROR: failed to properly write line to graftpoints file\n");
			cdw_graftpoints_delete_file();
			return CDW_ERROR;
		}

	}

	cdw_assert (i > 0, "ERROR: there were 0 files on list of selected files\n");

	/* calls fsync, so all written lines are in file for sure */
	crv = cdw_graftpoints_close_file();
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to properly close graftpoints file\n");
		cdw_graftpoints_delete_file();
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Delete graftpoints file correctly created with last call to cdw_graftpoints_create_file()

   Delete graftpoints file that was used for last writing session.
   File should be closed with cdw_graftpoints_close_file() before calling
   this function.

   The function can also be called when you need to clean up resources in
   case of some errors with graftpoints file.

   The function does not return any error values, even if there are some errors.
*/
void cdw_graftpoints_delete_file(void)
{
	if (graftpoints_file != CDW_FILE_CLOSED && graftpoints_file != -1) {
		cdw_vdm ("ERROR: called the function before closing file\n");

		close(graftpoints_file);
		graftpoints_file = CDW_FILE_CLOSED;
	}

	if (graftpoints_fullpath != (char *) NULL) {
		int a = access(graftpoints_fullpath, F_OK);
		if (a != 0) {
			cdw_vdm ("ERROR: access() failed for graftpoints fullpath \"%s\": %s\n", graftpoints_fullpath, strerror(errno));
		} else {
			a = unlink(graftpoints_fullpath);
			if (a != 0) {
				cdw_vdm ("ERROR: unlink() failed for graftpoints fullpath \"%s\": %s\n", graftpoints_fullpath, strerror(errno));
			}
		}
		free(graftpoints_fullpath);
		graftpoints_fullpath = (char *) NULL;
	} else {
		cdw_vdm ("ERROR: graftpoints fullpath is NULL\n");
	}

	return;
}





/**
   \brief A getter for variable with full path to graftpoints file

   \returns full path to graftpoints file (the pointer may be (char *) NULL)
*/
const char *cdw_graftpoints_get_fullpath(void)
{
	cdw_assert (graftpoints_fullpath != (char *) NULL, "graftpoints fullpath is NULL\n");
	int r = access(graftpoints_fullpath, F_OK);
	cdw_assert (r == 0, "graftpoints file does not exist, you don't want a path at this moment\n");
	cdw_assert (graftpoints_file == CDW_FILE_CLOSED, "file should be closed at this point\n");

	return graftpoints_fullpath;
}




/* private functions */




/**
   \brief Create new graftpoints file

   Function opens in tmp directory new file with random name.
   The file can be used as graftpoint file for mkisofs.

   Variable char *graftpoints_fullpath (global in file) is set to
   full path to graftpoint file. It can also be set to (char *) NULL if
   file cannot be open.

   File descriptor of open graftpoints file should be unique - there cannot
   be two open graftpoints at the same time. Therefore this function
   returns CDW_ERROR if some graftpoints file is already open. If such
   situation occurs, then this is a sign of programming error.

   In case of errors graftpoints_fullpath is set to (char *) NULL, and
   graftpoints_file is set to CDW_FILE_CLOSED.

   \return CDW_ERROR on errors
   \return CDW_OK on success
*/
cdw_rv_t cdw_graftpoints_open_file(void)
{
	cdw_assert (graftpoints_file == CDW_FILE_CLOSED, "called the function to open second file\n");
	cdw_assert (graftpoints_fullpath == (char *) NULL,
		    "you forgot to clean up when closing previous file\n");

	const char *tmp_dir = cdw_fs_get_tmp_dir_fullpath();
	if (tmp_dir == (char *) NULL) {
		cdw_vdm ("ERROR: failed to get path to tmp dir\n");
		return CDW_ERROR;
	}

	/* random part of file name, 10 digits + '\0' */
	/* address of the table will be also used as pseudo random string;
	   I know that this is a poor choice, but it will be used only
	   when time() fails, so not too often */
	char random_string[RANDOM_STRING_LEN + 1];

	set_string_random(random_string);
	/* TODO: file name is (more or less) random, now make it safe (think
	   race conditions etc.) */

	graftpoints_fullpath = cdw_string_concat(tmp_dir, "cdw_tmp_graft_", random_string, (char * ) NULL);
	if (graftpoints_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create path to graftpoint file\n");
		return CDW_ERROR;
	}

	graftpoints_file = open(graftpoints_fullpath, O_CREAT | O_TRUNC | O_RDWR, S_IRUSR | S_IWUSR);
	if (graftpoints_file == -1) {
		cdw_vdm ("ERROR: failed to open graftpoints file\n");

		free(graftpoints_fullpath);
		graftpoints_fullpath = (char *) NULL;
		graftpoints_file = CDW_FILE_CLOSED;

		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Close current graftpoints file

   Close file descriptor of unique graftpoints file.
   File name buffer is not affected. On successful closing of file flag
   is set to indicate that file is closed.

   \return CDW_OK on success
   \return CDW_ERROR if fsync() or close() fail
*/
cdw_rv_t cdw_graftpoints_close_file(void)
{
	int rv1 = fsync(graftpoints_file);
	if (rv1 != 0) {
		cdw_vdm ("ERROR: fsync() failed, errno: \"%s\"\n", strerror(errno));
		cdw_assert (rv1 == 0, "ERROR: fsync() returned %d\n", rv1);
	}

	int rv2 = close(graftpoints_file);
	if (rv2 != 0) {
		cdw_vdm ("ERROR: close() failed, errno: \"%s\"\n", strerror(errno));
		cdw_assert (rv2 == 0, "ERROR: close() returned %d\n", rv2);
	} else {
		graftpoints_file = CDW_FILE_CLOSED;
	}

	/* don't free file name buffer yet, we will use it when
	   unlink()ing old file */

	if (rv1 != 0 || rv2 != 0) {
		return CDW_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Write given strings to graftpoints file

   The function writes given strings to graftpoints file adding '\n'
   char at the end of line. The strings are written to file like this:
   left=right
   so if \p left is a name in target ISO filesystem, and \p right
   is path to file existing in local filesystem, then mkisofs
   will recognize such line as a valid specification, a graftpoint.
   See mkisofs man page for details.

   Function checks if graftpoints file is open. It returns error if file
   is not open. Use cdw_graftpoints_file_open() first to open the file.

   \param left - string representing point in created ISO file system
   \param right - full path to file existing in local file system

   \return CDW_ERROR if writing to file fails
   \return CDW_OK if success
*/
cdw_rv_t cdw_graftpoints_write_to_file(const char *left, const char *right)
{
	cdw_assert (graftpoints_file != CDW_FILE_CLOSED, "trying to write before opening file\n");
	cdw_assert (left != (char *) NULL && right != (char *) NULL,
		    "ERROR: function called with null argument(s)\n");
	cdw_assert (left[0] != '/', "ERROR: left side of graftpoint starts with '/'\n");

#ifndef NDEBUG /* for testing purposes only */
	size_t llen = strlen(left);
	if (left[llen] == '/') {
		size_t rlen = strlen(right);
		cdw_assert (right[rlen] == '/', "ERROR: left side of graftpoint ends with slash, but right one does not\n");
	} else {
		size_t rlen = strlen(right);
		cdw_assert (right[rlen] != '/', "ERROR: left side of graftpoint doesn't end with slash, but right one does\n");
	}
#endif

	cdw_vdm ("INFO: writing graft point \"%s=%s\"\n", left, right);
	ssize_t rv1 = write(graftpoints_file, left, strlen(left));
	ssize_t rv2 = write(graftpoints_file, "=", strlen("="));
	ssize_t rv3 = write(graftpoints_file, right, strlen(right));
	ssize_t rv4 = write(graftpoints_file, "\n", strlen("\n"));

	if (rv1 == -1 || rv2 == -1 || rv3 == -1 || rv4 == -1) {
		return CDW_ERROR;
	} else {
		fsync(graftpoints_file);
		return CDW_OK;
	}
}





/**
   \brief Set content of current buffer to some random value

   This function sets content of buffer of known size
   (RANDOM_STRING_LEN + 1 for ending '\0') with random readable chars.
   The function is used to prepare random file name.

   It is a string utility, but very specialized, so it won't be moved to
   utils_string.c

   \param buffer - buffer that you want to initialize with random chars
*/
void set_string_random(char *buffer)
{
	time_t t = time((time_t *) NULL);
	static unsigned int seed = 31601; /* some prime number */
	if (t == (time_t) -1) {
		/* pseudo randomness */
		seed = (unsigned int) ((((size_t) buffer) * seed) >> 16);
	} else {
		seed = (unsigned int) t;
	}
	srand(seed);

	int rand1 = rand();
	int rand2 = rand();

	/* 32767 is minimal value of RAND_MAX required by C standard
	   so we are sure to get 5 random digits */
	snprintf(buffer, RANDOM_STRING_LEN + 1, "%05d%05d", rand1 % 32767, rand2 % 32767);

	return;
}



