/*
 * dsyslog - a dumb syslog (e.g. syslog for people who have a clue)
 * Copyright (c) 2008 William Pitcock <nenolod@sacredspiral.co.uk>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice is present in all copies.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "dsyslog.h"

static void
cond_regexp_destructor(dsyslog_cond_t *cond)
{
	_ENTER;

	g_regex_unref(cond->opaque);

	_LEAVE;
}

static gboolean
cond_regexp_handler(dsyslog_event_t *event, dsyslog_cond_t *cond)
{
	char *value;
	gboolean invert = FALSE;

	_ENTER;

	value = cond->value;
	if (*value == '!') {
		value++;
		invert = TRUE;
	}

	if (!cond->opaque) {
		GError *error = NULL;
		cond->opaque = g_regex_new(value, G_REGEX_CASELESS, 0, &error);
		if (error) {
			cond->opaque = NULL;
			_ERROR("cond->opaque failed to compile: %s", error->message);
			g_clear_error(&error);

			_LEAVE TRUE;	/* TODO: should we bail gracefully or abort() in this condition? */
		}
		cond->destructor = cond_regexp_destructor;
	}

	if (!g_ascii_strcasecmp(cond->cond, "program")) {
		if (g_regex_match(cond->opaque, event->program, 0, NULL)) {
			_LEAVE invert ? FALSE : TRUE;
		}
	}

	if (!g_ascii_strcasecmp(cond->cond, "source")) {
		if (g_regex_match(cond->opaque, event->source, 0, NULL)) {
			_LEAVE invert ? FALSE : TRUE;
		}
	}

	_LEAVE invert ? TRUE : FALSE;
}

void
_modinit(void)
{
	_ENTER;

	dsyslog_cond_type_register("regexp", cond_regexp_handler);

	_LEAVE;
}

void
_modfini(void)
{
	_ENTER;

	dsyslog_cond_type_unregister("regexp");

	_LEAVE;
}
