/* $Id: graphics.h,v 1.33 2003/06/15 13:01:06 sjoerd Exp $ */
#ifndef FFRENZY_GRAPHICS_H
#define FFRENZY_GRAPHICS_H

#include <stdio.h>
#include <SDL.h>

#include "model.h"
#include "svg.h"
#include "sprite.h"
#include "vector.h"
#include "player.h"

/** @file
 * Functions and data struct of the graphics module
 */

/** \defgroup graphics
 * Graphics abstraction layer
 */

enum {
  GRAPHICS_LAYER_FIRST = 0,
  GRAPHICS_LAYER_BACK = 0,
  GRAPHICS_LAYER_MIDDLE,
  GRAPHICS_LAYER_FRONT,
  GRAPHICS_NRLAYERS
};
/** \ingroup graphics
 * Graphics type
 * \a screen_width and \a screen_height are the dimensions of the viewport of the player
 */
typedef struct {
	Vector *ratio; /**< Ratio between world coordinates and screen coordinates */
  Vector *gsize; /** Size of the game coords */
  Vector *goffset; /** Location of the game coords */
	SDL_Surface *screen; /**< The main surface */
  SDL_Surface *background ; /**< background of the screen */
  int background_changed; /**< True if the background was changed */
  List *sprites[OT_NRTYPES]; /**< List of graphics cache for every type */
  List *layers[GRAPHICS_NRLAYERS];
	Graphics_sprite_source *svg[OT_NRTYPES]; /**< Array of sprite sources 
                                               one for every type */
  SDL_Rect *update_rects_queue; /**< queue of rects that need updating */
  int update_number;             /**< number of rects that need updating */
  int update_queue_len;         /**< lenght of the rects queue */ 

  List *players;                /* player list */

  Uint32 scorecolor; /**< background of the scoreboard */
  int maxlife;
  TTF_Font *font;
} Graphics;

typedef struct {
  int type;
  Vector *size;
  Graphics_sprite_anim *animations[SS_NRSTATES];
  Graphics_sprite_source *svg;
} Graphics_cache;

#define PINFO_RELOCATED       0x1
#define PINFO_SCORE_CHANGED   0x2
#define PINFO_LIFE_CHANGED    0x4
#define PINFO_STACK_CHANGED   0x8
#define PINFO_IMAGE_CHANGED   0xf

typedef struct {
	char *name; /**< Printable name of the player */
	int32_t score; /**< Current score */
	int32_t life; /**< Life status */
	int32_t maxlife; /**< Maximum life status */
	int is_local; /**< Player is a local player */
  int stack_top; /* Whats on top of the stack */
  int stacklength; /* Number of items in the food stack */
  int is_upgraded;
  Vector *place; /**< locations of this players score on the board */
  Graphics_cache *foodcaches[OT_NRFOOD] ; /**< caches for the food displays */
  int flags; /**< flags for things that happened */
  SDL_Surface *player_graphic;
  SDL_Surface *name_graphic;
  int height;
} Graphics_player_info;

#define Graphics_POSCHANGED 0x1
#define Graphics_SPRITECHANGED 0x2

typedef struct {
  SDL_Rect cur;
  SDL_Rect prev;
  SDL_Rect blitrect;
  SDL_Surface *sprite;
  int flags;
  int layer;
} Graphics_drawinfo;

/** \ingroup graphics
 * Graphics object type 
 * Stores for every object (whether it is an object type or unique object) the
 * drawing info  plus the current frames of an animation */
typedef struct {
	int frame; /**< Current animations frame */
  Sprite_state state;
  Vector *size;
  Graphics_drawinfo *dinfo; /**< information for drawing */
  Graphics_cache *scache;
	Graphics_player_info *pinfo; /** < information about player, undefined for !players */
  int private_graphic; /**< indicates if this object's graphics is private 
                            or not */
} Graphics_object;

/** \ingroup graphics
 * Initialize a graphics object width screen size \a width x \a height
 * \a wwidth is the world width in the model, \a wheight the corresponding world height
 **/
Graphics *new_graphics(int width, int height, int wwidth, int wheight,
                       int fullscreen);

/** \ingroup graphics
 * Destroys the viewport, unregisters the SDL, clears all graphics objects
 */
void del_graphics(Graphics * g, Model * m);

/** \ingroup graphics
 * Update the graphics \a g with new data from the model \a m
 */
int graphics_update(Graphics * g, Model * m);

/** \ingroup grpahics
 * Displayer the scores on the displayer
 */
int graphics_showscores(Graphics *g);

/** \ingroup graphics
 * Link a graphic to a specific object.
 * Indicates that the \a object must be drawn using the \a SVGserie
 */
int graphics_add_object(Graphics * g, Model_object *object, char *SVGserie);

/** \ingroup graphics
 * Register a graphic for a type of objects.
 * The \a SVGserie will be used for all objects of \a type. (unless they have
 * their own unique graphic)
 */
int graphics_register_type(Graphics * g, Object_type type,char *SVGserie);

/** \ingroup graphics
 * Adds a local player \a player_id
 * This tells the graphics layer the \a name of the object (which is a player)
 * and indicates if it is a local player (\a is_local)
 */
int graphics_add_player(Graphics * g, Model_object *object, 
                        char *name,int is_local);

int
graphics_set_background(Graphics *g, char *SVGserie);
/** \ingroup graphics
 * Converts world coordinates into screen coordinates
 * \a g is the Graphics object, \a v the vector to be converted
 */
//Vector *graphics_scale(Graphics *g, Vector *v);

#endif /* FFRENZY_GRAPHICS_H */
