/*
 *  $Id: raveraging.c 22285 2019-07-19 08:51:51Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/correct.h>
#include <libprocess/filters.h>
#include <libprocess/arithmetic.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define RAVERAGING_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

typedef struct {
    gdouble xc;
    gdouble yc;
    gdouble phiscale;
    gdouble phioff;
} TransformData;

typedef struct {
    gdouble radius;
    gdouble angle;
    GwyInterpolationType interp;
} RAveragingArgs;

typedef struct {
    RAveragingArgs *args;
    GtkObject *radius;
    GtkObject *angle;
    GtkWidget *interp;
} RAveragingControls;

static gboolean module_register         (void);
static void     raveraging              (GwyContainer *data,
                                         GwyRunType run);
static gboolean raveraging_dialog       (RAveragingArgs *args);
static void     radius_changed          (GtkAdjustment *adj,
                                         RAveragingControls *controls);
static void     angle_changed           (GtkAdjustment *adj,
                                         RAveragingControls *controls);
static void     interp_changed          (GtkComboBox *combo,
                                         RAveragingControls *controls);
static void     raveraging_load_args    (GwyContainer *container,
                                         RAveragingArgs *args);
static void     raveraging_save_args    (GwyContainer *container,
                                         RAveragingArgs *args);
static void     raveraging_dialog_update(RAveragingControls *controls,
                                         RAveragingArgs *args);
static void     filter_radial           (GwyDataField *dfield,
                                         gdouble radius,
                                         gdouble angle,
                                         GwyInterpolationType interp);

static const RAveragingArgs raveraging_defaults = {
    10.0,
    10.0,
    GWY_INTERPOLATION_BILINEAR,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Smooths images in polar coordinates."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2019",
};

GWY_MODULE_QUERY2(module_info, raveraging)

static gboolean
module_register(void)
{
    gwy_process_func_register("raveraging",
                              (GwyProcessFunc)&raveraging,
                              N_("/_Distortion/_Radial Smoothing..."),
                              NULL,
                              RAVERAGING_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Smooth image in polar coordinates"));

    return TRUE;
}

static void
raveraging(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield;
    RAveragingArgs args;
    gboolean ok;
    gint oldid, newid;

    g_return_if_fail(run & RAVERAGING_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &oldid,
                                     0);
    g_return_if_fail(dfield);

    raveraging_load_args(gwy_app_settings_get(), &args);
    if (run == GWY_RUN_INTERACTIVE) {
        ok = raveraging_dialog(&args);
        raveraging_save_args(gwy_app_settings_get(), &args);
        if (!ok)
            return;
    }

    dfield = gwy_data_field_duplicate(dfield);

    filter_radial(dfield, args.radius, args.angle, args.interp);

    newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);
    gwy_app_sync_data_items(data, data, oldid, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);

    g_object_unref(dfield);
    gwy_app_set_data_field_title(data, newid, _("Radially smoothed"));
    gwy_app_channel_log_add_proc(data, oldid, newid);
}

static gboolean
raveraging_dialog(RAveragingArgs *args)
{
    GtkWidget *dialog, *table;
    RAveragingControls controls;
    gint response, row;

    gwy_clear(&controls, 1);
    controls.args = args;

    dialog = gtk_dialog_new_with_buttons(_("Radial Smoothing"), NULL, 0,
                                         _("_Reset"), RESPONSE_RESET,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);

    table = gtk_table_new(3, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table,
                       FALSE, FALSE, 4);
    row = 0;

    controls.radius = gtk_adjustment_new(args->radius, 0.0, 1000.0, 0.1, 10, 0);
    gwy_table_attach_adjbar(table, row++, _("_Radius:"), _("px"),
                            controls.radius, GWY_HSCALE_DEFAULT);
    g_signal_connect(controls.radius, "value-changed",
                     G_CALLBACK(radius_changed), &controls);

    controls.angle = gtk_adjustment_new(args->angle, 0.0, 180.0, 0.01, 10, 0);
    gwy_table_attach_adjbar(table, row++, _("_Angle:"), _("deg"),
                            controls.angle, GWY_HSCALE_DEFAULT);
    g_signal_connect(controls.angle, "value-changed",
                     G_CALLBACK(angle_changed), &controls);

    controls.interp
        = gwy_enum_combo_box_new(gwy_interpolation_type_get_enum(), -1,
                                 G_CALLBACK(interp_changed), &controls,
                                 args->interp, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Interpolation type:"), NULL,
                            GTK_OBJECT(controls.interp),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            *args = raveraging_defaults;
            raveraging_dialog_update(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    return TRUE;
}

static void
raveraging_dialog_update(RAveragingControls *controls,
                         RAveragingArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->radius), args->radius);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->angle), args->angle);
    gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->interp),
                                  args->interp);
}

static void
raverage_rphi_to_xy(gdouble r, gdouble phi,
                    gdouble *x, gdouble *y,
                    gpointer user_data)
{
    const TransformData *td = (const TransformData*)user_data;

    *x = td->xc + (r - 0.5)*cos((phi - 0.5)*td->phiscale);
    *y = td->yc + (r - 0.5)*sin((phi - 0.5)*td->phiscale);
}

static void
raverage_xy_to_rphi(gdouble x, gdouble y,
                    gdouble *r, gdouble *phi,
                    gpointer user_data)
{
    const TransformData *td = (const TransformData*)user_data;
    gdouble rx = x - td->xc, ry = y - td->yc;

    *r = sqrt(rx*rx + ry*ry);
    /* Additional 180 to cover large angular sigma. */
    *phi = td->phioff + atan2(-ry, -rx)/td->phiscale;
}

static void
filter_radial(GwyDataField *dfield,
              gdouble radius, gdouble angle,
              GwyInterpolationType interp)
{
    GwyDataField *rdfield, *erdfield;
    gint xres, yres, rres, ares;
    TransformData transformdata;

    xres = gwy_data_field_get_xres(dfield);
    yres = gwy_data_field_get_yres(dfield);
    rres = sqrt(xres*xres + yres*yres)/2;
    ares = (GWY_ROUND(G_PI*MAX(xres, yres)) + 1)/2*2;
    transformdata.phiscale = 2.0*G_PI/ares;
    transformdata.phioff = 1.5*ares + 0.5;
    transformdata.xc = xres/2 + 0.5;
    transformdata.yc = yres/2 + 0.5;

    rdfield = gwy_data_field_new(rres, ares, rres, ares, FALSE);
    gwy_data_field_distort(dfield, rdfield, raverage_rphi_to_xy,
                           &transformdata, interp,
                           GWY_EXTERIOR_BORDER_EXTEND, 0.0);

    /* Additional 180 to cover large angular sigma. */
    erdfield = gwy_data_field_extend(rdfield, 0, 0, ares, ares,
                                     GWY_EXTERIOR_PERIODIC, 0.0, FALSE);
    g_object_unref(rdfield);

    if (radius > 0)
        gwy_data_field_row_gaussian(erdfield, radius);
    if (angle > 0)
        gwy_data_field_column_gaussian(erdfield, angle);

    gwy_data_field_distort(erdfield, dfield, raverage_xy_to_rphi,
                           &transformdata, interp,
                           GWY_EXTERIOR_BORDER_EXTEND, 0.0);
    g_object_unref(erdfield);
}

static void
radius_changed(GtkAdjustment *adj, RAveragingControls *controls)
{
    controls->args->radius = gtk_adjustment_get_value(adj);
}

static void
angle_changed(GtkAdjustment *adj, RAveragingControls *controls)
{
    controls->args->angle = gtk_adjustment_get_value(adj);
}

static void
interp_changed(GtkComboBox *combo, RAveragingControls *controls)
{
    controls->args->interp = gwy_enum_combo_box_get_active(combo);
}

static const gchar angle_key[]  = "/module/raveraging/angle";
static const gchar interp_key[] = "/module/raveraging/interp";
static const gchar radius_key[] = "/module/raveraging/radius";

static void
raveraging_sanitize_args(RAveragingArgs *args)
{
    args->radius = CLAMP(args->radius, 0.0, 1000.0);
    args->angle = CLAMP(args->angle, 0.0, 180.0);
    args->interp = gwy_enum_sanitize_value(args->interp,
                                           GWY_TYPE_INTERPOLATION_TYPE);
}

static void
raveraging_load_args(GwyContainer *container, RAveragingArgs *args)
{
    *args = raveraging_defaults;

    gwy_container_gis_double_by_name(container, radius_key, &args->radius);
    gwy_container_gis_double_by_name(container, angle_key, &args->angle);
    gwy_container_gis_enum_by_name(container, interp_key, &args->interp);
    raveraging_sanitize_args(args);
}

static void
raveraging_save_args(GwyContainer *container, RAveragingArgs *args)
{
    gwy_container_set_double_by_name(container, radius_key, args->radius);
    gwy_container_set_double_by_name(container, angle_key, args->angle);
    gwy_container_set_enum_by_name(container, interp_key, args->interp);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
