/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2012 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iconfigure.h"
#if ISHELL_INCLUDED(ISHELL_GG)


#include "iggmainwindow.h"


#include "ianimator.h"
#include "ibasicdatasubjects.h"
#include "icontrolmodule.h"
#include "icrosssectionviewsubject.h"
#include "idata.h"
#include "idataexplorer.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "iedition.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ifile.h"
#include "ifileloader.h"
#include "ihelpfactory.h"
#include "iimagefactory.h"
#include "iparallelmanager.h"
#include "iparticlesviewsubject.h"
#include "irendertool.h"
#include "ishell.h"
#include "isurfaceviewsubject.h"
#include "isystem.h"
#include "itensorfieldviewsubject.h"
#include "itextactor.h"
#include "ivectorfieldviewsubject.h"
#include "iversion.h"
#include "iviewmodule.h"
#include "ivolumeviewsubject.h"
#include "iwriter.h"

#include "iggdialoganimatingprogress.h"
#include "iggdialogauto.h"
#include "iggdialogcommandline.h"
#include "iggdialogdataexplorer.h"
#include "iggdialogfilesetexplorer.h"
#include "iggdialoghelp.h"
#include "iggdialogimagecomposer.h"
#include "iggdialogloadfile.h"
#include "iggdialogpaletteeditor.h"
#include "iggdialogparallelcontroller.h"
#include "iggdialogperformancemeter.h"
#include "iggdialogpickerwindow.h"
#include "iggdialogrenderingprogress.h"
#include "iggdialogscriptdebugger.h"
#include "iggextensionwindow.h"
#include "iggframetopparent.h"
#include "iggrenderwindow.h"
#include "iggpagecrosssection.h"
#include "iggpagedata.h"
#include "iggpageparticles.h"
#include "iggpagesurface.h"
#include "iggpagetensorfield.h"
#include "iggpagevectorfield.h"
#include "iggpageview.h"
#include "iggpagevolume.h"
#include "iggshell.h"
#include "iggwidgetarea.h"
#include "iggwidgetkeyhandler.h"
#include "iggwidgetkeylineedit.h"
#include "iggwidgetkeyslider.h"
#include "iggwidgetmisc.h"
#include "iggwidgetotherbutton.h"
#include "iggwidgetprogressbar.h"
#include "iggwidgetrendermodebutton.h"
#include "iggwidgettext.h"

#include "ibgdialogsubject.h"
#include "ibgextensionwindowsubject.h"
#include "ibgframesubject.h"
#include "ibgmainwindowsubject.h"
#include "ibgmenuwindowsubject.h"
#include "ibgrenderwindowsubject.h"
#include "ibgwidgetareasubject.h"
#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgetselectionboxsubject.h"
#include "ibgwindowhelper.h"

#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkRenderWindowCollection.h>
#include <vtkStructuredPoints.h>
#include <vtkVersion.h>

#include "iggsubjectfactory.h"

//
//  templates
//
#include "iarraytemplate.h"


using namespace iParameter;
using namespace iParameter;


IOBJECT_DEFINE_TYPE(iggMainWindow,MainWindow,-mw,iObjectType::_Helper);  // helper type

IOBJECT_DEFINE_KEY(iggMainWindow,BookOrientation,bo,Int,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Docked,d,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,ExtensionGeometry,eg,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,WindowUnderFocusCurrent,fc,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Geometry,g,Int,4);
IOBJECT_DEFINE_KEY(iggMainWindow,InteractorHelp,ih,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,IsIdiosyncratic,ii,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,OptionsAreGlobal,og,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,AllowPrePopulateToolBar,pp,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,SlidersAreEditable,se,Bool,1);
IOBJECT_DEFINE_KEY(iggMainWindow,Theme,t,String,1);
IOBJECT_DEFINE_KEY(iggMainWindow,TabMode,tm,Int,1);
IOBJECT_DEFINE_KEY(iggMainWindow,ToolTips,tt,Bool,1);


namespace iggMainWindow_Private
{
	//
	//  Menu ids
	//
	namespace Menu
	{
		namespace File
		{
			const int OpenUniformScalars = 1;
			const int OpenBasicParticles = 2;
			const int OpenUniformVectors = 3;
			const int OpenUniformTensors = 4;
			const int SaveState = 90;
			const int SaveStateAs = 91;
			const int LoadStateFrom = 92;
			const int Exit = 99;
			const int Max = 100;
		};

		namespace Dialog
		{
			const int ScriptDebugger = 101;
			const int PaletteEditor = 102;
			const int PickerWindow = 103;
			const int ParallelController = 105;
			const int FileSetExplorer = 106;
			const int DataExplorer = 107;
			const int ImageComposer = 108;
			const int CommandLine = 109;
			const int PerformanceMeter = 110;
			const int AutoBlock = 120;
			const int AutoBegin = 121;
			const int AutoEnd = 200;
			const int Max = 200;
		};

		namespace Style
		{
			const int DockWindow = 201;
			const int InteractorHelp = 202;
			const int ToolTips = 203;
			const int TabText = 211;
			const int TabIcon = 212;
			const int TabBoth = 213;
			const int RenderImmediate = 221;
			const int RenderDelayed = 222;
			const int RenderResetAll = 223;
			const int SlidersEditable = 224;
			const int IsIdiosyncratic = 225;
			const int DetailLoadedDataInfo = 226;
			const int AutoRender = 227;
			const int Theme = 230;
			const int RotateBook = 240;
			const int Max = 300;
		};

		namespace Option
		{
			const int Antialiasing = 301;
			const int IfritBox = 311;
			const int ClassicBox = 312;
			const int HairBox = 313;
			const int AxesBox = 314;
			const int FontTypeVector = 322;
			const int FontTypeArial = 323;
			const int FontTypeCourier = 324;
			const int FontTypeTimes = 325;
			const int FontSizem5 = 331;
			const int FontSizem4 = 332;
			const int FontSizem3 = 333;
			const int FontSizem2 = 334;
			const int FontSizem1 = 335;
			const int FontSizec0 = 336;
			const int FontSizep1 = 337;
			const int FontSizep2 = 338;
			const int FontSizep3 = 339;
			const int FontSizep4 = 340;
			const int FontSizep5 = 341;
			const int ImageFormatPNG = 351;
			const int ImageFormatJPG = 352;
			const int ImageFormatPPM = 353;
			const int ImageFormatBMP = 354;
			const int ImageFormatTIF = 355;
			const int ImageFormatEPS = 356;
			const int ImageZoom001 = 361;
			const int ImageZoom002 = 362;
			const int ImageZoom003 = 363;
			const int ImageZoom004 = 364;
			const int ImageZoom005 = 365;
			const int ImageZoom006 = 366;
			const int ImageZoom008 = 367;
			const int ImageZoom010 = 368;
			const int ImageZoom015 = 369;
			const int ImageZoom020 = 370;
			const int ImageZoom030 = 371;
			const int ImageZoom040 = 372;
			const int ImageZoom050 = 373;
			const int ImageZoom060 = 374;
			const int ImageZoom080 = 375;
			const int ImageZoom100 = 376;
			const int PostScriptPaperFormatA0 = 401;
			const int PostScriptPaperFormatA1 = 402;
			const int PostScriptPaperFormatA2 = 403;
			const int PostScriptPaperFormatA3 = 404;
			const int PostScriptPaperFormatA4 = 405;
			const int PostScriptPaperFormatA5 = 406;
			const int PostScriptPaperFormatA6 = 407;
			const int PostScriptPaperFormatA7 = 408;
			const int PostScriptPaperFormatA8 = 409;
			const int PostScriptPaperFormatL1 = 410;
			const int PostScriptPaperFormatL2 = 411;
			const int PostScriptOrientationPortrait = 421;
			const int PostScriptOrientationLandscape = 422;
			const int ArtifactCorrection = 431;
			const int BackgroundImageFixedAspect = 432;
			const int SettingsGlobal = 499;
			const int Max = 500;
		};

		namespace Help
		{
			const int Contents = 501;
			const int About = 502;
			const int Max = 599;
		};
	};
	
	namespace ToolBar
	{
		const int OpenWindowsPage = 801;
		const int MinimizeWindows = 802;
		const int MaximizeWindows = 803;
		const int MoveAllTogether = 804;

		const int ShowSurface = 901;
		const int ShowCrossSection = 902;
		const int ShowVolume = 903;
		const int ShowParticles = 904;
		const int ShowVectorField = 905;
		const int ShowTensorField = 906;
		const int Max = 999;
	};

	class InteractorHelpTextView : public iggWidgetTextBrowser
	{

	private:

		struct Entry
		{
			const char* Name;
			const char* Body;
			Entry(const char *n, const char *b) : Name(n), Body(b) {}
		};

	public:

		InteractorHelpTextView(iggFrame *parent) : iggWidgetTextBrowser(true,false,parent)
		{
			mShownData = 0;
		}

		virtual void Show(bool s)
		{
			this->UpdateWidget();
			iggWidgetTextBrowser::Show(s);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			static Entry data0[] = 
			{ 
				Entry("U key","dump the current view into an image."),
				Entry("F key","toggle the Full Screen mode on and off."),
				Entry("","--------------------"),
				Entry("","In addition, when the measuring box is activated, the following keys are accepted:"),
				Entry("+/- keys","scale the measuring box up/down."),
				Entry("</> keys","adjust box opacity."),
				Entry("","(use Style->Show interactor help menu to turn this help off)")
			};

			static Entry dataD[] = 
			{ 
				Entry(   "Trackball/Joystick mode",""),
				Entry("","-------------------------"),
				Entry("Left button","rotate the camera around its focal point."),
				Entry("[Ctrl]+Left button","spin the camera around the Z-axis (axis perpendicular to the screen)."),
				Entry("Middle button","pan (move sideways) the camera."),
				Entry("Right button","zoom in/out."),
				Entry("P key","perform a pick operation."),
				Entry("R key","reset the camera view along the current view direction."),
				Entry("S key","show all solid objects as surfaces."),
				Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataF[] = 
			{ 
				Entry(   "Fly-by mode",""),
				Entry("","-------------"),
				Entry("Left button","forward motion."),
				Entry("Right button","reverse motion."),
				Entry("[Shift] key","accelerator in mouse and key modes."),
				Entry("[Ctrl]+[Shift] keys","causes sidestep instead of steering."),
				Entry("+/- keys","increase/deacrease normal speed.")
			};

			static Entry dataK[] = 
			{ 
				Entry(   "Keyboard mode",""),
					Entry("","---------------"),
					Entry("Arrow keys (or HJKL keys)","rotate the camera around its focal point."),
					Entry("[Ctrl]+Arrow keys (or [Ctrl]+HJKL keys)","spin the camera around the Z-axis (axis perpendicular to the screen)."),
					Entry("[Shift]+Arrow keys (or [Shift]+HJKL keys)","pan (move sideways) the camera."),
					Entry("+/- keys","zoom in/out."),
					Entry("A/Z keys","speed up/slow down the interaction."),
					Entry("P key","perform a pick operation."),
					Entry("R key","reset the camera view along the current view direction."),
					Entry("S key","show all solid objects as surfaces."),
					Entry("W key","show all solid objects as wireframe (faster).")
			};

			static Entry dataC[] = 
			{ 
				Entry(   "Camera path mode",""),
					Entry("","------------------"),
					Entry("Left button","move a handle or the whole path."),
					Entry("Middle button","move the path (if you seem unable to grab the line, change the number of steps slightly)."),
					Entry("[Ctrl]+Middle button","spin the path."),
					Entry("Right button","scale the path.")
			};

			int i, n = 0;
			Entry *data = 0;
			if(this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyInteractorStyle(),i))
			{
				switch(i)
				{
				case InteractorStyle::Trackball:
				case InteractorStyle::Joystick:
					{
						data = dataD;
						n = sizeof(dataD)/sizeof(Entry);
						break;
					}
				case InteractorStyle::Flight:
					{
						data = dataF;
						n = sizeof(dataF)/sizeof(Entry);
						break;
					}
				case InteractorStyle::Keyboard:
					{
						data = dataK;
						n = sizeof(dataK)/sizeof(Entry);
						break;
					}
				}
			}
			if(this->GetShell()->GetControlModule()->QueryValue(iAnimator::KeyStyle(),i) && i==4)
			{
				data = dataC;
				n = sizeof(dataC)/sizeof(Entry);
			}
			if(n == 0)
			{
#ifdef I_CHECK1
				IERROR_LOW("Should not be here");
#endif
			}

			if(data != mShownData)
			{
				this->Clear();
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data[i].Name,data[i].Body,iColor(0,0,255));
				}
				n = sizeof(data0)/sizeof(Entry);
				for(i=0; i<n; i++)
				{
					this->AppendTextLine(data0[i].Name,data0[i].Body,iColor(0,0,255));
				}
				mShownData = data;
			}
			this->iggWidgetTextBrowser::UpdateWidgetBody();
		}

		Entry* mShownData;
	};


	class BusyIndicator : public iggWidgetImageFlipper
	{

	public:

		BusyIndicator(iggFrame *parent) : iggWidgetImageFlipper(parent)
		{
			mSubject->AddImage(*iImageFactory::FindIcon("light_green.png"),true);
			mSubject->AddImage(*iImageFactory::FindIcon("light_red.png"),true);

			this->SetBaloonHelp("Ready/busy indicator","A green light indicates that IFrIT is ready for user input. The red light comes on during the time IFrIT is busy doing some work. User input is blocked while the red light is on.");
		}

		void SetBusy(bool s)
		{
			if(s) mSubject->ShowImage(1); else mSubject->ShowImage(0);
		}

	protected:

		virtual void UpdateWidgetBody()
		{
		}
	};


	class VisitedFileList : public iggWidget
	{

	public:

		VisitedFileList(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,"File:");

			this->SetBaloonHelp("List of visited files","This drop-down box shows the list of recently visited files. A file from the list can be reloaded by simply selecting it.");
		}

	protected:

		virtual void UpdateWidgetBody()
		{
			int i, k;
			const iArray<iDataReader::VisitedFile> &list(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList());

			k = 0;
			do
			{
				mSubject->Clear();
				for(i=list.MaxIndex(); i>=0; i--)
				{
					mSubject->InsertItem(iString((k>0)?"...":"")+list[i].Name.Part(k));
				}
				if(k == 0) k += 4; else k++;
			}
			while(!mSubject->DoesContentFit());
		}

		virtual void OnInt1Body(int v)
		{
			const iArray<iDataReader::VisitedFile> &list(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetVisitedFilesList());

			if(v>=0 && v<list.Size())
			{
				this->GetMainWindow()->LoadData(list[list.MaxIndex()-v].Loader->GetDataType(0),list[list.MaxIndex()-v].Name);
			}
		}

		ibgWidgetComboBoxSubject *mSubject;
	};


	class ImageButton : public iggWidgetSimpleButton
	{

	public:

		ImageButton(iggFrame *parent) : iggWidgetSimpleButton("Image",parent)
		{
			this->SetBaloonHelp("Dump an image","Dump the scene in the current visualization window as an image. Image format and zoom factor are set in the Options menu.");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetShell()->GetControlModule()->GetViewModule()->DumpImages();
			this->GetMainWindow()->Block(false);
		}
	};


	class RenderButton : public iggWidgetSimpleButton
	{

	public:

		RenderButton(iggFrame *parent) : iggWidgetSimpleButton("Render",parent)
		{
			this->SetBaloonHelp("Render the scene","Render the current scene in the current visualization window.");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetShell()->GetControlModule()->Render();
			this->GetMainWindow()->Block(false);
		}
	};


	class HideInteractorHelpButton : public iggWidgetSimpleButton
	{

	public:

		HideInteractorHelpButton(iggFrame *parent) : iggWidgetSimpleButton("Close help",parent)
		{
			this->SetBaloonHelp("Close interactor help");
		}

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			this->GetMainWindow()->ShowInteractorHelp(false);
			this->GetMainWindow()->Block(false);
		}
	};


	class Book : public iggFrameBook
	{
		
	public:
		
		Book(iggFrame *parent) : iggFrameBook(parent,true)
		{
		}

		void PolishPages()
		{
			int i;
			for(i=0; i<this->mPages.Size(); i++)
			{
				iDynamicCast<iggPageMain,iggFrame>(INFO,this->GetPage(i))->Polish();
			}
		}

	protected:

		virtual void OnInt1Body(int i)
		{
			if(!iRequiredCast<iggShell>(INFO,this->GetShell())->CheckCondition(iParameter::Condition::SlowRemoteConnection) && this->GetMainWindow()->GetExtensionWindow()!=0)
			{
				this->GetMainWindow()->GetExtensionWindow()->OpenBookPageByIndex(i);
			}
		}
	};


	class DialogAbout : public iggDialog
	{

	public:

		DialogAbout(iggMainWindow *parent) : iggDialog(parent,DialogFlag::Modal,0,"About",0,3,"Ok")
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			int i, n;
			iString s;

			s = "Version " + iVersion::GetVersion();
			switch(sizeof(void*))
			{
			case 4:
				{
					s += " (32-bit)";
					break;
				}
			case 8:
				{
					s += " (64-bit)";
					break;
				}
			}

			mFrame->AddLine(new iggWidgetTextArea("%b%+IFrIT - Ionization FRont Interactive Tool",mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea(s,mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea(iEdition::GetEditionName(),mFrame),3);

			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);

			mFrame->AddLine(new iggWidgetTextArea(iString("VTK version ")+vtkVersion::GetVTKVersion(),mFrame),3);

			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following extensions:",mFrame),3);
			s = iVersion::GetIncludedExtensions();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("%bThis installation includes the following shells:",mFrame),3);
			s = iVersion::GetIncludedShells();
			n = s.Contains(';');
			for(i=0; i<n; i++) mFrame->AddLine(new iggWidgetTextArea(s.Section(";",i,i),mFrame),3);
			mFrame->AddLine(new iggWidgetTextArea("",mFrame),3);
		}

		virtual void ShowBody(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::ShowBody(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	class DialogDocking : public iggDialog
	{

	public:

		DialogDocking(iggMainWindow *parent) : iggDialog(parent,DialogFlag::Blocking|DialogFlag::NoTitleBar,0,"Docking...",0,3,0)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mFlipper = new iggWidgetLogoFlipper(mFrame);
			mFrame->AddLine(0,mFlipper);
			mFrame->AddLine(new iggWidgetTextArea("%b%+Rearranging windows...",mFrame),3);
		}

		virtual void ShowBody(bool s)
		{
			if(s) mFlipper->Start(); else mFlipper->Abort();
			iggDialog::ShowBody(s);
		}

	protected:

		iggWidgetLogoFlipper *mFlipper;
	};


	//
	//  A small dialog with three line edit to set the axes labels
	//
	class AxesLabelsDialog : public iggDialogAuto
	{

	public:

		AxesLabelsDialog(iggMainWindow *parent) : iggDialogAuto(parent,"Axes Labels",3)
		{
			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

		void CompleteConstructionBody()
		{
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"X: Label",iViewModule::KeyAxesBoxLabels(),RenderMode::UseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,0),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,1));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Y: Label",iViewModule::KeyAxesBoxLabels(),RenderMode::UseGlobal,mFrame,1),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,3));
			mFrame->AddLine(new iggWidgetKeyTextLineEdit(false,"Z: Label",iViewModule::KeyAxesBoxLabels(),RenderMode::UseGlobal,mFrame,2),new iggWidgetKeyFloatLineEdit("Min",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,4),new iggWidgetKeyFloatLineEdit("Max",iViewModule::KeyAxesBoxRanges(),RenderMode::UseGlobal,mFrame,5));
		}
	};

#ifdef I_DEBUG
	class DebugHelperEmphasizeAllWidgetsCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeAllWidgetsCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Show all layouts",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		void OnVoid1Body()
		{
			this->GetMainWindow()->Block(true);
			iggWidget::EmphasizeLayouts(mSubject->IsDown());
			this->GetMainWindow()->Block(false);
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperEmphasizeUnderCursorCheckBox : public iggWidget
	{

	public:

		DebugHelperEmphasizeUnderCursorCheckBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetButtonSubject(this,ButtonType::CheckBox,"Show under cursor",1);
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
		}

		void OnVoid1Body()
		{
			ibgMainWindowSubject::SetEmphasizeUnderCursor(mSubject->IsDown());
		}

		ibgWidgetButtonSubject *mSubject;
	};

	class DebugHelperFlipAllPagesButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperFlipAllPagesButton(iggFrame *sf, iggFrame *parent) : iggWidgetSimpleButton("Flip all pages",parent)
		{
			mStartFrame = sf;
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			if(mStartFrame != 0) mStartFrame->FlipThroughAllChildren();
			this->GetMainWindow()->Block(false);
		}

		iggFrame *mStartFrame;
	};

	class DebugHelperFlashAllDialogsButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperFlashAllDialogsButton(iggFrame *parent) : iggWidgetSimpleButton("Flash all dialogs",parent)
		{
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			iggDialog::FlashAllDialogs();
			this->GetMainWindow()->Block(false);
		}
	};

	class DebugHelperCreateUGButton : public iggWidgetSimpleButton
	{

	public:

		DebugHelperCreateUGButton(iggFrame *parent) : iggWidgetSimpleButton("Create User Guide(s)",parent)
		{
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void Execute()
		{
			this->GetMainWindow()->Block(true);
			iHelpFactory::CreateUserGuide();
			this->GetMainWindow()->Block(false);
		}
	};

	class DebugHelperParallelRadioBox : public iggWidget
	{

	public:

		DebugHelperParallelRadioBox(iggFrame *parent) : iggWidget(parent)
		{
			mSubject = iggSubjectFactory::CreateWidgetRadioBoxSubject(this,1,"Parallel execution");
			mSubject->InsertItem("Parallel");
			mSubject->InsertItem("Proc 1");
			mSubject->InsertItem("Proc 2");
			mSubject->InsertItem("Procs 1+2");
			mSubject->InsertItem("No stiches");
			mNeedsBaloonHelp = false;
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			if(iParallelManager::DebugSwitch>0 && iParallelManager::DebugSwitch<mSubject->Count()) mSubject->SetValue(iParallelManager::DebugSwitch); else mSubject->SetValue(0);
		}

		void OnInt1Body(int v)
		{
			this->GetMainWindow()->Block(true);
			iParallelManager::DebugSwitch = v;
			this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->ResetPipeline();
			this->GetShell()->GetControlModule()->GetViewModule()->Render();
			this->GetMainWindow()->Block(false);
		}

		ibgWidgetRadioBoxSubject *mSubject;
	};

	class DebugHelperDialog : public iggDialog
	{

	public:

		DebugHelperDialog(iggFrame *sf, iggMainWindow *parent) : iggDialog(parent,0U,0,"Debug Helper",0,1)
		{
			mStartFrame = sf;

			if(this->ImmediateConstruction()) this->CompleteConstruction();
		}

	protected:

		void CompleteConstructionBody()
		{
			mFrame->AddLine(new DebugHelperParallelRadioBox(mFrame));
			mFrame->AddLine(new DebugHelperEmphasizeAllWidgetsCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperEmphasizeUnderCursorCheckBox(mFrame));
			mFrame->AddLine(new DebugHelperFlipAllPagesButton(mStartFrame,mFrame));
			mFrame->AddLine(new DebugHelperFlashAllDialogsButton(mFrame));
			mFrame->AddLine(new DebugHelperCreateUGButton(mFrame));
		}

		iggFrame *mStartFrame;
	};
#endif
};


using namespace iggMainWindow_Private;


//
//  **************************************************************************************
//
//      INITIALIZATION
//
//  **************************************************************************************
//
iggMainWindow::iggMainWindow(iShell *shell) : iObject("MainWindow"), iggMenuWindow(shell), mReloadDataInfo(new iDataInfo), mEraseDataInfo(new iDataInfo)
{
	mInitialized = mDocked = mDoNotOfferToReload = mDetailLoadedDataInfo = mAllowPrePopulateToolBar = mMoveTogether = false;
	mInteractorHelp = mToolTips = mOptionsAreGlobal = mWindowUnderFocusCurrent = mAutoRender = true;
	mIdiosyncratic = true;

	mProgressBarMode = 0;
	mCurrentTheme = -1;
	mBlockLevel = 0;

	mStateFileName = this->GetShell()->GetEnvironment(Environment::Base) + "ifrit.ini";

	mInitialGeometry[0] = mInitialGeometry[1] = mInitialGeometry[2] = mInitialGeometry[3] = 0;

	mInMove = false;
	mPrevPos[0] = mPrevPos[1] = -1;
	mNextPos[0] = mNextPos[1] = -1;

	mMainWindow = this;

	mLog = 0;

	IERROR_ASSERT(mReloadDataInfo);
	IERROR_ASSERT(mEraseDataInfo);
}


void iggMainWindow::StartInitialization()
{
	//
	//  MainSubject must be created first
	//
	mMainSubject = iggSubjectFactory::CreateMainWindowSubject(this,3);

	//
	//  Have we created MenuSubject as well?
	//
	if(this->GetSubject() == 0)
	{
		mTwoSubjectsAreTheSame = false;
		this->AttachSubject(iggSubjectFactory::CreateMenuWindowSubject(this,iImageFactory::FindIcon("genie1gui.png"),"Ionization Front Interactive Tool"),3);
	}
	else
	{
		mTwoSubjectsAreTheSame = true;
	}

	//
	//  Top frames
	//
	mBusyIndicatorFrame = new iggFrameTopParent(this->GetShell());
	mVisitedFileListFrame = new iggFrameTopParent(this->GetShell());

	mMainSubject->SetTopParentSubjects(mBusyIndicatorFrame,mVisitedFileListFrame);

	IERROR_ASSERT(mBusyIndicatorFrame->GetSubject());
	IERROR_ASSERT(mVisitedFileListFrame->GetSubject());

	//
	//  Other components
	//
	mProgressBar = new iggWidgetProgressBar(mGlobalFrame,true);

	//
	//  Status bar
	//
	mBusyIndicator = new BusyIndicator(mBusyIndicatorFrame);
	mBusyIndicatorFrame->AddLine(mBusyIndicator);
	mVisitedFileList = new VisitedFileList(mVisitedFileListFrame);
	mVisitedFileListFrame->AddLine(mVisitedFileList);

	mMainSubject->PopulateStatusBar(mBusyIndicatorFrame,mProgressBar->GetSubject(),mVisitedFileListFrame);

	mGlobalFrame->SetPadding(true);

	//
	//  Extension
	//
	mExtensionWindow = new iggExtensionWindow(this);
	mExtensionWindow->CompleteInitialization();

	//
	//  Base frame
	//
	mBaseFrame = new iggFrameFlip(mGlobalFrame);

	//
	//  Create the book frame but not layout it
	//
	iggFrame *tmpf = new iggFrame(mBaseFrame);
	mLog = new iggWidgetTextBrowser(false,false,mGlobalFrame);
	mLog->SetBaloonHelp("Log window","In this window IFrIT logs information about the data files, non-critical execution error, performance information, etc.");

	//
	//  Adjust our behaviour depending on the running conditions
	//
	iggShell *s = iRequiredCast<iggShell>(INFO,this->GetShell());
	
	if(s->CheckCondition(iParameter::Condition::SlowRemoteConnection))
	{
		ibgRenderWindowSubject::RenderOnFocus = true;
		mAutoRender = false;
	}

	//
	//  Create dialogs (needed for book pages)
	//
	mDialogAbout = new DialogAbout(this); IERROR_ASSERT(mDialogAbout);
	mDialogAnimatingProgress = new iggDialogAnimatingProgress(this); IERROR_ASSERT(mDialogAnimatingProgress);
	mDialogAxesLabels = new AxesLabelsDialog(this); IERROR_ASSERT(mDialogAxesLabels);
	mDialogCommandLine = new iggDialogCommandLine(this); IERROR_ASSERT(mDialogCommandLine);
	mDialogDataExplorer = new iggDialogDataExplorer(this); IERROR_ASSERT(mDialogDataExplorer);
	mDialogDocking = new DialogDocking(this); IERROR_ASSERT(mDialogDocking);
	mDialogFileSetExplorer = new iggDialogFileSetExplorer(this); IERROR_ASSERT(mDialogFileSetExplorer);
	mDialogHelp = new iggDialogHelp(this); IERROR_ASSERT(mDialogHelp);
	mDialogImageComposer = new iggDialogImageComposer(this); IERROR_ASSERT(mDialogImageComposer);
	mDialogLoadFile = new iggDialogLoadFile(this); IERROR_ASSERT(mDialogLoadFile);
	mDialogPaletteEditor = new iggDialogPaletteEditor(this); IERROR_ASSERT(mDialogPaletteEditor);
	mDialogParallelController = new iggDialogParallelController(this); IERROR_ASSERT(mDialogParallelController);
	mDialogPerformanceMeter = new iggDialogPerformanceMeter(this); IERROR_ASSERT(mDialogPerformanceMeter);
	mDialogPickerWindow = new iggDialogPickerWindow(this); IERROR_ASSERT(mDialogPickerWindow);
	mDialogRenderingProgress = 0; // we create this one later, so that it does not pop up during the initialization process
	mDialogScriptDebugger = new iggDialogScriptDebugger(this); IERROR_ASSERT(mDialogScriptDebugger);

	//
	//  Create the book
	//
	Book *book = new Book(tmpf);
	mBook = book;

	mViewPage = new iggPageView(mBook);
	mDataPage = new iggPageData(mBook);
	
	mPages[0] = new iggPageSurface(mBook);
	mPages[1] = new iggPageCrossSection(mBook);
	mPages[2] = new iggPageVolume(mBook);
	mPages[3] = mParticlesPage = new iggPageParticles(mBook);
	mPages[4] = new iggPageVectorField(mBook);
	mPages[5] = new iggPageTensorField(mBook);

	iParticlesViewSubject::UseFullState(false); // optimization, since we use iParticleGroup keys

	mBook->AddPage("View",iImageFactory::FindIcon("view.png"),mViewPage);
	mBook->AddPage("Surface",iImageFactory::FindIcon("surf.png"),mPages[0]);
	mBook->AddPage("Cross section",iImageFactory::FindIcon("xsec.png"),mPages[1]);
	mBook->AddPage("Volume",iImageFactory::FindIcon("volv.png"),mPages[2]);
	mBook->AddPage("Particles",iImageFactory::FindIcon("part.png"),mPages[3]);
	mBook->AddPage("Vector field",iImageFactory::FindIcon("vect.png"),mPages[4]);
	mBook->AddPage("Tensor field",iImageFactory::FindIcon("tens.png"),mPages[5]);
	mBook->AddPage("Data",iImageFactory::FindIcon("data.png"),mDataPage);

	//
	//  Layout the book frame
	//
	tmpf->AddLine(mBook);
	mBaseFrame->AddLayer(tmpf);

	//
	//  Interactor help frame
	//
	tmpf = new iggFrame("",mBaseFrame,3);
	tmpf->AddLine(new iggWidgetTextArea("%b%+Interactor Help",tmpf),3);
	tmpf->AddLine(new InteractorHelpTextView(tmpf),3);
	tmpf->AddLine((iggWidget *)0,new HideInteractorHelpButton(tmpf),(iggWidget *)0);
	tmpf->SetRowStretch(0,1);
	tmpf->SetRowStretch(1,10);
	mBaseFrame->AddLayer(tmpf);
	mBaseFrame->ShowLayer(0);

	mGlobalFrame->AddLine(mBaseFrame,3);

	iggFrameFlip *flip1 = new iggFrameFlip(mGlobalFrame,false);
	iggFrame *tmp1 = new iggFrame(flip1,3);
	mRenderButton = new RenderButton(tmp1);
	mRenderButton->Show(!mAutoRender);
	tmp1->SetPadding(true);
	tmp1->AddLine(new ImageButton(tmp1),mRenderButton);
	tmp1->SetColStretch(2,10);
	flip1->AddLayer(tmp1);
	flip1->ShowLayer(0);

	mDataTypeFrame = new iggFrameFlip(mGlobalFrame,false);

	//mGlobalFrame->AddLine(tmp1,new iggWidgetTextArea("",mGlobalFrame),tmp2);
	mGlobalFrame->AddLine(flip1,(iggWidget *)0,mDataTypeFrame);
	mGlobalFrame->AddLine(mLog,3);
	mGlobalFrame->SetRowStretch(0,1);
	mGlobalFrame->SetRowStretch(1,0);
	if(s->IsDesktopSmall(true))
	{
		mLog->Show(false);
		mGlobalFrame->SetRowStretch(2,0);
	}
	else
	{
		mGlobalFrame->SetRowStretch(2,10);
	}

	book->PolishPages();

	iEdition::ApplySettings(this,this->Type());

	this->ShowToolTips(mToolTips);

#if defined(I_DEBUG)
	mDialogDebugHelper = new DebugHelperDialog(mGlobalFrame,this);
#endif

	//
	//  Connect pre-existing RenderWindows
	//
	int i;
	for(i=0; i<this->GetShell()->GetControlModule()->GetNumberOfViewModules(); i++)
	{
		iRequiredCast<iggRenderWindow>(INFO,this->GetShell()->GetControlModule()->GetViewModule(i)->GetRenderWindow())->AttachToMainWindow(this);
	}
}


void iggMainWindow::PreShowInitialization()
{
	this->BuildMenus();
	if(mInitialGeometry[2]>0 && mInitialGeometry[3]>0) this->GetSubject()->SetWindowGeometry(mInitialGeometry);
}


void iggMainWindow::PostShowInitialization()
{
	mDialogRenderingProgress = new iggDialogRenderingProgress(this); IERROR_ASSERT(mDialogRenderingProgress);

	int wg[4];
	this->GetSubject()->GetWindowGeometry(wg);
	mPrevPos[0] = wg[0];
	mPrevPos[1] = wg[1];

	mInitialized = true;
}


iggMainWindow::~iggMainWindow()
{
	if(mDocked)
	{
		mMainSubject->RestoreWindowsFromDockedPositions();
	}

	delete mExtensionWindow;
	mExtensionWindow = 0; // it does not exist any more

	if(mDialogAbout != 0) delete mDialogAbout;
	if(mDialogAnimatingProgress != 0) delete mDialogAnimatingProgress;
	if(mDialogAxesLabels != 0) delete mDialogAxesLabels;
	if(mDialogCommandLine != 0) delete mDialogCommandLine;
	if(mDialogDataExplorer != 0) delete mDialogDataExplorer;
	if(mDialogDocking != 0) delete mDialogDocking;
	if(mDialogFileSetExplorer != 0) delete mDialogFileSetExplorer;
	if(mDialogHelp != 0) delete mDialogHelp;
	if(mDialogImageComposer != 0) delete mDialogImageComposer;
	if(mDialogLoadFile != 0) delete mDialogLoadFile;
	if(mDialogPaletteEditor != 0) delete mDialogPaletteEditor;
	if(mDialogParallelController != 0) delete mDialogParallelController;
	if(mDialogPerformanceMeter != 0) delete mDialogPerformanceMeter;
	if(mDialogPickerWindow != 0) delete mDialogPickerWindow;
	if(mDialogRenderingProgress != 0) delete mDialogRenderingProgress;
	if(mDialogScriptDebugger != 0) delete mDialogScriptDebugger;
#if defined(I_DEBUG)
	if(mDialogDebugHelper != 0) delete mDialogDebugHelper;
#endif

	delete mProgressBar;

	delete mGlobalFrame;
	delete mBusyIndicatorFrame;
	delete mVisitedFileListFrame;

	if(!mTwoSubjectsAreTheSame)
	{
		delete mMainSubject;
	}

	delete mReloadDataInfo;
	delete mEraseDataInfo;
}


void iggMainWindow::UpdateContents()
{
	mGlobalFrame->UpdateWidget();

	//
	//  Also update widgets that are not children of mGlobalFrame
	//
	mBusyIndicator->UpdateWidget();
	mVisitedFileList->UpdateWidget();

	//
	//  Update some of the dialogs
	//
	if(mDialogDataExplorer != 0) mDialogDataExplorer->UpdateDialog();
	if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->UpdateDialog();
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateDialog();
	if(mDialogPickerWindow != 0) mDialogPickerWindow->UpdateDialog();
	if(mViewPage != 0) mViewPage->GetWindowListDialog()->UpdateDialog();

	//
	//  Update script debugger
	//
	if(mDialogScriptDebugger != 0) mDialogScriptDebugger->UpdateAll();

	//
	//  Update extension too
	//
	mExtensionWindow->UpdateAll();
}


void iggMainWindow::Register(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.AddUnique(window);
}


void iggMainWindow::UnRegister(ibgWindowSubject *window)
{
	if(window != 0) mWindowList.Remove(window);
}


void iggMainWindow::RegisterAutoDialog(iggDialogAuto *d)
{
	if(d != 0) mAutoDialogList.AddUnique(d);
}


void iggMainWindow::UnRegisterAutoDialog(iggDialogAuto *d)
{
	if(d != 0) mAutoDialogList.Remove(d);
}


//
//  **************************************************************************************
//
//      DATA MANIPULATION
//
//  **************************************************************************************
//
void iggMainWindow::LoadData(const iDataType &type, const iString &filename, int mod)
{
	iDataInfo info(type);
	
	mProgressBarMode = 1;
	mDialogLoadFile->LoadData(info,filename,mod);
	mProgressBarMode = 0;
}


void iggMainWindow::AddReloadDataType(const iDataType &type)
{
	*mReloadDataInfo += type;
}


void iggMainWindow::ReloadData()
{
	mProgressBarMode = 1;
	mDialogLoadFile->ReloadData(*mReloadDataInfo);
	mProgressBarMode = 0;
	mReloadDataInfo->Erase();
}


void iggMainWindow::RequestReloadData(const iDataInfo &info)
{
	int i;

	for(i=0; i<info.Count(); i++) this->AddReloadDataType(info.Type(i));

	if(!mDoNotOfferToReload && this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsThereData(info))
	{
		int ret = this->PopupWindow("This control only takes effect when the data set is reloaded.",PopupWindow::Message,"Reload now","Later","Do not ask again");
		if(ret == 0) this->ReloadData();
		if(ret == 2) mDoNotOfferToReload = true;
	}
}


void iggMainWindow::AddEraseDataType(const iDataType &type)
{
	*mEraseDataInfo += type;
}


void iggMainWindow::RemoveEraseDataType(const iDataType &type)
{
	*mEraseDataInfo -= type;
}


void iggMainWindow::EraseData()
{
	int i;

	this->Block(true);
	
	for(i=0; i<mEraseDataInfo->Count(); i++)
	{
		this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->EraseData(mEraseDataInfo->Type(i));
		this->AfterEraseData(mEraseDataInfo->Type(i));
	}
	
	this->Block(false);
	mEraseDataInfo->Erase();
}


void iggMainWindow::AfterLoadData(const iDataType &type, const iString &filename)
{
	iDataReader *r = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();

	if(r->GetErrorStatus()->NoError() || r->GetErrorStatus()->Level()<0)  // warning only
	{
		this->WriteToLog("");
		this->WriteToLog("Loaded file",filename);
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(RenderOption::Clones);
		this->LogDataInfo(type,mDetailLoadedDataInfo);

		//
		//  Provide the FileSet info
		//
		if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->IsSet())
		{
			int m = 0;
			iString s;
			while(!this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m).IsNull())
			{
				if(m > 0) s += " + ";
				s += this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetFileSetDataType(m).GetTextName();
				m++;
			}
			this->WriteToLog("File Set: ",s);
		}
		mDataPage->UpdateOnDataLoad();

		if(r->GetErrorStatus()->IsError()) this->PopupWindow(r->GetErrorStatus()->Message());
	}
	else
	{
		if(!r->GetErrorStatus()->IsAbort()) this->PopupWindow("Error in loading data:\n"+r->GetErrorStatus()->Message(),PopupWindow::Error);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterReloadData(const iDataType &)
{
	if(this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorStatus()->Message().IsEmpty())
	{
		if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
		this->GetShell()->GetControlModule()->Render(RenderOption::Clones);
	}
	else
	{
		this->PopupWindow("Reloading of the current file set failed:\n"+this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetErrorStatus()->Message(),PopupWindow::Warning);
	}
	this->UpdateAll();
}


void iggMainWindow::AfterEraseData(const iDataType &)
{
	if(mDialogDataExplorer!=0 && mDialogDataExplorer->IsVisible()) mDialogDataExplorer->UpdateDialog();
	this->GetShell()->GetControlModule()->Render(RenderOption::Clones);

	this->UpdateAll();
}


void iggMainWindow::LogDataInfo(const iFileLoader *loader, bool details)
{
	int i, j;

	IERROR_ASSERT(loader);

	bool work = false;
	iDataSubject *subject;
	for(j=0; j<loader->NumStreams(); j++) if((subject = loader->GetSubject(j))->IsThereData())
	{
		iDataLimits *limits = subject->GetLimits();
		work = true;

		//
		//  Extension did not produce its own info, use the default one
		//
		this->WriteToLog(subject->GetDataType().GetTextName(),subject->GetDataType().MatchesKeyword("basic")?"":iString("( VTK class name: ")+subject->GetData()->GetClassName()+")");

		//
		//  Are we ImageData?
		//
		vtkImageData *im = vtkImageData::SafeDownCast(subject->GetData());
		if(im != 0)
		{
			int dims[3];
			im->GetDimensions(dims);
			this->WriteToLog("Dimensions: "+iString::FromNumber(dims[0])+" x "+iString::FromNumber(dims[1])+" x "+iString::FromNumber(dims[2]));
		}

		vtkPolyData *pl = vtkPolyData::SafeDownCast(subject->GetData());
		if(pl != 0)
		{
			this->WriteToLog("Number of particles: "+iString::FromNumber(pl->GetNumberOfPoints()));
		}

		vtkPointData *pd = subject->GetData()->GetPointData();
		if(pd!=0 && pd->GetScalars() != 0)
		{
			this->WriteToLog("Number of scalar components: "+iString::FromNumber(pd->GetScalars()->GetNumberOfComponents()));
		}
		if(pd!=0 && pd->GetVectors() != 0)
		{
			this->WriteToLog("Number of vector components: "+iString::FromNumber(pd->GetVectors()->GetNumberOfComponents()));
		}
		if(pd!=0 && pd->GetTensors() != 0)
		{
			this->WriteToLog("Number of tensor components: "+iString::FromNumber(pd->GetTensors()->GetNumberOfComponents()));
		}

		if(details)
		{
			iDataExplorer *de = iDataExplorer::New(this->GetShell()->GetControlModule()->GetViewModule()); IERROR_ASSERT(de);
			de->SetActiveDataType(subject->GetDataType());
			for(i=0; i<de->GetNumberOfInputComponents(); i++)
			{
				de->SetInputComponent(i);
				iDataExplorer::Info di = de->GetInfo(true);
				this->WriteToLog(subject->GetLimits()->GetName(i)+": from "+iString::FromNumber(di.Minimum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Minimum)))+" dex) to "+iString::FromNumber(di.Maximum)+" ("+iString::FromNumber(log10(1.0e-30+fabs(di.Maximum)))+" dex)");
			}
			de->Delete();
		}
	}

	if(!work) this->WriteToLog("This file contains no data that IFrIT can use.");
}


void iggMainWindow::LogDataInfo(const iDataType &type, bool details)
{
	iDataReader *reader = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();
	iDataSubject *subject = reader->GetSubject(type);

	if(!mExtensionWindow->LogDataInfo(type,details))
	{
		this->LogDataInfo(subject->GetLoader(),details);
	}

	if(subject!=0 && subject->GetLoader()->IsBoxPeriodic())
	{
		iString ws;
		if(subject->GetLoader()->IsDirectionPeriodic(0)) ws += "X, ";
		if(subject->GetLoader()->IsDirectionPeriodic(1)) ws += "Y, ";
		if(subject->GetLoader()->IsDirectionPeriodic(2)) ws += "Z";
		if(!ws.IsEmpty()) this->WriteToLog("Data periodic in "+ws);
	}
}


//
//  **************************************************************************************
//
//      MISC FUNCTIONS
//
//  **************************************************************************************
//
iggWidgetProgressBar* iggMainWindow::GetProgressBar() const
{
	switch(mProgressBarMode)
	{
	case 1:
		{
			return mDialogLoadFile->GetProgressBar();
		}
	case 0:
	default:
		{
			return mProgressBar;
		}
	}
}


int iggMainWindow::PopupWindow(const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(mGlobalFrame,text,type,b0,b1,b2);
}


int iggMainWindow::PopupWindow(const iggFrame *parent, const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(parent,text,type,b0,b1,b2);
}

	
int iggMainWindow::PopupWindow(const iggRenderWindow *parent, const iString &text, int type, const char *b0, const char *b1, const char *b2)
{
	return mMainSubject->PopupWindow(parent,text,type,b0,b1,b2);
}


void iggMainWindow::ShowToolTips(bool s)
{
	mToolTips = s;
	mMainSubject->ShowToolTips(s);
	this->ClearCache();
}


void iggMainWindow::ShowInteractorHelp(bool s, iViewModule* /*vm*/)
{
	mBaseFrame->ShowLayer(s?1:0);
}


void iggMainWindow::Block(bool s)
{
	if(s)
	{
		if(mBlockLevel == 0)
		{
			ibgWindowSubject::Block(true);
			iDynamicCast<BusyIndicator,iggWidget>(INFO,mBusyIndicator)->SetBusy(true);
		}
		mBlockLevel++;
	}
	else
	{
		mBlockLevel--;
		if(mBlockLevel == 0)
		{
			iDynamicCast<BusyIndicator,iggWidget>(INFO,mBusyIndicator)->SetBusy(false);
			ibgWindowSubject::Block(false);
		}
		if(mDialogRenderingProgress!=0 && mDialogRenderingProgress->IsCancelled() && mDialogRenderingProgress->IsVisible())
		{
			mDialogRenderingProgress->Show(false);
		}
		if(mBlockLevel < 0)
		{
#ifdef I_CHECK1
			IERROR_HIGH("Incorrect order if MainWindow->Block() calls. This is a bug, but IFrIT may continue to work properly.");
#endif
			mBlockLevel = 0;
		}
	}
}


bool iggMainWindow::IsExitAllowed()
{
	return this->AskForConfirmation("Are you sure you want to exit IFrIT?","Exit");
}


void iggMainWindow::WriteToLog(const iString &prefix, const iString &message, const iColor &color)
{
	if(mLog != 0)
	{
		mLog->AppendTextLine(prefix,message,color);
		if(mLog->GetNumberOfLines() > 1000)
		{
			iString text = mLog->GetText();
			mLog->SetText(text.Part(text.Length()/5));
		}
	}
}


void iggMainWindow::WriteToLog(const iString &message)
{
	if(mLog != 0)
	{
		mLog->AppendTextLine("\t",message);
		if(mLog->GetNumberOfLines() > 1000)
		{
			iString text = mLog->GetText();
			mLog->SetText(text.Part(text.Length()/5));
		}
	}
}


void iggMainWindow::ClearLog()
{
	mLog->Clear();
}


void iggMainWindow::ShowLog(bool s)
{
	mLog->Show(s);
	if(s)
	{
		mGlobalFrame->SetRowStretch(2,10);
		this->ProcessEvents();
	}
	else
	{
		mGlobalFrame->SetRowStretch(2,0);
		this->ProcessEvents();

		int wg[4];
		this->GetSubject()->GetWindowGeometry(wg,true);
		wg[3] = 10;
		this->GetSubject()->SetWindowGeometry(wg,true);
	}
}


bool iggMainWindow::IsLogVisible() const
{
	return mLog->IsVisible();
}


void iggMainWindow::OpenBookPage(int n)
{
	mBook->OpenPage(n);
}


void iggMainWindow::SetTabMode(int m)
{
	switch(m)
	{
	case 0:
		{
			mBook->SetTabMode(BookTab::TitleOnly);
			break;
		}
	case 1:
		{
			mBook->SetTabMode(BookTab::ImageOnly);
			break;
		}
	case 2:
		{
			mBook->SetTabMode(BookTab::TitleAndImage);
			break;
		}
	}
	if(mExtensionWindow != 0) mExtensionWindow->SetTabMode(m);
	this->ClearCache();
}


void iggMainWindow::ShowAll(bool s)
{
	this->GetSubject()->Show(s);
	if(!mDocked)
	{
		mExtensionWindow->GetSubject()->Show(s);
		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->Show(s);
		}
	}
}


void iggMainWindow::DockWindows(bool s, bool show)
{
	if(s == mDocked) return;
	
	this->ClearCache();

	if(show) mDialogDocking->Show(true);

	this->ProcessEvents(true); // god knows why this is needed

	if(s)
	{
		mDocked = true;

		//
		//  Save all geometries
		//
		this->GetSubject()->GetHelper()->SaveWindowGeometry();
		mExtensionWindow->GetSubject()->GetHelper()->SaveWindowGeometry();

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->SaveWindowGeometry();
		}

		mGlobalFrame->SetPadding(false);
		mExtensionWindow->SetPadding(false);

		mMainSubject->PlaceWindowsInDockedPositions();
	} 
	else 
	{
		mMainSubject->RestoreWindowsFromDockedPositions();

		mGlobalFrame->SetPadding(true);
		mExtensionWindow->SetPadding(true);

		this->GetSubject()->GetHelper()->RestoreWindowGeometry();
		this->GetSubject()->Show(true);
		mExtensionWindow->GetSubject()->GetHelper()->RestoreWindowGeometry();
		mExtensionWindow->GetSubject()->GetHelper()->RestoreDecoration();
		mExtensionWindow->GetSubject()->Show(true);

		FOR_EVERY_RENDER_WINDOW(w)
		{
			w->GetSubject()->GetHelper()->RestoreWindowGeometry();
			w->GetSubject()->GetHelper()->RestoreDecoration();
			w->GetSubject()->Show(true);
		}

		mDocked = false;
	}

	this->ProcessEvents(true); // god knows why this is needed

	bool aa;
	this->GetShell()->GetControlModule()->QueryValue(iViewModule::KeyAntialiasing(),aa);
	iString ws;
	this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyAntialiasing(),!aa);
	this->GetShell()->GetControlModule()->Execute(ws,false,ObjectOption::One | ModuleOption::All | RenderOption::Auto);
	ws.Clear();
	this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyAntialiasing(),aa);
	this->GetShell()->GetControlModule()->Execute(ws,false,ObjectOption::One | ModuleOption::All | RenderOption::Auto);

	FOR_EVERY_RENDER_WINDOW(w)
	{
		w->Render();
	}

	this->ProcessEvents(true); // god knows why this is needed

	if(show) mDialogDocking->Show(false);
}


void iggMainWindow::PlaceAutoDialogs()
{
	this->ProcessEvents(true); // god knows why this is needed
	//
	//  Shift dialogs off the way
	//
	int mwg[4], fs[2], wg[4];
	this->GetSubject()->GetWindowGeometry(mwg);
	this->GetSubject()->GetFrameSize(fs[0],fs[1]);

	int i, j, tmp, n = 0, topOffset = 0;
	int *width = new int[mAutoDialogList.Size()];
	int *index = new int[mAutoDialogList.Size()];
	if(width!=0 && index!=0)
	{
		for(i=0; i<mAutoDialogList.Size(); i++) if(mAutoDialogList[i]->IsVisible())
		{
			mAutoDialogList[i]->GetSubject()->GetWindowGeometry(wg);
			index[n] = i;
			width[n] = wg[2];
			n++;
		}

		for(i=0; i<n-1; i++)
		{
			for(j=i+1; j<n; j++)
			{
				if(width[i] < width[j])
				{
					tmp = width[i];
					width[i] = width[j];
					width[j] = tmp;
					tmp = index[i];
					index[i] = index[j];
					index[j] = tmp;
				}
			}
		}

		for(i=0; i<n; i++)
		{
			this->ProcessEvents(true); // god knows why this is needed

			mAutoDialogList[index[i]]->GetSubject()->GetWindowGeometry(wg);
			wg[0] = mwg[0] + mwg[2] + fs[0];
			wg[1] = mwg[1] + topOffset;
			mAutoDialogList[index[i]]->GetSubject()->SetWindowGeometry(wg);

			this->ProcessEvents(true); // god knows why this is needed

			topOffset += (wg[3]+fs[1]);
		}

		delete [] width;
		delete [] index;
	}
}


bool iggMainWindow::AskForConfirmation(const iString &message, const char *action)
{
	return !mIdiosyncratic || this->PopupWindow(message,PopupWindow::Message,action,"Cancel")==0;
}


void iggMainWindow::UpdateOnPick()
{
	mExtensionWindow->UpdateOnPick();
}


void iggMainWindow::UpdateMarkerWidgets()
{
	if(mViewPage != 0) mViewPage->UpdateMarkerWidgets();
}


void iggMainWindow::UpdateParticleWidgets(const iImage *icon)
{
	if(icon != 0)
	{
		mBook->ChangeIcon(4,icon);
	}
	if(mParticlesPage != 0)
	{
		if(icon != 0)
		{
			mParticlesPage->GetBook()->ChangeIcon(-1,icon);
		}
//		mParticlesPage->UpdateWidget();
	}
	if(icon != 0)
	{
		this->GetSubject()->SetToolBarIcon(ToolBar::ShowParticles,*icon);
	}
	this->GetSubject()->UpdateMenus();

	mExtensionWindow->UpdateParticleWidgets(icon);
}


void iggMainWindow::AddTheme(const iString &name)
{
	mThemeList.Add(name);
}


void iggMainWindow::SetTheme(int n)
{
	if(mCurrentTheme!=n && n>=0 && n<mThemeList.Size())
	{
		mCurrentTheme = n;
		mMainSubject->SetTheme(mThemeList[n]);
		this->ClearCache();
	}
}


void iggMainWindow::ProcessEvents(bool sync) const
{
	iRequiredCast<iggShell>(INFO,this->GetShell())->ProcessEvents(sync);
}


//
//  **************************************************************************************
//
//  iObject functionality
//
//  **************************************************************************************
//
void iggMainWindow::PackStateBody(iString &s) const
{
	int wg[4];

	this->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyGeometry(),wg,4);

	mExtensionWindow->GetSubject()->GetWindowGeometry(wg);
	this->PackValue(s,KeyExtensionGeometry(),wg,4);

	this->PackValue(s,KeyDocked(),mDocked);
	this->PackValue(s,KeyInteractorHelp(),mInteractorHelp);
	this->PackValue(s,KeyToolTips(),mToolTips);
	this->PackValue(s,KeyOptionsAreGlobal(),mOptionsAreGlobal);
	this->PackValue(s,KeyWindowUnderFocusCurrent(),mWindowUnderFocusCurrent);
	this->PackValue(s,KeyAllowPrePopulateToolBar(),mAllowPrePopulateToolBar);
	this->PackValue(s,KeyIsIdiosyncratic(),mIdiosyncratic);
	this->PackValue(s,KeyTabMode(),mBook->GetTabMode());
	this->PackValue(s,KeyBookOrientation(),mBook->GetOrientation());
	this->PackValue(s,KeyTheme(),(mCurrentTheme>=0 && mCurrentTheme<mThemeList.Size())?mThemeList[mCurrentTheme]:"");
}


void iggMainWindow::UnPackStateBody(const iString &s)
{
	int i, wg[4]; bool b; iString t;

	if(this->UnPackValue(s,KeyGeometry(),wg,4))
	{
		if(mInitialized)
		{
			this->GetSubject()->SetWindowGeometry(wg);
			this->ClearCache();
		}
		else
		{
			for(i=0; i<4; i++) mInitialGeometry[i] = wg[i];
		}
	}

	if(this->UnPackValue(s,KeyExtensionGeometry(),wg,4))
	{
		mExtensionWindow->GetSubject()->SetWindowGeometry(wg);
		this->ClearCache();
	}

	if(this->UnPackValue(s,KeyDocked(),b)) this->DockWindows(b,false);
	if(this->UnPackValue(s,KeyInteractorHelp(),b))
	{
		mInteractorHelp = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyToolTips(),b)) this->ShowToolTips(b);
	if(this->UnPackValue(s,KeyOptionsAreGlobal(),b))
	{
		mOptionsAreGlobal = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyWindowUnderFocusCurrent(),b))
	{
		mWindowUnderFocusCurrent = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyAllowPrePopulateToolBar(),b))
	{
		mAllowPrePopulateToolBar = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyIsIdiosyncratic(),b))
	{
		mIdiosyncratic = b;
		this->ClearCache();
	}
	if(this->UnPackValue(s,KeyTabMode(),i))	this->SetTabMode(i);
	if(this->UnPackValue(s,KeyBookOrientation(),i)) mBook->SetOrientation(i);
	if(this->UnPackValue(s,KeyTheme(),t))
	{
		for(i=0; i<mThemeList.Size(); i++)
		{
			if(t == mThemeList[i])
			{
				this->SetTheme(i);
				break;
			}
		}
	}

	//
	//  Special "action" keys
	//
	iObject::ReportMissingKeys(false); //action keys are not part of the states
	
	if(this->UnPackValue(s,KeySlidersAreEditable(),b))
	{
		iggWidgetSlider::SetAllEditable(b);
	}

	iObject::ReportMissingKeys(true);
}


//
//  **************************************************************************************
//
//    WINDOW MANIPULATION
//
//  **************************************************************************************
//
void iggMainWindow::OnRenderWindowMove(int /*wn*/)
{
}


void iggMainWindow::OnRenderWindowResize(int /*wn*/)
{
	if(mDialogImageComposer != 0) mDialogImageComposer->UpdateView();
}


void iggMainWindow::OnRenderWindowFocusIn(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mWindowUnderFocusCurrent && wn>=0)
	{
		if(this->GetShell()->GetControlModule()->SetCurrentViewModuleIndex(wn)) this->UpdateAll();
	}

//	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetView(wn));
}


void iggMainWindow::OnRenderWindowFocusOut(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::OnRenderWindowEnter(int wn)
{
	if(wn<-1 || wn>=this->GetShell()->GetControlModule()->GetNumberOfViewModules())
	{
		return; // can happen when a docked window is deleted
	}
	
	if(mInteractorHelp) this->ShowInteractorHelp(true,this->GetShell()->GetControlModule()->GetViewModule(wn));
}


void iggMainWindow::OnRenderWindowLeave(int /*wn*/)
{
	if(mInteractorHelp) this->ShowInteractorHelp(false);
}


void iggMainWindow::DisplayWindowsAsIcons()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++)
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsIcon();
		}
	}
}


void iggMainWindow::DisplayWindowsAsWindows()
{
	if(!mDocked)
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++) 
		{
			//
			//  On multi-desktop systems this may cause weird behaviour unless all windows are on the same desktop.
			//
			mWindowList[i]->GetHelper()->ShowAsWindow();
		}
	}
}


void iggMainWindow::MoveWindows(int pos[2])
{
	//
	//  We must make this function safe against multiple concurrent calls.
	//  Save the last call.
	//
	mNextPos[0] = pos[0];
	mNextPos[1] = pos[1];

	if(mInMove) return; // ignore rapidly arriving events
	mInMove = true;

	if(mMoveTogether && mInitialized && !mDocked) // do not move the first render window
	{
		int wg[4];
		int i, dp[2];
		dp[0] = pos[0] - mPrevPos[0];
		dp[1] = pos[1] - mPrevPos[1];
		if(dp[0]!=0 || dp[1]!=0)
		{
			ibgWindowSubject::Block(true); // needed to disable rendering while moving
			for(i=0; i<mWindowList.Size(); i++) if(mWindowList[i]->IsVisible())
			{
				mWindowList[i]->GetWindowGeometry(wg);
				wg[0] += dp[0];
				wg[1] += dp[1];
				mWindowList[i]->SetWindowGeometry(wg);
			}
			ibgWindowSubject::Block(false);
			this->ProcessEvents(true);
		}
	}

	mPrevPos[0] = pos[0];
	mPrevPos[1] = pos[1];

	mInMove = false;

	//
	//  Check that this function was not called while being executed.
	//
	if(mNextPos[0]!=pos[0] || mNextPos[1]!=pos[1])
	{
		this->MoveWindows(mNextPos);
	}
}


void iggMainWindow::Exit()
{
	if(this->IsExitAllowed())
	{
		int i;
		for(i=0; i<mWindowList.Size(); i++) mWindowList[i]->Show(false);
		if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(false);
		this->GetShell()->Exit();
	}
}



//
//  **************************************************************************************
//
//      MENU INTERACTION
//
//  **************************************************************************************
//
void iggMainWindow::BuildMenus()
{
	//
	//  1. File menu
	//
	this->GetSubject()->BeginMenu("&File",false);
	{
		this->GetSubject()->AddMenuItem(Menu::File::OpenUniformScalars,"Open Uniform&Scalars data file",iImageFactory::FindIcon("fileopenmesh.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(Menu::File::OpenBasicParticles,"Open Basic&Particles data file",iImageFactory::FindIcon("fileopenpart.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(Menu::File::OpenUniformVectors,"Open Uniform&Vectors data file",iImageFactory::FindIcon("fileopenvect.png"),"",false,false,&iViewModule::KeyNoClone());
		this->GetSubject()->AddMenuItem(Menu::File::OpenUniformTensors,"Open Uniform&Tensors data file",iImageFactory::FindIcon("fileopentens.png"),"",false,false,&iViewModule::KeyNoClone());

		mExtensionWindow->PopulateFileMenu();


		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(Menu::File::SaveState,"Save state",0,"",false,false);
		this->GetSubject()->AddMenuItem(Menu::File::SaveStateAs,"Save state as...",0,"",false,false);
		this->GetSubject()->AddMenuItem(Menu::File::LoadStateFrom,"Load state from...",0,"",false,false);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->AddMenuItem(Menu::File::Exit,"&Exit",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  2. Dialog menu
	//
	this->GetSubject()->BeginMenu("&Tools",false);
	{
		if(mDialogScriptDebugger != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::ScriptDebugger,"&Animation Script Debugger",iImageFactory::FindIcon("debug.png"),"Ctrl+A",false,false);
		if(mDialogPaletteEditor != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::PaletteEditor,"Palette &Editor",iImageFactory::FindIcon("paled.png"),"Ctrl+E",false,false);
		if(mDialogPickerWindow != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::PickerWindow,"&Picker Window",iImageFactory::FindIcon("picks.png"),"Ctrl+P",false,false);
		if(mDialogFileSetExplorer != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::FileSetExplorer,"&File Set Explorer",iImageFactory::FindIcon("setexp.png"),"Ctrl+F",false,false,&iDataReader::KeyIsSet());
		mExtensionWindow->PopulateLocalDialogMenu();
	
		this->GetSubject()->BeginMenu("Automatic Dialogs",false);
		{
			this->GetSubject()->AddMenuItem(Menu::Dialog::AutoBlock,"Do not launch automatically",0,"",true,iggDialogAuto::mBlockShowing);
			int i;
			iString ws;
			for(i=0; i<mAutoDialogList.Size() && i<Menu::Dialog::AutoEnd-Menu::Dialog::AutoBegin; i++)
			{
				this->GetSubject()->AddMenuItem(Menu::Dialog::AutoBegin+i,"  "+mAutoDialogList[i]->GetTitle(),0,"",false,false);
			}
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuSeparator();
		if(mDialogPerformanceMeter != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::PerformanceMeter,"Performance &Meter",iImageFactory::FindIcon("perf.png"),"Ctrl+M",false,false);
		if(mDialogParallelController!=0 && this->GetSubject()->GetShell()->GetControlModule()->GetParallelManager()->GetMaxNumberOfProcessors()>1) this->GetSubject()->AddMenuItem(Menu::Dialog::ParallelController,"Parallel &Controller",iImageFactory::FindIcon("parallel.png"),"Ctrl+C",false,false);
		if(mDialogDataExplorer != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::DataExplorer,"&Data Explorer",iImageFactory::FindIcon("dataexp.png"),"Ctrl+D",false,false);
		if(mDialogImageComposer != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::ImageComposer,"&Image Composer",iImageFactory::FindIcon("imcomp.png"),"Ctrl+I",false,false);
		if(mDialogCommandLine != 0) this->GetSubject()->AddMenuItem(Menu::Dialog::CommandLine,"Command I&nterpreter",iImageFactory::FindIcon("comline.png"),"Ctrl+N",false,false);
		mExtensionWindow->PopulateGlobalDialogMenu();
	}
	this->GetSubject()->EndMenu();
	//
	//  3. Style menu
	//
	this->GetSubject()->BeginMenu("&Style",false);
	{
		this->GetSubject()->AddMenuItem(Menu::Style::InteractorHelp,"Show &interactor help",0,"",true,mInteractorHelp);
		this->GetSubject()->AddMenuItem(Menu::Style::ToolTips,"Show t&ooltips",0,"",true,mToolTips);

		this->GetSubject()->AddMenuSeparator();

		this->GetSubject()->BeginMenu("&Tab style",true);
		{
			this->GetSubject()->AddMenuItem(Menu::Style::TabText,"&Text only",0,"",true,mBook->GetTabMode()==BookTab::TitleOnly);
			this->GetSubject()->AddMenuItem(Menu::Style::TabIcon,"&Icon only",0,"",true,mBook->GetTabMode()==BookTab::ImageOnly);
			this->GetSubject()->AddMenuItem(Menu::Style::TabBoth,"Icon &and text",0,"",true,mBook->GetTabMode()==BookTab::TitleAndImage);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Slider render style",true);
		{
			this->GetSubject()->AddMenuItem(Menu::Style::RenderImmediate,"&Immediate render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==RenderMode::Immediate);
			this->GetSubject()->AddMenuItem(Menu::Style::RenderDelayed,"&Delayed render",0,"",true,iggWidgetKeyHandlerBase::GetGlobalRenderMode()==RenderMode::Delayed);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(Menu::Style::RenderResetAll,"&Reset all sliders to global mode",0,"",false,false);

		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->BeginMenu("&Advanced",false);
		{
			this->GetSubject()->AddMenuItem(Menu::Style::DockWindow,"&Dock windows",0,"",true,mDocked);
			this->GetSubject()->AddMenuItem(Menu::Style::SlidersEditable,"&Sliders are editable",0,"",true,iggWidgetSlider::IsEditableByDefault());
			this->GetSubject()->AddMenuItem(Menu::Style::IsIdiosyncratic,"&Ask for confirmations",0,"",true,mIdiosyncratic);
			this->GetSubject()->AddMenuItem(Menu::Style::DetailLoadedDataInfo,"&Print detail info for loaded data",0,"",true,mDetailLoadedDataInfo);
			this->GetSubject()->AddMenuItem(Menu::Style::AutoRender,"&Render automatically",0,"",true,mAutoRender);
		}
		this->GetSubject()->EndMenu();
		this->GetSubject()->BeginMenu("The&me",true);
		{
			int i;
			for(i=0; i<mThemeList.Size(); i++) this->GetSubject()->AddMenuItem(Menu::Style::Theme+i,mThemeList[i],0,"",true,false);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(Menu::Style::RotateBook,"&Rotate the book widget",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	//
	//  4. Options menu
	//
	this->GetSubject()->BeginMenu("&Options",false);
	{
		this->GetSubject()->AddMenuItem(Menu::Option::Antialiasing,"Use &antialiasing",0,"Shift+Ctrl+Z",true,false,0,false,&iViewModule::KeyAntialiasing());

		this->GetSubject()->BeginMenu("&Bounding box style",true);
		{
			this->GetSubject()->AddMenuItem(Menu::Option::IfritBox,"&Ifrit style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),BoundingBoxType::Default);
			this->GetSubject()->AddMenuItem(Menu::Option::ClassicBox,"&Classic style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),BoundingBoxType::Classic);
			this->GetSubject()->AddMenuItem(Menu::Option::HairBox,"&Hair-thin style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),BoundingBoxType::HairThin);
			this->GetSubject()->AddMenuItem(Menu::Option::AxesBox,"&Axes style",0,"",true,false,0,false,&iViewModule::KeyBoundingBoxType(),BoundingBoxType::Axes);
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Fonts",false);
		{
			this->GetSubject()->BeginMenu("Font &type",true);
			{
				this->GetSubject()->AddMenuItem(Menu::Option::FontTypeArial,"&Arial",0,"",true,false,0,false,&iViewModule::KeyFontType(),TextType::Arial);
				this->GetSubject()->AddMenuItem(Menu::Option::FontTypeCourier,"&Courier",0,"",true,false,0,false,&iViewModule::KeyFontType(),TextType::Courier);
				this->GetSubject()->AddMenuItem(Menu::Option::FontTypeTimes,"&Times",0,"",true,false,0,false,&iViewModule::KeyFontType(),TextType::Times);
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Font &size",true);
			{
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizem5," 20%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-5);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizem4," 30%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-4);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizem3," 40%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-3);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizem2," 60%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-2);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizem1," 75%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),-1);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizec0,"100%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),0);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizep1,"130%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+1);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizep2,"180%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+2);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizep3,"250%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+3);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizep4,"330%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+4);
				this->GetSubject()->AddMenuItem(Menu::Option::FontSizep5,"450%",0,"",true,false,0,false,&iViewModule::KeyFontScale(),+5);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->BeginMenu("&Images",false);
		{
			this->GetSubject()->BeginMenu("Image &format",true);
			{
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatPNG,"PNG (&Portable Network Graphics)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::PNG);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatJPG,"JPG (&Joint Photographic Experts Group)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::JPG);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatPPM,"PPM (Portable Pi&xmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::PNM);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatBMP,"BMP (Window &Bitmap)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::BMP);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatTIF,"TIF (&Tag Image File Format)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::TIF);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageFormatEPS,"EPS (&Encapsulated PostScript)",0,"",true,false,0,false,&iViewModule::KeyImageFormat(),ImageFormat::EPS);

				this->GetSubject()->BeginMenu("  Postscript paper format",true);
				{
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA0,"A0",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),0);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA1,"A1",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),1);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA2,"A2",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),2);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA3,"A3",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),3);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA4,"A4",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),4);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA5,"A5",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),5);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA6,"A6",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),6);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA7,"A7",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),7);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatA8,"A8",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),8);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatL1,"Letter",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),9);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptPaperFormatL2,"10x17",0,"",true,false,0,false,&iViewModule::KeyPostScriptPaperFormat(),10);
				}
				this->GetSubject()->EndMenu();

				this->GetSubject()->BeginMenu("  Postscript orientation",true);
				{
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptOrientationPortrait,"Portrait",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),0);
					this->GetSubject()->AddMenuItem(Menu::Option::PostScriptOrientationLandscape,"Landscape",0,"",true,false,0,false,&iViewModule::KeyPostScriptOrientation(),1);
				}
				this->GetSubject()->EndMenu();
			}
			this->GetSubject()->EndMenu();

			this->GetSubject()->BeginMenu("Image &zoom",true);
			{
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom001,"x 1",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),1);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom002,"x 2",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),2);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom003,"x 3",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),3);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom004,"x 4",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),4);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom005,"x 5",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),5);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom006,"x 6",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),6);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom008,"x 8",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),8);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom010,"x 10",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),10);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom015,"x 15",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),15);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom020,"x 20",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),20);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom030,"x 30",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),30);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom040,"x 40",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),40);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom050,"x 50",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),50);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom060,"x 60",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),60);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom080,"x 80",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),80);
				this->GetSubject()->AddMenuItem(Menu::Option::ImageZoom100,"x 100",0,"",true,false,0,false,&iViewModule::KeyImageMagnification(),100);
			}
			this->GetSubject()->EndMenu();
		}
		this->GetSubject()->EndMenu();

		this->GetSubject()->AddMenuItem(Menu::Option::ArtifactCorrection,"&Correct visual artifacts (may be slow)",0,"",true,true,0,false,&iViewModule::KeyArtifactCorrection());
		this->GetSubject()->AddMenuItem(Menu::Option::BackgroundImageFixedAspect,"&Preserve aspect ratio of background images",0,"",true,true,0,false,&iViewModule::KeyBackgroundImageFixedAspect());
		
		this->GetSubject()->AddMenuItem(Menu::Option::SettingsGlobal,"Options apply to all windows",0,"",true,mOptionsAreGlobal);
	}
	this->GetSubject()->EndMenu();
	//
	//  5. Help menu
	//
	this->GetSubject()->BeginMenu("&Help",false);
	{
		this->GetSubject()->AddMenuItem(Menu::Help::Contents,"Contents",0,"F1",false,false);
#if defined(I_DEBUG)
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(Menu::Help::Max-1,"Check Blocking",0,"",false,false);
		this->GetSubject()->AddMenuItem(Menu::Help::Max,"Debug Helper",0,"",false,false);
#endif
		this->GetSubject()->AddMenuSeparator();
		this->GetSubject()->AddMenuItem(Menu::Help::About,"About IFrIT",0,"",false,false);
	}
	this->GetSubject()->EndMenu();
	this->GetSubject()->CompleteMenu();

	//
	//  Build a tool bar
	//
	if(mAllowPrePopulateToolBar)
	{
		mExtensionWindow->PrePopulateFileToolBar();
		this->GetSubject()->AddToolBarSeparator();
	}
	this->GetSubject()->AddToolBarButton(Menu::File::OpenUniformScalars,"Open UniformScalars data file");
	this->GetSubject()->AddToolBarButton(Menu::File::OpenBasicParticles,"Open BasicParticles data file");
	this->GetSubject()->AddToolBarButton(Menu::File::OpenUniformVectors,"Open UniformVectors data file");
	this->GetSubject()->AddToolBarButton(Menu::File::OpenUniformTensors,"Open UniformTensors data file");
	mExtensionWindow->PopulateFileToolBar(mAllowPrePopulateToolBar);
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(ToolBar::ShowSurface,"Show surface",iImageFactory::FindIcon("surf.png"),true,&iSurfaceViewSubject::KeyReady(),false,&iSurfaceViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(ToolBar::ShowCrossSection,"Show cross section",iImageFactory::FindIcon("xsec.png"),true,&iCrossSectionViewSubject::KeyReady(),false,&iCrossSectionViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(ToolBar::ShowVolume,"Show volume",iImageFactory::FindIcon("volv.png"),true,&iVolumeViewSubject::KeyReady(),false,&iVolumeViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(ToolBar::ShowParticles,"Show particles",iImageFactory::FindIcon("part.png"),true,&iParticlesViewSubject::KeyReady(),false,&iParticlesViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(ToolBar::ShowVectorField,"Show vector field",iImageFactory::FindIcon("vect.png"),true,&iVectorFieldViewSubject::KeyReady(),false,&iVectorFieldViewSubject::KeyVisible());
	this->GetSubject()->AddToolBarButton(ToolBar::ShowTensorField,"Show tensor field",iImageFactory::FindIcon("tens.png"),true,&iTensorFieldViewSubject::KeyReady(),false,&iTensorFieldViewSubject::KeyVisible());
	mExtensionWindow->PopulateShowToolBar();
	this->GetSubject()->AddToolBarSeparator();
	this->GetSubject()->AddToolBarButton(ToolBar::OpenWindowsPage,"Switch to page with multiple windows controls",iImageFactory::FindIcon("wins.png"),false);
	this->GetSubject()->AddToolBarButton(ToolBar::MinimizeWindows,"Minimize all windows",iImageFactory::FindIcon("minimize.png"),false);
	this->GetSubject()->AddToolBarButton(ToolBar::MaximizeWindows,"Restore all windows to normal size",iImageFactory::FindIcon("maximize.png"),false);
	this->GetSubject()->AddToolBarButton(ToolBar::MoveAllTogether,"Move all windows together",iImageFactory::FindIcon("together.png"),true);
}


void iggMainWindow::OnMenuBody(int id, bool on)
{
	iString fname;

	if(id < 0)
	{
		IERROR_LOW("Invalid menu item id.");
	}

	if(id <= Menu::File::Max)
	{
		switch(id)
		{
		case Menu::File::OpenUniformScalars:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformScalarsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformScalarsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Scalar field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformScalarsDataSubject::DataType(),fname);
				break;
			}
		case Menu::File::OpenUniformVectors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformVectorsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformVectorsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Vector field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformVectorsDataSubject::DataType(),fname);
				break;
			}
		case Menu::File::OpenUniformTensors:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iUniformTensorsDataSubject::DataType());
				if(ws.IsEmpty()) ws = iUniformTensorsDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Tensor field files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iUniformTensorsDataSubject::DataType(),fname);
				break;
			}
		case Menu::File::OpenBasicParticles:
			{
				iString ws = this->GetShell()->GetControlModule()->GetViewModule()->GetReader()->GetLastFileName(iBasicParticlesDataSubject::DataType());
				if(ws.IsEmpty()) ws = iBasicParticlesDataSubject::DataType().GetEnvironment(this->GetShell());
				fname = this->GetFileName("Choose a file",ws,"Particle set files (*.bin *.txt)");
				if(!fname.IsEmpty()) this->LoadData(iBasicParticlesDataSubject::DataType(),fname);
				break;
			}
		case Menu::File::SaveState:
			{
				if(mStateFileName.IsEmpty()) mStateFileName = this->GetFileName("Choose a file",this->GetShell()->GetEnvironment(Environment::Base),"IFrIT state files (*.ini)",false);
				if(!mStateFileName.IsEmpty())
				{
					bool ok = true;
					if(iFile::IsReadable(mStateFileName))
					{
						if(this->PopupWindow("Do you want to overwrite the existing file?",iParameter::PopupWindow::Warning,"Ok","Cancel") == 1) ok = false;
					}
					if(ok)
					{
						if(!this->GetShell()->GetControlModule()->SaveStateToFile(mStateFileName)) this->PopupWindow("Saving options failed for unknown reason.\n The current state will not be saved",PopupWindow::Error);
					}
				}
				return;
			}
		case Menu::File::SaveStateAs:
			{
				fname = this->GetFileName("Choose a file",mStateFileName,"IFrIT state files (*.ini)",false);
				if(!fname.IsEmpty())
				{
					mStateFileName = fname;
					if(!this->GetShell()->GetControlModule()->SaveStateToFile(mStateFileName)) this->PopupWindow("Saving options failed for unknown reason.\n The current state will not be saved.",PopupWindow::Error);
				}
				return;
			}
		case Menu::File::LoadStateFrom:
			{
				fname = this->GetFileName("Choose a file",mStateFileName,"IFrIT state files (*.ini)");
				if(!fname.IsEmpty())
				{
					if(this->GetShell()->GetControlModule()->LoadStateFromFile(fname))
					{
						this->UpdateAll();
						this->GetShell()->GetControlModule()->Render(RenderOption::All);
					}
					else
					{
						this->PopupWindow("Loading options failed for unknown reason.",PopupWindow::Error);
					}
				}
				return;
			}
		case Menu::File::Exit:
			{
				if(this->IsExitAllowed()) this->GetShell()->Exit();
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= Menu::Dialog::Max)
	{
		switch(id)
		{
		case Menu::Dialog::ScriptDebugger:
			{
				if(mDialogScriptDebugger != 0) mDialogScriptDebugger->Show(true);
				break;
			}
		case Menu::Dialog::PaletteEditor:
			{
				if(mDialogPaletteEditor != 0) mDialogPaletteEditor->Show(true);
				break;
			}
		case Menu::Dialog::PickerWindow:
			{
				if(mDialogPickerWindow != 0) mDialogPickerWindow->Show(true);
				break;
			}
		case Menu::Dialog::ParallelController:
			{
				if(mDialogParallelController != 0) mDialogParallelController->Show(true);
				break;
			}
		case Menu::Dialog::PerformanceMeter:
			{
				if(mDialogPerformanceMeter != 0) mDialogPerformanceMeter->Show(true);
				break;
			}
		case Menu::Dialog::FileSetExplorer:
			{
				if(mDialogFileSetExplorer != 0) mDialogFileSetExplorer->Show(true);
				break;
			}
		case Menu::Dialog::DataExplorer:
			{
				if(mDialogDataExplorer != 0) mDialogDataExplorer->Show(true);
				break;
			}
		case Menu::Dialog::ImageComposer:
			{
				if(mDialogImageComposer != 0) mDialogImageComposer->Show(true);
				break;
			}
		case Menu::Dialog::CommandLine:
			{
				if(mDialogCommandLine != 0) mDialogCommandLine->Show(true);
				break;
			}
		case Menu::Dialog::AutoBlock:
			{
				iggDialogAuto::mBlockShowing = on;
				break;
			}
		default:
			{
				if(id>=Menu::Dialog::AutoBegin && id<Menu::Dialog::AutoBegin+mAutoDialogList.Size())
				{
					mAutoDialogList[id-Menu::Dialog::AutoBegin]->ForceShow(on);
				}
				else IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= Menu::Style::Max)
	{
		switch(id)
		{
		case Menu::Style::DockWindow:
			{
				this->DockWindows(on,true);
				break;
			}
		case Menu::Style::InteractorHelp:
			{
				mInteractorHelp = on;
				break;
			}
		case Menu::Style::ToolTips:
			{
				this->ShowToolTips(on);
				break;
			}
		case Menu::Style::TabText:
			{
				if(on) this->SetTabMode(BookTab::TitleOnly);
				break;
			}
		case Menu::Style::TabIcon:
			{
				if(on) this->SetTabMode(BookTab::ImageOnly);
				break;
			}
		case Menu::Style::TabBoth:
			{
				if(on) this->SetTabMode(BookTab::TitleAndImage);
				break;
			}
		case Menu::Style::RenderImmediate:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(RenderMode::Immediate);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case Menu::Style::RenderDelayed:
			{
				if(on)
				{
					iggWidgetKeyHandlerBase::SetGlobalRenderMode(RenderMode::Delayed);
					iggWidgetRenderModeButton::UpdateAll();
				}
				break;
			}
		case Menu::Style::RenderResetAll:
			{
				iggWidgetRenderModeButton::ResetAllToGlobal();
				break;
			}
		case Menu::Style::SlidersEditable:
			{
				iggWidgetSlider::SetAllEditable(on);
				break;
			}
		case Menu::Style::IsIdiosyncratic:
			{
				mIdiosyncratic = on;
				break;
			}
		case Menu::Style::DetailLoadedDataInfo:
			{
				mDetailLoadedDataInfo = on;
				break;
			}
		case Menu::Style::AutoRender:
			{
				mAutoRender = on;
				mRenderButton->Show(!on);
				break;
			}
		case Menu::Style::RotateBook:
			{
				mBook->SetOrientation(mBook->GetOrientation()+1);
				break;
			}
		default:
			{
				if(id>=Menu::Style::Theme && id<Menu::Style::Theme+mThemeList.Size())
				{
					if(on) this->SetTheme(id-Menu::Style::Theme);
				}
				else IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= Menu::Option::Max)
	{
		iString ws;

		int option = ObjectOption::One | RenderOption::Auto;
		if(mOptionsAreGlobal)
		{
			option |= ModuleOption::All;
		}
		else
		{
			option |= ModuleOption::One;
		}

		switch(id)
		{
		case Menu::Option::Antialiasing:
			{
				this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyAntialiasing(),on);
				break;
			}
		case Menu::Option::IfritBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),BoundingBoxType::Default);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case Menu::Option::ClassicBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),BoundingBoxType::Classic);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case Menu::Option::HairBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),BoundingBoxType::HairThin);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(false);
				}
				break;
			}
		case Menu::Option::AxesBox:
			{
				if(on)
				{
					this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBoundingBoxType(),BoundingBoxType::Axes);
					if(mDialogAxesLabels != 0) mDialogAxesLabels->Show(true);
				}
				break;
			}
		case Menu::Option::FontTypeArial:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),TextType::Arial);
				break;
			}
		case Menu::Option::FontTypeCourier:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),TextType::Courier);
				break;
			}
		case Menu::Option::FontTypeTimes:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontType(),TextType::Times);
				break;
			}
		case Menu::Option::FontSizem5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-5);
				break;
			}
		case Menu::Option::FontSizem4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-4);
				break;
			}
		case Menu::Option::FontSizem3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-3);
				break;
			}
		case Menu::Option::FontSizem2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-2);
				break;
			}
		case Menu::Option::FontSizem1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),-1);
				break;
			}
		case Menu::Option::FontSizec0:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),0);
				break;
			}
		case Menu::Option::FontSizep1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),1);
				break;
			}
		case Menu::Option::FontSizep2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),2);
				break;
			}
		case Menu::Option::FontSizep3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),3);
				break;
			}
		case Menu::Option::FontSizep4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),4);
				break;
			}
		case Menu::Option::FontSizep5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyFontScale(),5);
				break;
			}
		case Menu::Option::ImageFormatPNG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::PNG);
				break;
			}
		case Menu::Option::ImageFormatJPG:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::JPG);
				break;
			}
		case Menu::Option::ImageFormatPPM:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::PNM);
				break;
			}
		case Menu::Option::ImageFormatBMP:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::BMP);
				break;
			}
		case Menu::Option::ImageFormatTIF:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::TIF);
				break;
			}
		case Menu::Option::ImageFormatEPS:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageFormat(),ImageFormat::EPS);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA0:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),0);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),1);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),2);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA3:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),3);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA4:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),4);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA5:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),5);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA6:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),6);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA7:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),7);
				break;
			}
		case Menu::Option::PostScriptPaperFormatA8:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),8);
				break;
			}
		case Menu::Option::PostScriptPaperFormatL1:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),9);
				break;
			}
		case Menu::Option::PostScriptPaperFormatL2:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptPaperFormat(),10);
				break;
			}
		case Menu::Option::PostScriptOrientationPortrait:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),0);
				break;
			}
		case Menu::Option::PostScriptOrientationLandscape:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyPostScriptOrientation(),1);
				break;
			}
		case Menu::Option::ImageZoom001:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),1);
				break;
			}
		case Menu::Option::ImageZoom002:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),2);
				break;
			}
		case Menu::Option::ImageZoom003:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),3);
				break;
			}
		case Menu::Option::ImageZoom004:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),4);
				break;
			}
		case Menu::Option::ImageZoom005:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),5);
				break;
			}
		case Menu::Option::ImageZoom006:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),6);
				break;
			}
		case Menu::Option::ImageZoom008:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),8);
				break;
			}
		case Menu::Option::ImageZoom010:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),10);
				break;
			}
		case Menu::Option::ImageZoom015:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),15);
				break;
			}
		case Menu::Option::ImageZoom020:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),20);
				break;
			}
		case Menu::Option::ImageZoom030:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),30);
				break;
			}
		case Menu::Option::ImageZoom040:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),40);
				break;
			}
		case Menu::Option::ImageZoom050:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),50);
				break;
			}
		case Menu::Option::ImageZoom060:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),60);
				break;
			}
		case Menu::Option::ImageZoom080:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),80);
				break;
			}
		case Menu::Option::ImageZoom100:
			{
				if(on) this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyImageMagnification(),100);
				break;
			}
		case Menu::Option::ArtifactCorrection:
			{
				this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyArtifactCorrection(),on);
				break;
			}
		case Menu::Option::BackgroundImageFixedAspect:
			{
				this->GetShell()->GetControlModule()->PackCommand(ws,iViewModule::KeyBackgroundImageFixedAspect(),on);
				break;
			}
		case Menu::Option::SettingsGlobal:
			{
				mOptionsAreGlobal = on;
				return;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		if(!ws.IsEmpty()) this->GetShell()->GetControlModule()->Execute(ws,mAutoRender,option);
		return;
	}

	if(id <= Menu::Help::Max)
	{
		switch(id)
		{
		case Menu::Help::Contents:
			{
				if(mDialogHelp != 0) mDialogHelp->Show(true);
				break;
			}
		case Menu::Help::About:
			{
				if(mDialogAbout != 0) mDialogAbout->Show(true);
				break;
			}
#if defined(I_DEBUG)
		case Menu::Help::Max-1:
			{
				if(mBlockLevel != 1)
				{
					this->PopupWindow(mGlobalFrame,"Nonzero block level: "+iString::FromNumber(mBlockLevel-1),PopupWindow::Error);
					mBlockLevel = 1;
				}
				break;
			}
		case Menu::Help::Max:
			{
				if(mDialogDebugHelper != 0) mDialogDebugHelper->Show(true);
				break;
			}
#endif
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(id <= ToolBar::Max)
	{
		switch(id)
		{
		case ToolBar::OpenWindowsPage:
			{
				mBook->OpenPage(0);
				mViewPage->ShowPage(4);
				mViewPage->GetWindowListDialog()->Show(true);
				break;
			}
		case ToolBar::MinimizeWindows:
			{
				this->DisplayWindowsAsIcons();
				this->GetSubject()->GetHelper()->ShowAsIcon();
				break;
			}
		case ToolBar::MaximizeWindows:
			{
				this->DisplayWindowsAsWindows();
				break;
			}
		case ToolBar::MoveAllTogether:
			{
				mMoveTogether = on;
				break;
			}
		case ToolBar::ShowSurface:
			{
				this->GetShell()->GetControlModule()->Show(iSurfaceViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(1);
					mExtensionWindow->OpenBookPageByIndex(1);
					mBook->GetPage(1)->UpdateWidget();
				}
				break;
			}
		case ToolBar::ShowCrossSection:
			{
				this->GetShell()->GetControlModule()->Show(iCrossSectionViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(2);
					mExtensionWindow->OpenBookPageByIndex(2);
					mBook->GetPage(2)->UpdateWidget();
				}
				break;
			}
		case ToolBar::ShowVolume:
			{
				this->GetShell()->GetControlModule()->Show(iVolumeViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(3);
					mExtensionWindow->OpenBookPageByIndex(3);
					mBook->GetPage(3)->UpdateWidget();
				}
				break;
			}
		case ToolBar::ShowParticles:
			{
				this->GetShell()->GetControlModule()->Show(iParticlesViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(4);
					mExtensionWindow->OpenBookPageByIndex(4);
					mBook->GetPage(4)->UpdateWidget();
				}
				break;
			}
		case ToolBar::ShowVectorField:
			{
				this->GetShell()->GetControlModule()->Show(iVectorFieldViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(5);
					mExtensionWindow->OpenBookPageByIndex(5);
					mBook->GetPage(5)->UpdateWidget();
				}
				break;
			}
		case ToolBar::ShowTensorField:
			{
				this->GetShell()->GetControlModule()->Show(iTensorFieldViewSubject::Type(),on,true);
				if(on)
				{
					mBook->OpenPage(6);
					mExtensionWindow->OpenBookPageByIndex(6);
					mBook->GetPage(6)->UpdateWidget();
				}
				break;
			}
		default:
			{
				IERROR_LOW("Invalid menu item id.");
			}
		}
		return;
	}

	if(!mExtensionWindow->OnMenuBody(id,on))
	{
		IERROR_LOW("Invalid menu item id.");
	}
}

#endif
