/*
 * File    : LibraryManager.java
 * Created : 04-jun-2002 11:13
 * By      : fbusquets
 *
 * JClic - Authoring and playing system for educational activities
 *
 * Copyright (C) 2000 - 2005 Francesc Busquets & Departament
 * d'Educacio de la Generalitat de Catalunya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details (see the LICENSE file).
 */

package edu.xtec.jclic.project;

import edu.xtec.jclic.*;
import edu.xtec.util.ResourceManager;
import edu.xtec.util.JDomUtility;
import edu.xtec.jclic.fileSystem.FileSystem;
import edu.xtec.jclic.misc.Utils;

import java.io.File;
import java.util.Iterator;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.awt.Frame;
import java.awt.BorderLayout;
import java.awt.Component;
import java.io.FileOutputStream;
import edu.xtec.jclic.activities.panels.Menu;
import edu.xtec.util.Options;
import edu.xtec.util.Domable;
import edu.xtec.util.Messages;
import edu.xtec.util.ResourceBridge;


/**
 *
 * @author Francesc Busquets (fbusquets@xtec.net)
 * @version 1.0
 */
public class LibraryManager implements Domable {
    
    protected DefaultListModel libraries;
    protected PlayerSettings settings;
    protected boolean modified;
    public boolean autoRun;
    
    public static final String AUTO_RUN="autoRun";
    public static final String ELEMENT_NAME="libraryManager";
    private static final String MSG_ID="libraryManager_";

    /** Creates new LibraryManager */
    public LibraryManager(PlayerSettings settings) {
        this.settings=settings;
        libraries=new DefaultListModel();
        modified=false;
        autoRun=true;
    }
    
    public static LibraryManager getLibraryManager(PlayerSettings st, org.jdom.Element e) throws Exception{
        LibraryManager lm=new LibraryManager(st);
        lm.setProperties(e, null);
        return lm;
    }
    
    public void setProperties(org.jdom.Element e, Object aux) throws Exception{
        JDomUtility.checkName(e, ELEMENT_NAME);
        autoRun=JDomUtility.getBoolAttr(e, AUTO_RUN, autoRun);
        Iterator it=e.getChildren(LibraryManagerElement.ELEMENT_NAME).iterator();
        while(it.hasNext()){            
            libraries.addElement(LibraryManagerElement.getLibraryManagerElement((org.jdom.Element)it.next(), settings.rb.getOptions()));
        }         
    }
    
    public void addNewLibrary(String path, String title) throws Exception{
        File f=new File(path);
        if(f.exists()){
            ProjectLibrary pl=ProjectLibrary.loadProjectLibrary(path, settings.rb);
            title=pl.settings.title;
        }
        else{
            if(title==null){
                title=settings.rb.getMsg("libraryManager_mainLibraryTitle");
            }
            ProjectLibrary pl=ProjectLibrary.createNewProjectLibrary(settings.rb, settings.fileSystem);
            pl.settings.title=title;
            pl.save(path);
        }
        libraries.addElement(new LibraryManagerElement(title, path, settings.rb.getOptions()));
    }
    
    public org.jdom.Element getJDomElement(){
        org.jdom.Element e=new org.jdom.Element(ELEMENT_NAME);
        e.setAttribute(AUTO_RUN, JDomUtility.boolString(autoRun));
        for(int i=0; i<libraries.size(); i++){
            LibraryManagerElement lme=(LibraryManagerElement)libraries.get(i);
            e.addContent(lme.getJDomElement());
        }
        return e;
    }
    
    public boolean isEmpty(){
        return libraries.isEmpty();
    }
    
    public ProjectLibrary getAutoStartProjectLibrary() throws Exception{
        ProjectLibrary result=null;
        if(autoRun && !isEmpty()){
            if(libraries.size()>1)
                result=selectProjectLibrary(false, false);
            else{
                result=ProjectLibrary.loadProjectLibrary(((LibraryManagerElement)libraries.get(0)).path, settings.rb);
                if(result!=null){
                    Menu m=result.getRootMenu();
                    if(m==null || m.getMenuElementCount()<1)
                        result=null;
                }
            }
        }
        return result;        
    }
    
    public ProjectLibrary selectProjectLibrary(boolean allowEdit, boolean selectOnlyEditable) throws Exception{
        ProjectLibrary result=null;
        if(!libraries.isEmpty() || allowEdit){
            LibraryDialog plDlg=new LibraryDialog(allowEdit, selectOnlyEditable);
            plDlg.setVisible(true);
            if(plDlg.result!=null){
                result=ProjectLibrary.loadProjectLibrary(plDlg.result.path, settings.rb);
            }
        }
        return result;
    }
    
    public int getNumLibraries(){
        return libraries.getSize();
    }
    
    public LibraryManagerElement locateNewProjectLibrary(String path){
        
        LibraryManagerElement result=null;
        ProjectLibrary pl=null;
        ResourceBridge rb=settings.rb;
        
        if(path==null){
            int[] filters={Utils.JCLIC_FF};
            String s=settings.fileSystem.chooseFile(settings.rootPath+File.separator+".", false, filters, rb.getOptions(), null, rb.getComponent(), false);
            if(s!=null)
                path=settings.fileSystem.getFullFileNamePath(s);
        }
        
        if(path!=null && !elementExists(path, true)){
            try{
                pl=ProjectLibrary.loadProjectLibrary(path, rb);
                result=new LibraryManagerElement(pl.settings.title, path, rb.getOptions());            
            } catch(Exception ex){
                settings.rb.getOptions().getMessages().showErrorWarning(rb.getComponent(), "FILE_ERR_READING", path, ex, null);
            }
        }        
        return result;        
    }
    
    public LibraryManagerElement createNewProjectLibrary(String path, String name){
        
        LibraryManagerElement result=null;
        ResourceBridge rb=settings.rb;
        
        if(name==null)
            name=rb.getMsg("library_newLibraryName");
        
        if(path==null){
            int[] filters={Utils.JCLIC_FF};
            String s=settings.fileSystem.chooseFile(settings.rootPath+File.separator+"library.jclic", true, filters, rb.getOptions(), null, rb.getComponent(), false);
            if(s!=null)
                path=settings.fileSystem.getFullFileNamePath(s);
        }
        
        if(path!=null && !elementExists(path, true)){
            ProjectLibrary pl=ProjectLibrary.createNewProjectLibrary(rb, settings.fileSystem);
            try{
                pl.save(path);
                result=new LibraryManagerElement(pl.settings.title, path, rb.getOptions());
            } catch(Exception ex){
                settings.rb.getOptions().getMessages().showErrorWarning(rb.getComponent(), "FILE_ERR_SAVING", path, ex, null);
            }
        }        
        return result;        
    }
    
    public boolean elementExists(String path, boolean warn){
        boolean result=false;
        for(int i=0; i<libraries.size(); i++){
            LibraryManagerElement lme=(LibraryManagerElement)libraries.get(i);
            if(path.equals(settings.fileSystem.getFullFileNamePath(lme.path))){
                result=true;
                if(warn)
                    settings.rb.getOptions().getMessages().showAlert(settings.rb.getComponent(), "libraryManager_new_exists");
                break;
            }
        }
        return result;
    }
    
    public LibraryPane getLibraryPane(boolean allowEdit, boolean selectOnlyEditable){
        return new LibraryPane(allowEdit, selectOnlyEditable);
    }
    
    public class LibraryPane extends JPanel implements ListSelectionListener{
        Action editAction, newLibraryAction, deleteAction;
        JList list;
        boolean allowEdit, onlyEditable;
        LibraryManagerElement current=null;
        
        LibraryPane(boolean allowEdit, boolean selectOnlyEditable){
            super(new BorderLayout());
            this.allowEdit=allowEdit;
            onlyEditable=selectOnlyEditable;
            buildActions();
            list=new JList(libraries);            
            list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            list.addListSelectionListener(this);
            if(list.getModel().getSize()>0)
                list.setSelectedIndex(0);
            list.setCellRenderer(new DefaultListCellRenderer(){
                public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus){
                    super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
                    if(value instanceof LibraryManagerElement){
                        LibraryManagerElement lme=(LibraryManagerElement)value;
                        setIcon(lme.getIcon());
                        if(!lme.exists || (!lme.editable && onlyEditable))
                            setForeground(java.awt.SystemColor.textInactiveText);                            
                    }
                    return this;
                }
            });
            add(new JScrollPane(list), BorderLayout.CENTER);
            if(allowEdit){
                JPanel buttonsPanel=new JPanel();
                buttonsPanel.setOpaque(false);
                buttonsPanel.add(new JButton(editAction));
                buttonsPanel.add(new JButton(newLibraryAction));
                buttonsPanel.add(new JButton(deleteAction));
                add(buttonsPanel, BorderLayout.SOUTH);                
            }
        }
        
        void buildActions(){
            final edu.xtec.util.Options options=settings.rb.getOptions();                        
            editAction=new AbstractAction(
                options.getMsg(MSG_ID+"edit_caption"),
                ResourceManager.getImageIcon("icons/edit.gif")
                ){
                public void actionPerformed(ActionEvent ev){
                    LibraryManagerElement lme=(LibraryManagerElement)list.getSelectedValue();
                    if(lme!=null){
                        try{                            
                            if(!settings.promptPassword(LibraryPane.this, null))
                                return;                            
                            ProjectLibrary pl=ProjectLibrary.loadProjectLibrary(lme.path, settings.rb);
                            if(pl.editProjectLibrary(LibraryPane.this)){
                                if(!pl.settings.title.equals(lme.name)){
                                    lme.name=pl.settings.title;
                                    modified=true;
                                    list.repaint();
                                }                                
                            }
                        } catch(Exception ex){
                            System.err.println("Unable edit projectLibrary:\n"+ex);
                            return;
                        }
                    }
                }
            };
            editAction.putValue(AbstractAction.SHORT_DESCRIPTION, options.getMsg(MSG_ID+"edit_tooltip"));
            editAction.setEnabled(false);
            
            deleteAction=new AbstractAction(
                options.getMsg(MSG_ID+"delete_caption"),
                ResourceManager.getImageIcon("icons/delete.gif")
                ){
                public void actionPerformed(ActionEvent ev){                    
                    LibraryManagerElement lme=(LibraryManagerElement)list.getSelectedValue();
                    if(lme!=null){
                        int currentIndex=list.getSelectedIndex();
                        try{                            
                            if(!settings.promptPassword(LibraryPane.this, null)
                            || !(options.getMessages().showQuestionDlg(null, MSG_ID+"delete_confirm", null, "yn")==Messages.YES))
                                return;                            
                            libraries.removeElement(lme);                            
                            modified=true;
                            list.setSelectedIndex(Math.max(0, currentIndex-1));
                        } catch(Exception ex){
                            System.err.println("Unable to delete projectLibrary:\n"+ex);
                            return;
                        }
                    }
                }
            };
            deleteAction.putValue(AbstractAction.SHORT_DESCRIPTION, options.getMsg(MSG_ID+"delete_tooltip"));
            deleteAction.setEnabled(false);
            
            newLibraryAction=new AbstractAction(
                options.getMsg(MSG_ID+"new_caption"),
                ResourceManager.getImageIcon("icons/database_new.gif")
                ){
                public void actionPerformed(ActionEvent ev){
                    if(!settings.promptPassword(LibraryPane.this, null))
                        return;                                        
                    
                    /*
                    Object[] opcions = { options.getMsg(MSG_ID+"createNewLibrary"), options.getMsg(MSG_ID+"connectToLibrary"), options.getMsg("CANCEL")};
                    int n = JOptionPane.showOptionDialog(LibraryPane.this, 
                    options.getMsg(MSG_ID+"new_prompt"), 
                    null,
                    JOptionPane.YES_NO_CANCEL_OPTION,
                    JOptionPane.QUESTION_MESSAGE,
                    null,
                    opcions,
                    opcions[0]);
                    
                    LibraryManagerElement lme=null;
                    if(n==JOptionPane.YES_OPTION)
                        lme=createNewProjectLibrary(null, null);
                    else if(n==JOptionPane.NO_OPTION)
                        lme=locateNewProjectLibrary(null);
                    
                     */
                    
                    LibraryManagerElement lme=NewLibraryDlg.getLibraryManagerElement(LibraryManager.this, LibraryPane.this);                    
                    if(lme!=null){
                        modified=true;
                        libraries.addElement(lme);
                        list.setSelectedValue(lme, true);
                    }
                }
            };            
            newLibraryAction.putValue(AbstractAction.SHORT_DESCRIPTION, options.getMsg(MSG_ID+"new_tooltip"));
        }
        
        public LibraryManagerElement getCurrentSelected(){
            return current;
        }
        
        public void addListSelectionListener(ListSelectionListener lst){
            list.addListSelectionListener(lst);
        }
        
        public void valueChanged(ListSelectionEvent ev){
            if(ev!=null && !ev.getValueIsAdjusting()){
                current=(LibraryManagerElement)list.getSelectedValue();            
                editAction.setEnabled(current!=null && current.editable);
                deleteAction.setEnabled(current!=null);                
                //list.setToolTipText(item==null ? null : item.settings.description);
            }
        }
    }
    
    class LibraryDialog extends edu.xtec.util.ExtendedJDialog implements ListSelectionListener{        
        
        LibraryManagerElement result=null;        
        Action selectAction, cancelAction;
        LibraryPane pane;
        boolean allowEdit, onlyEditable;
        
        LibraryDialog(boolean allowEdit, boolean selectOnlyEditable){
            super(settings.rb.getComponent(), settings.rb.getMsg(MSG_ID+"caption"), true);
            this.allowEdit=allowEdit;
            onlyEditable=selectOnlyEditable;
            pane=new LibraryPane(allowEdit, selectOnlyEditable);
            pane.setOpaque(false);
            buildActions();
            pane.addListSelectionListener(this);
            getContentPane().add(pane, BorderLayout.CENTER);
            JPanel buttonsPanel=new JPanel();
            buttonsPanel.setOpaque(false);
            //buttonsPanel.add(new JButton(selectAction));
            JButton btSelect=new JButton(selectAction);
            btSelect.setDefaultCapable(true);
            buttonsPanel.add(btSelect);
            buttonsPanel.add(new JButton(cancelAction));
            getContentPane().add(buttonsPanel, BorderLayout.SOUTH);
            getRootPane().setDefaultButton(btSelect);
            pack();     
            centerOver(settings.rb.getComponent());            
            pane.list.setSelectedIndex(0);
            valueChanged(new ListSelectionEvent(pane.list, 0, 0, false));
            valueChanged(null);
        }
        
        public void valueChanged(ListSelectionEvent ev){
            if(ev!=null && !ev.getValueIsAdjusting()){
                LibraryManagerElement item=pane.getCurrentSelected();
                selectAction.setEnabled(item!=null && item.exists && (!onlyEditable || item.editable));
            }
        }
        
        void closeDialog(){
            if(modified){
                settings.save();
                modified=false;
            }
            setVisible(false);
        }
        
        void buildActions(){
            final edu.xtec.util.Options options=settings.rb.getOptions();
            selectAction=new AbstractAction(
                options.getMsg(MSG_ID+"select_caption"),
                ResourceManager.getImageIcon("icons/file_open.gif")
                ){
                public void actionPerformed(ActionEvent ev){
                    result=(LibraryManagerElement)pane.list.getSelectedValue();
                    closeDialog();
                }
            };
            selectAction.putValue(AbstractAction.SHORT_DESCRIPTION, options.getMsg(MSG_ID+"select_tooltip"));
            selectAction.setEnabled(false);
            
            cancelAction=new AbstractAction(
                options.getMsg(MSG_ID+"cancel_caption"),
                ResourceManager.getImageIcon("icons/cancel.gif")
                ){
                public void actionPerformed(ActionEvent ev){
                    result=null;
                    closeDialog();
                }
            };            
            cancelAction.putValue(AbstractAction.SHORT_DESCRIPTION, options.getMsg(MSG_ID+"cancel_tooltip"));
        }
    }
}
