/*
    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "qldap.h"

#include <cstdlib>
#include <ldap.h>
#include <errno.h>

#define LDAP_FULL_ADD 0xff

class QLdap::Private
{
  public:
    Private()
      : handle( 0 ), port( 389 ), errorCode( 0 )
    {
    }

    LDAP *handle;

    QString hostName;
    quint16 port;
    QString baseDn;
    QString user;
    QString password;

    int errorCode;
};

QLdap::QLdap( QObject *parent )
  : QObject( parent ), d( new Private() )
{
}

QLdap::QLdap( const QString &hostName, quint16 port, QObject *parent )
  : QObject( parent ), d( new Private() )
{
  d->hostName = hostName;
  d->port = port;
}

QLdap::~QLdap()
{
  delete d;
  d = 0;
}

void QLdap::setHost( const QString &hostName )
{
  d->hostName = hostName;
}

QString QLdap::host() const
{
  return d->hostName;
}

void QLdap::setPort( int port )
{
  d->port = port;
}

int QLdap::port() const
{
  return d->port;
}

void QLdap::setUser( const QString &user )
{
  d->user = user;
}

QString QLdap::user() const
{
  return d->user;
}

void QLdap::setPassword( const QString &password )
{
  d->password = password;
}

QString QLdap::password() const
{
  return d->password;
}

void QLdap::setBaseDn( const QString &baseDn )
{
  d->baseDn = baseDn;
}

QString QLdap::baseDn() const
{
  return d->baseDn;
}

QLdapResponse QLdap::search( const QString &base, Scope scope, const QString &filter,
                             const QStringList &attributes, const QString &sortBy )
{
  QLdapResponse response;

  bool ok = connect();
  if ( !ok )
    return response;

  int ldapScope = LDAP_SCOPE_BASE;
  if ( scope == Base )
    ldapScope = LDAP_SCOPE_BASE;
  else if ( scope == OneLevel )
    ldapScope = LDAP_SCOPE_ONELEVEL;
  else if ( scope == Sub )
    ldapScope = LDAP_SCOPE_SUBTREE;

  // copy attributes in query structures
  char **attrs = 0;
  if ( !attributes.isEmpty() ) {
    attrs = ( char** )malloc( sizeof(char*) * (attributes.count() + 1) );

    int i;
    for ( i = 0; i < attributes.count(); ++i ) {
      QByteArray data = attributes[ i ].toUtf8();
      int length = data.length() + 1;
      attrs[ i ] = ( char* )malloc( length );
      memset( attrs[ i ], 0, length );
      memcpy( attrs[ i ], data.data(), length - 1 );
    }
    attrs[ attributes.count() ] = 0;
  }

  LDAPMessage *res;

  int result = ldap_search_s( d->handle, base.toUtf8(), ldapScope, filter.toUtf8(), attrs, 0, &res );
  if ( result != LDAP_SUCCESS ) {
    d->errorCode = result;
  } else {
    LDAPMessage *entry = 0;
    char *attr = 0;

    if ( !sortBy.isEmpty() )
      ldap_sort_entries( d->handle, &res, sortBy.toUtf8(), strcasecmp );

    QLdapEntry::List ldapEntries;

    for ( entry = ldap_first_entry( d->handle, res ); entry != 0; entry = ldap_next_entry( d->handle, entry ) ) {
      QLdapEntry ldapEntry;

      char *dn = ldap_get_dn( d->handle, entry );
      ldapEntry.setDn( QString::fromUtf8( dn ) );
      ldap_memfree( dn );

      BerElement *ber = 0;
      for ( attr = ldap_first_attribute( d->handle, entry, &ber ); attr != 0; attr = ldap_next_attribute( d->handle, entry, ber ) ) {
        char **vals = ldap_get_values( d->handle, entry, attr );
        if ( vals == 0 )
          continue;

        int valueCount = ldap_count_values( vals );
        for ( int i = 0; i < valueCount; ++i ) {
          ldapEntry.addValue( QString::fromUtf8( attr ), QString::fromUtf8( vals[ i ] ) );
        }

        ldap_value_free(vals);
        vals = 0;

        ldap_memfree(attr);
        attr = 0;
      }

      ldapEntries.append( ldapEntry );
    }

    response.setEntries( ldapEntries );
  }

  // free query structures
  for ( int i = 0; i < attributes.count(); ++i )
    free( attrs[ i ] );
  free( attrs );

  disconnect();

  return response;
}

bool QLdap::add( const QString &dn, const QLdapEntry &entry )
{
  bool ok = connect();
  if ( !ok )
    return false;

  ok = ok && change( dn, entry, LDAP_FULL_ADD );
  ok = ok && disconnect();

  return ok;
}

bool QLdap::addAttributes( const QString &dn, const QLdapEntry &entry )
{
  bool ok = connect();
  if ( !ok )
    return false;

  ok = ok && change( dn, entry, LDAP_MOD_ADD );
  ok = ok && disconnect();

  return ok;
}

bool QLdap::modifyAttributes( const QString &dn, const QLdapEntry &entry )
{
  bool ok = connect();
  if ( !ok )
    return false;

  ok = ok && change( dn, entry, LDAP_MOD_REPLACE );
  ok = ok && disconnect();

  return ok;
}

bool QLdap::removeAttributes( const QString &dn, const QLdapEntry &entry )
{
  bool ok = connect();
  if ( !ok )
    return false;

  ok = ok && change( dn, entry, LDAP_MOD_DELETE );
  ok = ok && disconnect();

  return ok;
}

bool QLdap::remove( const QString &dn )
{
  bool ok = connect();
  if ( !ok )
    return false;

  int result = ldap_delete_s( d->handle, dn.toUtf8().data() );
  if ( result != LDAP_SUCCESS ) {
    d->errorCode = result;
    disconnect();
    return false;
  }

  ok = ok && disconnect();

  return ok;
}

bool QLdap::rename( const QString &dn, const QString &newDn, const QString &baseDn, bool deleteOld )
{
  bool ok = connect();
  if ( !ok )
    return false;

  int result = ldap_rename_s( d->handle, dn.toUtf8().data(), newDn.toUtf8().data(),
                              baseDn.toUtf8().data(), ( deleteOld ? 1 : 0 ), 0, 0 );
  if ( result != LDAP_SUCCESS ) {
    d->errorCode = result;
    disconnect();
    return false;
  }

  ok = ok && disconnect();

  return ok;
}

bool QLdap::change( const QString &dn, const QLdapEntry &entry, int operation )
{
  LDAPMod **ldap_mods;

  int size = (entry.attributeNames().count()+1) * sizeof(LDAPMod*);
  ldap_mods = (LDAPMod**)malloc(size);
  memset(ldap_mods, 0, size);

  bool fullAdd = false;
  if ( operation == LDAP_FULL_ADD ) {
    operation = LDAP_MOD_ADD;
    fullAdd = true;
  }

  for ( int i = 0; i < entry.attributeNames().count(); ++i ) {
    ldap_mods[ i ] = (LDAPMod*)malloc( sizeof( LDAPMod ) );
    ldap_mods[ i ]->mod_type = strdup( entry.attributeNames()[ i ].toUtf8().data() );
    ldap_mods[ i ]->mod_op = operation | LDAP_MOD_BVALUES;

    QStringList values = entry.values( entry.attributeNames()[ i ] );

    if ( values.isEmpty() ) {
      ldap_mods[ i ]->mod_bvalues = 0;
    } else {
      int size = (values.count() + 1) * sizeof( struct berval* );
      ldap_mods[ i ]->mod_bvalues = (struct berval**)malloc( size );
      memset(ldap_mods[ i ]->mod_bvalues, 0, size );

      for ( int j = 0; j < values.count(); ++j ) {
        ldap_mods[ i ]->mod_bvalues[ j ] = (struct berval*)malloc( sizeof( struct berval ) );
        ldap_mods[ i ]->mod_bvalues[ j ]->bv_len = values[ j ].toUtf8().length();
        ldap_mods[ i ]->mod_bvalues[ j ]->bv_val = strdup( values[ j ].toUtf8().data() );

        qDebug( "%s: %s (%d)", qPrintable( entry.attributeNames()[ i ] ), ldap_mods[ i ]->mod_bvalues[ j ]->bv_val,
                (int)ldap_mods[ i ]->mod_bvalues[ j ]->bv_len );
      }

      ldap_mods[ i ]->mod_bvalues[ values.count() ] = 0;
    }
  }

  ldap_mods[ entry.attributeNames().count() ] = 0;

  int result = 0;
  if ( fullAdd )
    result = ldap_add_s( d->handle, dn.toUtf8().data(), ldap_mods );
  else
    result = ldap_modify_s( d->handle, dn.toUtf8().data(), ldap_mods );

  if ( result != LDAP_SUCCESS )
    d->errorCode = result;

  for ( int i = 0; i < entry.attributeNames().count(); ++i ) {
    free( ldap_mods[ i ]->mod_type );

    QStringList values = entry.values( entry.attributeNames()[ i ] );
    if ( !values.isEmpty() ) {
      for ( int j = 0; j < values.count(); ++j ) {
        free( ldap_mods[ i ]->mod_bvalues[ j ]->bv_val );
        free( ldap_mods[ i ]->mod_bvalues[ j ] );
      }
      free( ldap_mods[ i ]->mod_bvalues );
    }
    free( ldap_mods[ i ] );
  }

  free( ldap_mods );

  return ( result == LDAP_SUCCESS );
}

QString QLdap::errorString() const
{
  return QString::fromUtf8( ldap_err2string( d->errorCode ) );
}

bool QLdap::connect()
{
  if ( d->handle != 0 )
    return true;

  d->handle = ldap_open( d->hostName.toUtf8(), d->port );
  if ( d->handle == 0 ) {
    d->errorCode = errno;
    return false;
  }

  int version = 3;
  ldap_set_option( d->handle, LDAP_OPT_PROTOCOL_VERSION, &version );

  bool ok = bind();

  return ok;
}

bool QLdap::bind()
{
  int result = ldap_simple_bind_s( d->handle, d->user.toUtf8(), d->password.toUtf8() );
  if ( result != LDAP_SUCCESS ) {
    d->errorCode = result;
    return false;
  }

  return true;
}

bool QLdap::unbind()
{
  if ( d->handle == 0 )
    return false;

  int result = ldap_unbind_s( d->handle );
  if ( result != LDAP_SUCCESS ) {
    d->errorCode = result;
    return false;
  }

  return true;
}

bool QLdap::disconnect()
{
  bool ok = unbind();

  d->handle = 0;

  return ok;
}

QLdapEntry::QLdapEntry()
{
}

QLdapEntry::~QLdapEntry()
{
}

QString QLdapEntry::dn() const
{
  return mDn;
}

void QLdapEntry::setDn( const QString &dn )
{
  mDn = dn;
}

QStringList QLdapEntry::attributeNames() const
{
  return mValues.keys();
}

QString QLdapEntry::value( const QString &attributeName ) const
{
  if ( !mValues.contains( attributeName ) )
    return QString();
  else
    return mValues[ attributeName ].first();
}

QStringList QLdapEntry::values( const QString &attributeName ) const
{
  if ( !mValues.contains( attributeName ) )
    return QStringList();
  else
    return mValues[ attributeName ];
}

void QLdapEntry::setValue( const QString &attributeName, const QString &value )
{
  if ( value.isEmpty() )
    return;

  if ( !mValues.contains( attributeName ) )
    mValues.insert( attributeName, QStringList() );

  mValues[ attributeName ] = QStringList( value );
}

void QLdapEntry::addValue( const QString &attributeName, const QString &value )
{
  if ( value.isEmpty() )
    return;

  if ( !mValues.contains( attributeName ) )
    mValues.insert( attributeName, QStringList() );

  mValues[ attributeName ].append( value );
}

void QLdapEntry::clearValue( const QString &attributeName )
{
  mValues.remove( attributeName );
}

void QLdapEntry::dump() const
{
  qDebug( "dn: %s", qPrintable( mDn ) );

  QMap<QString, QStringList>::ConstIterator it;
  for ( it = mValues.begin(); it != mValues.end(); ++it ) {
    for ( int i = 0; i < it.value().count(); ++i ) {
      qDebug( "%s: %s", qPrintable( it.key() ), qPrintable( it.value()[ i ] ) );
    }
  }

  qDebug( " " );
}

QLdapResponse::QLdapResponse()
  : mValid( false )
{
}

QLdapResponse::~QLdapResponse()
{
}

bool QLdapResponse::isValid() const
{
  return mValid;
}

QLdapEntry::List QLdapResponse::entries() const
{
  return mEntries;
}

void QLdapResponse::setEntries( const QLdapEntry::List &entries )
{
  mEntries = entries;
  mValid = true;
}

void QLdapResponse::dump() const
{
  QLdapEntry::List::ConstIterator it;
  for ( it = mEntries.begin(); it != mEntries.end(); ++it )
    (*it).dump();
}

