//==============================================================================
//  Lynkeos
//  $Id: MyListProcessing.h,v 1.14 2005/01/27 23:10:04 j-etienne Exp $
//  Created by Jean-Etienne LAMIAUD on Fri Nov 07 2003.
//------------------------------------------------------------------------------
//  Copyright (c) 2003-2005. Jean-Etienne LAMIAUD
//------------------------------------------------------------------------------
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
//------------------------------------------------------------------------------

/*!
 * @header
 * @abstract Threadable classes for image list processing
 * @discussion The root class for threadable image list processing is 
 *    MyListProcessing. Each concrete subclass implements a dedicated method to
 *    start the processing.
 */

#ifndef __MYLISTPROCESSING_H
#define __MYLISTPROCESSING_H

#import <Foundation/Foundation.h>

#include "MyImageListItem.h"

#include "processing_core.h"

/*!
 * @class MyListProcessing
 * @abstract Threadable wrapper class for image list processing
 * @discussion It provides the thread environment to run legacy C routines
 *    dedicated to image processing.
 *
 *    The objects arguments to thread messages are passed by address inside a 
 *    NSData because we need the real object, not a proxy (it was never meant  
 *    to be a distributed application ;o).
 *
 *    The thread is created when the processing is started. It is done this way 
 *    to ensure that the best processor is selected. Were the thread created at
 *    document opening, the load would vary on the processor where the thread 
 *    runs, and could be unfavorable when a processing really starts.
 * @ingroup Processing
 */
#define K_RX_PORT_KEY @"rxPort"
#define K_TX_PORT_KEY @"txPort"

@interface MyListProcessing : NSObject
{
@protected
   MyIntegerRect        _cropRectangle;
   id                   _delegate;
   NSEnumerator*	_list;
   RGB                  *_darkFrame;
   RGB                  *_flatField;
   bool                 _processEnded;
   NSData*		_result;
}

/*!
 * @method threadWithAttributes:
 * @abstract Thread creation 
 * @param attr Thread attributes : communication ports in a dictionary
 * @discussion This method creates the thread, establishes the connection 
 *   with the calling thread, starts a run loop and handle its autorelease pool.
 */
+ (void) threadWithAttributes:(NSDictionary*)attr ;
- (oneway void) stopProcessing ;

@end

/*!
 * @class MyImageAligner
 * @abstract Concrete MyListProcessing dedicated to image alignment
 */
@interface MyImageAligner : MyListProcessing
{
@private
   NSLock*              _refSpectrumLock;
   MyImageListItem*	_referenceItem;
   FFT_DATA             *_referenceSpectrum;
   FFT_DATA             _bufferSpectrum;
   MyIntegerPoint       _referenceOrigin;
   u_short		_side;
   u_short		_cutoff;
   u_short		_precisionThreshold;
   BOOL                 _refSpectrumAvailable;
}

/*!
 * @method alignWithList:darkFrame:flatField:reference:rectangle:cutOff:precisionThreshold:
 * @abstract Start to align images in a list
 * @param list An enumerator on the list of images to align
 * @param refItem The reference image against which other images are aligned
 * @param dark The stacked dark frames
 * @param flat The stacked flat fields
 * @param rect The default square in which alignment is searched (some images 
 *    can have the square placed elsewhere).
 * @param cutoff The frequency above which the spectrum is cleared to avoid 
 *    spurious alignment on some systematic noise).
 * @param threshold The threshold on the correlation peak standard deviation 
 *    (above it, the alignment is considered failed).
 * @discussion The reference item spectrum is computed and stored. After return
 *    from this method, the processing starts. For each selected  item given by
 *    the enumerator, its spectrum is computed and used to correlate against the
 *    reference.
 *
 *    The image data has the dark frame substracted from it, and is then  
 *    divided by the flat field, before Fourier transform.
 *
 *    The correlation peak is the alignment offset which is sent back to the 
 *    calling thread by means of the processDidProgress delegate.
 */
- (oneway void) alignWithList:(NSData*)list reference:(NSData*)refItem
                    refBuffer:(NSData*)refSpectrum
                         lock:(NSData*)lock
                    darkFrame:(NSData*)dark flatField:(NSData*)flat
                    rectangle:(MyIntegerRect)rect 
                       cutOff:(double)cutoff 
           precisionThreshold:(double)threshold;
@end

/*!
 * @class MyImageAnalyzer
 * @abstract Concrete MyListProcessing dedicated to image quality
 */
typedef enum
{
   EntropyAnalysis,
   SpectrumAnalysis
} MyAnalysisMethod;

@interface MyImageAnalyzer : MyListProcessing
{
   u_short              _side;
   MyAnalysisMethod     _method;
   u_short              _lowerCutoff;
   u_short              _upperCutoff;
   FFT_DATA             _bufferSpectrum;
}

/*!
 * @method analyzeWithList:darkFrame:flatField:rectangle:lowCutoff:highCutoff:
 * @abstract Start the analysis of the images in a list
 * @param list An enumerator on the list of images to analyze
 * @param dark The stacked dark frames
 * @param flat The stacked flat fields
 * @param rect The square in which the image is analyzed
 * @param lCutoff Lower frequency cutoff (lower frequencies are not used)
 * @param hCutoff high frequency cutoff (higher frequencies are not used)
 * @discussion After return from this method, the processing starts. For each 
 *    selected item given by the enumerator, its power spectrum is computed and
 *    integrated between lCutoff and hCutoff frequencies.
 *
 *    The image data has the dark frame substracted from it, and is then  
 *    divided by the flat field, before Fourier transform.
 *
 *    The integrated value is the evaluated image quality which is sent back 
 *    to the calling thread by means of the processDidProgress delegate.
 */
- (oneway void) analyzeWithList:(NSData*)list  
                      darkFrame:(NSData*)dark flatField:(NSData*)flat
                      rectangle:(MyIntegerRect)rect
                         method:(MyAnalysisMethod)method
                      lowCutoff:(double)lCutoff highCutoff:(double)hCutoff ;

@end

/*!
 * @class MyImageStack
 * @abstract Concrete MyListProcessing dedicated to image stacking
 */
@interface MyImageStack : MyListProcessing
{
   RGB*                 _rgbSum;
   RGB*                 _rgbBuffer;
   u_short              _factor;
}

/*!
 * @method stackWithList:darkFrame:flatField:rectangle:sizeFactor:
 * @abstract Start to stack the images of a list
 * @param list An enumerator on the list of images to stack
 * @param dark The stacked dark frames
 * @param flat The stacked flat fields
 * @param rect The rectangle in which the images are stacked
 * @param factor Homothetic factor for the stacked image
 * @discussion After return from this method, the processing starts. For each 
 *    selected and aligned item given by the enumerator, is cropped along the 
 *    rectangle, expanded by factor and stacked with subpixel accuracy.
 *
 *    The image data has the dark frame substracted from it, and is then  
 *    divided by the flat field, before stacking.
 *
 *    The calling thread is informed after each frame is added to the stack by 
 *    means of the processDidProgress delegate.
 */
- (oneway void) stackWithList:(NSData*)list 
                    darkFrame:(NSData*)dark flatField:(NSData*)flat
                    rectangle:(MyIntegerRect)rect
                   sizeFactor:(u_short)factor ;
@end

@protocol MyListProcessingDelegate
- (oneway void) processDidCreate :(id)obj ;
- (oneway void) processDidProgress :(NSData*)item data:(NSData*)what ;
- (oneway void) processDidFinish :(id)sender data:(NSData*)result;
@end

/*!
 * @function normalize_rgb
 * @abstract Multiply the pixels by a scalar
 * @param rgb The sarray of rgb value to normalize
 * @param length The array length
 * @param scale The scale by wich the pixels will be multiplied. If 0, they will
 *    be multiplied by 1/max(rgb)
 * @discussion This function is used for dark frames or flat fields, to 
 *    normalize the stacking result according to its use (ie: dark frame stack 
 *    shall be a mean value, and flat fields stack shall have its maximum 
 *    equals to one).
 */
extern void normalize_rgb( RGB *rgb, u_long length, double scale, BOOL mono );

#endif
