#include <iostream>
#include <stdlib.h>
#include <stdio.h>
#include <string>
#include <vector>
#include "../Serialization.h"
#include "../helper.h"

#if !NO_MAIN || !defined(TEST_ASSERT)
static int iErrors = 0;
static int iChecks = 0;
#endif

#ifndef TEST_ASSERT
# define TEST_ASSERT(...) iChecks++; if (!(__VA_ARGS__)) ++iErrors
#endif

#ifndef TEST_VERIFY
# define TEST_VERIFY(...) do {                                               \
    if (!(__VA_ARGS__)) {                                                    \
        fprintf(stderr, "\n[ERROR] %s() failed:\n\n", __FUNCTION__);         \
        fprintf(stderr, "Violated expectation for this failure was:\n\n  "   \
                #__VA_ARGS__ "\n\n  [%s at line %d]\n\n",                    \
                lastPathComponent(__FILE__).c_str(), __LINE__);              \
    }                                                                        \
    TEST_ASSERT(__VA_ARGS__);                                                \
} while (false);
#endif

#define SRLZ(member) \
    archive->serializeMember(*this, member, #member);

struct SimpleStruct {
    bool a;
    bool b;
    int c;
    float d;
    double e;
    std::string f;

    void serialize(Serialization::Archive* archive) {
        SRLZ(a);
        SRLZ(b);
        SRLZ(c);
        SRLZ(d);
        SRLZ(e);
        SRLZ(f);
    }

    bool operator==(const SimpleStruct& o) const {
        return a == o.a &&
               b == o.b &&
               c == o.c &&
               d == o.d &&
               e == o.e &&
               f == o.f;
    }
};

struct AggregateStruct {
    int one;
    SimpleStruct two;

    void serialize(Serialization::Archive* archive) {
        SRLZ(one);
        SRLZ(two);
    }

    bool operator==(const AggregateStruct& o) const {
        return one == o.one &&
               two == o.two;
    }
};

struct VectorStruct {
    float f;
    std::vector<float> v;

    void serialize(Serialization::Archive* archive) {
        SRLZ(f);
        SRLZ(v);
    }

    bool operator==(const VectorStruct& o) const {
        return f == o.f &&
               v == o.v;
    }
};

struct VectorAggregateStruct {
    int i;
    std::vector<SimpleStruct> v;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(v);
    }

    bool operator==(const VectorAggregateStruct& o) const {
        return i == o.i &&
               v == o.v;
    }
};

struct SetStruct {
    int i;
    std::set<bool> bs;
    std::set<int> is;
    std::set<unsigned int> uis;
    std::set<float> fs;
    std::set<double> ds;
    std::set<std::string> ss;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(bs);
        SRLZ(is);
        SRLZ(uis);
        SRLZ(fs);
        SRLZ(ds);
        SRLZ(ss);
    }

    bool operator==(const SetStruct& o) const {
        return i == o.i &&
               bs == o.bs &&
               is == o.is &&
               uis == o.uis &&
               fs == o.fs &&
               ds == o.ds &&
               ss == o.ss;
    }
};

struct MapStruct {
    int i;
    std::map<int,int> iim;
    std::map<int,double> idm;
    std::map<float,int> fim;
    std::map<std::string,bool> sbm;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(iim);
        SRLZ(idm);
        SRLZ(fim);
        SRLZ(sbm);
    }

    bool operator==(const MapStruct& o) const {
        return i == o.i &&
               iim == o.iim &&
               idm == o.idm &&
               fim == o.fim &&
               sbm == o.sbm;
    }
};

struct MapAggregateStruct {
    int i;
    std::map<int,SimpleStruct> m;

    void serialize(Serialization::Archive* archive) {
        SRLZ(i);
        SRLZ(m);
    }

    bool operator==(const MapAggregateStruct& o) const {
        return i == o.i &&
               m == o.m;
    }
};

template<class T>
static Serialization::Archive transfer(const T& src, T& dst) {
    Serialization::RawData raw;

    // serialize
    {
        Serialization::Archive a;
        a.serialize(&src);
        raw = a.rawData();
    }

    // deserialize
    {
        Serialization::Archive a(raw);
        a.deserialize(&dst);
        return a;
    }
}


/////////////////////////////////////////////////////////////////////////////
// The actual test cases ...

static void test_SimpleStruct() {
    SimpleStruct src = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, dst;
    Serialization::Archive a = transfer(src, dst);
    TEST_VERIFY(src == dst);
    src = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "postchange",
    };
    TEST_VERIFY(!(src == dst));
    a.setBoolValue(a.objectByUID(a.rootObject().memberNamed("a").uid()), true);
    a.setBoolValue(a.objectByUID(a.rootObject().memberNamed("b").uid()), false);
    a.setIntValue(a.objectByUID(a.rootObject().memberNamed("c").uid()), 12);
    a.setRealValue(a.objectByUID(a.rootObject().memberNamed("d").uid()), 0.28f);
    a.setRealValue(a.objectByUID(a.rootObject().memberNamed("e").uid()), 188.2);
    a.setStringValue(a.objectByUID(a.rootObject().memberNamed("f").uid()), "postchange");
    a.deserialize(&dst);
    TEST_VERIFY(src == dst);
    src = {
        .a = false,
        .b = false,
        .c = 3127,
        .d = 3.14f,
        .e = 56.25,
        .f = "autotext",
    };
    TEST_VERIFY(!(src == dst));
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("a").uid()), "false");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("b").uid()), "false");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("c").uid()), "3127");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("d").uid()), "3.14");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("e").uid()), "56.25");
    a.setAutoValue(a.objectByUID(a.rootObject().memberNamed("f").uid()), "autotext");
    a.deserialize(&dst);
    TEST_VERIFY(src == dst);
}

static void test_AggregateStruct() {
    AggregateStruct src = {
        .one = 732,
        .two = {
            .a = true,
            .b = false,
            .c = 12,
            .d = 0.28f,
            .e = 188.2,
            .f = "aggregate",
        }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdVector() {
    VectorStruct src = {
        .f = 83.7,
        .v = { 1.0, 7.3, 0.35, 234.17 }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdVectorAggregate() {
    SimpleStruct one = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, two = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "something",
    };
    VectorAggregateStruct src = {
        .i = 1045,
        .v = { one, two }
    }, dst;
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdSet() {
    SetStruct src = {
        .i = 94783,
        .bs = { false, true, true, false, true },
        .is = { -7, 940, 0, -64 },
        .uis = { 7, 940, 0, 64 },
        .fs = { 3.0f, .3f, 893.54f, 63.924f },
        .ds = { 0.0, 3.345, 193.0, 1349.5479 },
        .ss = { "one", "two", "three", "four" }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdMap() {
    MapStruct src = {
        .i = 6107,
        .iim = {
            { 3, 5 },
            { 0, 9 },
            { 349, 0 }
        },
        .idm = {
            { 0, 5.0 },
            { 9, 0.34 },
            { 20, 120.7 },
            { 40098, 3089.2 }
        },
        .fim = {
            { 9.f, 1 },
            { 0.3f, 2 },
            { 0.34f, 3 },
            { 0.0f, 4 },
            { 22.4f, 5 }
        },
        .sbm = {
            { "one", false },
            { "two", false },
            { "three", true },
            { "four", false },
            { "five", true }
        }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

static void test_StdMapAggregate() {
    SimpleStruct one = {
        .a = false,
        .b = true,
        .c = 567,
        .d = 4.58f,
        .e = 89.4,
        .f = "cheers",
    }, two = {
        .a = true,
        .b = false,
        .c = 12,
        .d = 0.28f,
        .e = 188.2,
        .f = "something",
    };
    MapAggregateStruct src = {
        .i = 1045,
        .m = {
            { 5, one },
            { 7, two }
        }
    }, dst = { };
    TEST_VERIFY(!(src == dst));
    transfer(src, dst);
    TEST_VERIFY(src == dst);
}

#if !NO_MAIN

int main() {
    printf("\n");

    int iExceptions = 0;
    try {
        test_SimpleStruct();
        test_AggregateStruct();
        test_StdVector();
        test_StdVectorAggregate();
        test_StdSet();
        test_StdMap();
        test_StdMapAggregate();
    } catch (...) {
        iExceptions++;
    }

    if (iErrors || iExceptions)
        printf("\n!!! FAILED !!!  [ There were %d errors, %d exceptions ]\n\n",
               iErrors, iExceptions);
    else
        printf("\nOK (%d checks)\n\n", iChecks);

    return iErrors + iExceptions;
}

#endif // !NO_MAIN
