/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#include "mediascanner/dbustypes.h"

// Boost C++
#include <boost/date_time.hpp>
#include <boost/variant/static_visitor.hpp>

// C++ Standard Library
#include <set>
#include <string>
#include <utility>
#include <vector>

// Media Scanner Library
#include "mediascanner/locale.h"

namespace mediascanner {
namespace dbus {

Signature::Signature(const std::string &signature)
    : signature_(signature) {
}

Signature::Signature(const GVariantType *type)
    : signature_(g_variant_type_peek_string(type)) {
}

Signature::operator const char*() const {
    return signature_.c_str();
}

Signature::operator const GVariantType*() const {
    return G_VARIANT_TYPE(signature_.c_str());
}

const std::string& Signature::str() const {
    return signature_;
}

Signature Signature::array(const Signature &element_type) {
    return "a" + element_type.str();
}

Signature Signature::dictionary(const Signature &key_type,
                                const Signature &value_type) {
    return "a{" + key_type.str() + value_type.str() + "}";
}

Signature Signature::tuple(const Signature &element) {
    return "(" + element.str() + ")";
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<bool>::signature() {
    static const Signature signature = G_VARIANT_TYPE_BOOLEAN;
    return signature;
}

template<> GVariant* Type<bool>::make_variant(bool value) {
    return g_variant_new(signature(), value ? TRUE : FALSE);
}

template<> bool Type<bool>::make_value(GVariant *variant) {
    return variant ? g_variant_get_boolean(variant) : false;
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<int32_t>::signature() {
    static const Signature signature = G_VARIANT_TYPE_INT32;
    return signature;
}

template<> GVariant* Type<int32_t>::make_variant(int32_t value) {
    return g_variant_new(signature(), value);
}

template<> int32_t Type<int32_t>::make_value(GVariant *variant) {
    return variant ? g_variant_get_int32(variant) : 0;
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<uint32_t>::signature() {
    static const Signature signature = G_VARIANT_TYPE_UINT32;
    return signature;
}

template<> GVariant* Type<uint32_t>::make_variant(uint32_t value) {
    return g_variant_new(signature(), value);
}

template<> uint32_t Type<uint32_t>::make_value(GVariant *variant) {
    return variant ? g_variant_get_uint32(variant) : 0;
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<uint64_t>::signature() {
    static const Signature signature = G_VARIANT_TYPE_UINT64;
    return signature;
}

template<> GVariant* Type<uint64_t>::make_variant(uint64_t value) {
    return g_variant_new(signature(), value);
}

template<> uint64_t Type<uint64_t>::make_value(GVariant *variant) {
    return variant ? g_variant_get_uint64(variant) : 0;
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<double>::signature() {
    static const Signature signature = G_VARIANT_TYPE_DOUBLE;
    return signature;
}

template<> GVariant* Type<double>::make_variant(double value) {
    return g_variant_new(signature(), value);
}

template<> double Type<double>::make_value(GVariant *variant) {
    return variant ? g_variant_get_double(variant) : 0;
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<float>::signature() {
    static const Signature signature = G_VARIANT_TYPE_DOUBLE;
    return signature;
}

template<> GVariant* Type<float>::make_variant(float value) {
    return g_variant_new(signature(), static_cast<double>(value));
}

template<> float Type<float>::make_value(GVariant *variant) {
    return variant ? g_variant_get_double(variant) : 0;
}

////////////////////////////////////////////////////////////////////////////////

template<>
BoxedType<Fraction>::boxed_type
BoxedType<Fraction>::make_boxed(Fraction value) {
    return boxed_type(value.numerator(), value.denominator());
}

template<>
Fraction BoxedType<Fraction>::make_unboxed(boxed_type value) {
    return Fraction(value.first, value.second);
}

////////////////////////////////////////////////////////////////////////////////

static const DateTime kEpoch(boost::gregorian::date(1970, 1, 1));

template<> uint64_t BoxedType<DateTime>::make_boxed(DateTime value) {
    return (value - kEpoch).total_microseconds();
}

template<> DateTime BoxedType<DateTime>::make_unboxed(uint64_t value) {
    return kEpoch + boost::posix_time::microseconds(value);
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<std::string>::signature() {
    static const Signature signature = G_VARIANT_TYPE_STRING;
    return signature;
}

template<> GVariant* Type<std::string>::make_variant(std::string value) {
    return g_variant_new(signature(), value.c_str());
}

template<> std::string Type<std::string>::make_value(GVariant *variant) {
    if (variant == nullptr)
        return value_type();

    size_t len = 0;
    const char *const str = g_variant_get_string(variant, &len);
    return std::string(str, len);
}

// FIXME(M5): Figure out how to avoid this duplication.
template<> const std::string
Type<const std::string>::make_value(GVariant *variant) {
    return Type<std::string>::make_value(variant);
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<std::wstring>::signature() {
    return Type<std::string>::signature();
}

template<> GVariant* Type<std::wstring>::make_variant(std::wstring value) {
    return Type<std::string>::make_variant(FromUnicode(value));
}

template<> std::wstring Type<std::wstring>::make_value(GVariant *variant) {
    if (variant == nullptr)
        return value_type();

    return ToUnicode(Type<std::string>::make_value(variant));
}

////////////////////////////////////////////////////////////////////////////////

template<> const Signature& Type<Property>::signature() {
    return Type<std::wstring>::signature();
}

template<> GVariant* Type<Property>::make_variant(Property value) {
    return Type<std::wstring>::make_variant(value.field_name());
}

template<> Property Type<Property>::make_value(GVariant *variant) {
    if (variant == nullptr)
        return value_type();

    const std::wstring &field_name = Type<std::wstring>::make_value(variant);
    return mediascanner::Property::FromFieldName(field_name);
}

////////////////////////////////////////////////////////////////////////////////

const Signature& Type<Property::Value>::signature() {
    static const Signature signature = G_VARIANT_TYPE_VARIANT;
    return signature;
}

class MakeVariantVisitor : public boost::static_visitor<GVariant *> {
public:
    MakeVariantVisitor() {
        // clang++ requires a user-provided default constructor
    }

    template<typename T>
    GVariant* operator()(const T &value) const {
        return g_variant_new_variant(Type<T>::make_variant(value));
    }

    result_type operator()(const boost::blank &) const {
        return result_type();
    }
};

GVariant* Type<Property::Value>::make_variant(const Property::Value &value) {
    static const MakeVariantVisitor make_variant_visitor;
    return value.apply_visitor(make_variant_visitor);
}

////////////////////////////////////////////////////////////////////////////////

const Signature& Type<Property::ValueMap>::signature() {
    static const Signature signature =
            Signature::dictionary(Type<Property>::signature(),
                                  Type<Property::Value>::signature());

    return signature;
}

GVariant* Type<Property::ValueMap>::make_variant
                                            (const Property::ValueMap &value) {
    GVariantBuilder builder;

    g_variant_builder_init(&builder, signature());

    for (typename Property::ValueMap::const_iterator
         it = value.begin(), end = value.end(); it != end; ++it) {
        GVariant *const element = g_variant_new_dict_entry
                (Type<Property>::make_variant(it->first),
                 Type<Property::Value>::make_variant(it->second));
        g_variant_builder_add_value(&builder, element);
    }

    return g_variant_builder_end(&builder);
}

Property::ValueMap Type<Property::ValueMap>::make_value
                    (GVariant *variant, std::set<std::string> *bad_keys) {
    if (variant == nullptr)
        return Property::ValueMap();

    Property::ValueMap properties;

    GVariantIter iter;
    g_variant_iter_init(&iter, variant);

    while (const Wrapper<GVariant> element =
           take(g_variant_iter_next_value(&iter))) {
        const Wrapper<GVariant> key =
                take(g_variant_get_child_value(element.get(), 0));
        const std::string name = g_variant_get_string(key.get(), nullptr);

        const Property property = Property::FromFieldName(ToUnicode(name));

        if (not property) {
            if (bad_keys)
                bad_keys->insert(name);

            continue;
        }

        const Wrapper<GVariant> boxed_value =
                take(g_variant_get_child_value(element.get(), 1));
        Property::Value property_value;

        if (not property.TransformDBusVariant(boxed_value.get(),
                                              &property_value)) {
            if (bad_keys)
                bad_keys->insert(name);

            continue;
        }

        if (not properties.insert(std::make_pair(property,
                                                 property_value)).second) {
            if (bad_keys)
                bad_keys->insert(name);

            continue;
        }
    }

    return properties;
}

////////////////////////////////////////////////////////////////////////////////

const Signature& Type<MediaInfo>::signature() {
    static const Signature signature =
            Signature::array(Type<MediaInfo::value_type>::signature());

    return signature;
}

GVariant* Type<MediaInfo>::make_variant(const MediaInfo &value) {
    return internal::SequenceType<MediaInfo>::make_variant(value);
}

MediaInfo Type<MediaInfo>::make_value(GVariant *variant,
                                      std::set<std::string> *bad_keys) {
    MediaInfo metadata;

    if (variant) {
        for (size_t i = 0, l = g_variant_n_children(variant); i < l; ++i) {
            GVariant *const child = g_variant_get_child_value(variant, i);
            metadata.add_related(Type<MediaInfo::value_type>::
                                 make_value(child, bad_keys));
        }
    }

    return metadata;
}

} // namespace dbus
} // namespace mediascanner
