#
# ubi.py : UbiImageCreator class for creating UBI images
#
# Copyright 2010, Intel Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import os
import subprocess

from mic.imgcreate.errors import *
from mic.imgcreate.creator import *

class UbiImageCreator(LoopImageCreator):
    def __init__(self, ks, name, fslabel = None):
        LoopImageCreator.__init__(self, ks, name, fslabel)
        if self.ks:
            self.image_size = (4*kickstart.get_image_size(self.ks, 200L * 1024 * 1024))
        else:
            self.image_size = 0
        self._set_image_size(self.image_size)
        self.img_done = False
        self._mkubifs = find_binary_path("mkfs.ubifs")
        self._ubinize = find_binary_path("ubinize")
        self._dep_checks.append("mkfs.ubifs")
        self._dep_checks.append("ubinize")
        self._img_name = self.name + ".ubiimg"

    def _writeUbiConf(self, configfile, baseimage):
        conf  = "[ubifs]\n"
        conf += "mode=\"ubi\"\n"
        conf += "image=\"" + baseimage + "\"\n"
        conf += "vol_id=\"0\"\n"
        conf += "vol_size=\"" + str(self.image_size/4/1024/1024) + "MiB\"\n"
        conf += "vol_type=\"dynamic\"\n"
        conf += "vol_name=\"rootfs\"\n"
        conf += "vol_alignment=\"1\"\n"
        conf += "vol_flags=\"autoresize\"\n"

        f = file(configfile, "w")
        f.write(conf)
        f.close()
        os.chmod(configfile, 0644)

    def _genubiimg(self):
        self.img_done = True
        baseimage = self._outdir + "/" + "base.ubi.img"
        configfile = self._outdir + "/" + "ubinize.cfg"
        self._writeUbiConf(configfile, baseimage)
        max_leb_cnt = 2047
        try:
            rc = subprocess.call([self._mkubifs, "-m", "2048", "-e", "129024", "-c", str(max_leb_cnt), "-R", "4MiB", "-r", self._instroot, "-v", "-o", baseimage], stdout=sys.stdout, stderr=sys.stderr)
            if rc:
                raise CreatorError("Failed to make ubifs")
            rc = subprocess.call([self._ubinize, "-o", self._outdir + "/" + self._img_name, "-m", "2048", "-p", "128KiB", "-s", "512", configfile], stdout=sys.stdout, stderr=sys.stderr)
            if rc:
                raise CreatorError("Failed to generate ubi image")
        finally:
            os.unlink(configfile)
            os.unlink(baseimage)

    def _unmount_instroot(self):
        try:
            if not self.img_done:
                self._genubiimg()
        finally:
            LoopImageCreator._unmount_instroot(self)

    def _stage_final_image(self):
        return
