///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include "AtomTypeColorModifier.h"
#include <atomviz/atoms/datachannels/AtomTypeDataChannel.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(AtomTypeColorModifier, AtomsObjectModifierBase)

/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus AtomTypeColorModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Get the selection channel.
	DataChannel* selChannel = inputStandardChannel(DataChannel::SelectionChannel);

	AtomTypeDataChannel* typeChannel = static_object_cast<AtomTypeDataChannel>(expectStandardChannel(DataChannel::AtomTypeChannel));
	if(typeChannel->atomTypes().empty())
		throw Exception(tr("There are no atom types defined."));

	// Get the atom type colors.
	QVector<Vector3> typeColors(typeChannel->atomTypes().size(), Color(1,1,1));
	for(int i=0; i<typeChannel->atomTypes().size(); i++) {
		AtomType* atype = typeChannel->atomTypes()[i];
		if(atype && atype->colorController())
			atype->colorController()->getValue(time, typeColors[i], validityInterval);
	}

	DataChannel* colorChannel = outputStandardChannel(DataChannel::ColorChannel);

	// Assign colors based on atom type.
	const int* t = typeChannel->constDataInt();
	Vector3* c = colorChannel->dataVector3();
	Vector3* cend = c + colorChannel->size();

	if(selChannel != NULL) {
		OVITO_ASSERT(colorChannel->size() == selChannel->size());
		const int* s = selChannel->constDataInt();

		if(inputStandardChannel(DataChannel::ColorChannel) == NULL) {
			QVector<Color> oldColors = input()->getAtomColors(time, validityInterval);
			QVector<Color>::const_iterator oldc = oldColors.constBegin();
			for(size_t i = selChannel->size(); i != 0; i--, ++c, ++s, ++oldc, ++t) {
				if(*s) *c = typeColors[(*t) % typeColors.size()]; else *c = *oldc;
			}
		}
		else {
			for(size_t i = selChannel->size(); i != 0; i--, ++c, ++s, ++t)
				if(*s) *c = typeColors[(*t) % typeColors.size()];
		}

		// Hide selection channel to make new color visible.
		if(selChannel->isVisible())
			outputStandardChannel(DataChannel::SelectionChannel)->setVisible(false);
	}
	else {
		while(c != cend) {
			*c++ = typeColors[(*t++) % typeColors.size()];
		}
	}

	return EvaluationStatus();
}

};	// End of namespace AtomViz
