///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <atomviz/atoms/AtomsObject.h>
#include "LAMMPSDumpWriterSettingsDialog.h"

namespace AtomViz {

/******************************************************************************
* Constructor.
******************************************************************************/
LAMMPSDumpWriterSettingsDialog::LAMMPSDumpWriterSettingsDialog(LAMMPSDumpWriter* writer, AtomsObject* atomsObj, QWidget* parent)
	: QDialog(parent)
{
	setWindowTitle(tr("LAMMPS Dump File Export Settings"));
	this->writer = writer;

	QVBoxLayout* layout1 = new QVBoxLayout(this);

	// Interval group
	QGroupBox* intervalGroupBox = new QGroupBox(tr("Time Interval"), this);
	layout1->addWidget(intervalGroupBox);

	QHBoxLayout* intervalLayout = new QHBoxLayout(intervalGroupBox);
	intervalLayout->setSpacing(0);

	intervalLayout->addWidget(new QLabel(tr("Export from time: ")));
	startTimeSpinner = new SpinnerWidget(this);
	startTimeSpinner->setUnit(UNITS_MANAGER.integerIdentity());
	startTimeSpinner->setIntValue(writer->startFrame());
	startTimeSpinner->setTextBox(new QLineEdit(this));
	startTimeSpinner->setMinValue(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().start()));
	startTimeSpinner->setMaxValue(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().end()));
	intervalLayout->addWidget(startTimeSpinner->textBox());
	intervalLayout->addWidget(startTimeSpinner);
	intervalLayout->addSpacing(8);
	intervalLayout->addWidget(new QLabel(tr("to time: ")));
	endTimeSpinner = new SpinnerWidget(this);
	endTimeSpinner->setUnit(UNITS_MANAGER.integerIdentity());
	endTimeSpinner->setIntValue(writer->endFrame());
	endTimeSpinner->setTextBox(new QLineEdit(this));
	endTimeSpinner->setMinValue(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().start()));
	endTimeSpinner->setMaxValue(ANIM_MANAGER.timeToFrame(ANIM_MANAGER.animationInterval().end()));
	intervalLayout->addWidget(endTimeSpinner->textBox());
	intervalLayout->addWidget(endTimeSpinner);

	// Multi frame group
	QGroupBox* multiframeGroupBox = new QGroupBox(tr("File Generation"), this);
	layout1->addWidget(multiframeGroupBox);

	QGridLayout* multiframeLayout = new QGridLayout(multiframeGroupBox);
	multiframeLayout->setColumnStretch(0, 5);
	multiframeLayout->setColumnStretch(1, 95);
	multiframeButtonGroup = new QButtonGroup(this);

	QRadioButton* radioBtn = new QRadioButton(tr("Write single file containing one or more time steps"));
	multiframeButtonGroup->addButton(radioBtn, 0);
	multiframeLayout->addWidget(radioBtn, 0, 0, 1, 2);
	radioBtn->setChecked(!writer->useWildcardFilename());

	radioBtn = new QRadioButton(tr("Use following wild-card pattern to generate multiple files:"));
	multiframeButtonGroup->addButton(radioBtn, 1);
	multiframeLayout->addWidget(radioBtn, 1, 0, 1, 2);
	radioBtn->setChecked(writer->useWildcardFilename());

	wildcardTextbox = new QLineEdit(writer->wildcardFilename(), multiframeGroupBox);
	multiframeLayout->addWidget(wildcardTextbox, 2, 1, 1, 1);
	wildcardTextbox->setEnabled(radioBtn->isChecked());
	connect(radioBtn, SIGNAL(toggled(bool)), wildcardTextbox, SLOT(setEnabled(bool)));

	// Data channel to column mapping
	QGroupBox* mappingGroupBox = new QGroupBox(tr("Data Columns"), this);
	layout1->addWidget(mappingGroupBox);

	QVBoxLayout* layout3 = new QVBoxLayout(mappingGroupBox);
	mappingEditor = new ChannelColumnMappingEditor(mappingGroupBox);
	mappingEditor->setMapping(writer->channelMapping(), atomsObj);
	layout3->addWidget(mappingEditor);

	QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this);
	connect(buttonBox, SIGNAL(accepted()), this, SLOT(onOk()));
	connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
	layout1->addWidget(buttonBox);
}

/******************************************************************************
* This is called when the user has pressed the OK button.
******************************************************************************/
void LAMMPSDumpWriterSettingsDialog::onOk()
{
	try {
		// Write settings back to the writer.
		writer->setChannelMapping(mappingEditor->mapping());
		writer->setUseWildcardFilename(multiframeButtonGroup->checkedId() == 1);
		writer->setWildcardFilename(wildcardTextbox->text());
		writer->setStartFrame(startTimeSpinner->intValue());
		writer->setEndFrame(endTimeSpinner->intValue());

		// Close dialog box.
		accept();
	}
	catch(const Exception& ex) {
		ex.showError();
		return;
	}
}

};	// End of namespace AtomViz
