///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/properties/RefTargetListParameterUI.h>

namespace Core {

// Gives the class run-time type information.
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(RefTargetListParameterUI, ParameterUI)

DEFINE_FLAGS_VECTOR_REFERENCE_FIELD(RefTargetListParameterUI, RefTarget, "Targets", PROPERTY_FIELD_NO_UNDO | PROPERTY_FIELD_NO_CHANGE_MESSAGE, _targets)

/******************************************************************************
* The constructor.
******************************************************************************/
RefTargetListParameterUI::RefTargetListParameterUI(PropertiesEditor* parentEditor, const PropertyFieldDescriptor& refField, const RolloutInsertionParameters& rolloutParams, PluginClassDescriptor* defaultEditorClass, int listWidgetHeight)
	: ParameterUI(parentEditor), _refField(refField), _rolloutParams(rolloutParams), _defaultEditorClass(defaultEditorClass)
{
	INIT_PROPERTY_FIELD(RefTargetListParameterUI, _targets);
	OVITO_ASSERT_MSG(refField.isVector(), "RefTargetListParameterUI constructor", "The reference field bound to this parameter UI must be a vector reference field.");
	
	_model = new ListViewModel(this);

	class MyListView : public QListView {
	private:
		int _listWidgetHeight;
	public:
		MyListView(int listWidgetHeight) : QListView(), _listWidgetHeight(listWidgetHeight) {}
		virtual QSize sizeHint() const { return QSize(256, _listWidgetHeight); }
	}; 
		
	_listWidget = new MyListView(listWidgetHeight);
	_listWidget->setModel(_model);
	connect(_listWidget->selectionModel(), SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)), this, SLOT(onListSelectionChanged()));

	if(_defaultEditorClass)
		openSubEditor();
}

/******************************************************************************
* Destructor, that releases all GUI controls.
******************************************************************************/
RefTargetListParameterUI::~RefTargetListParameterUI()
{
	/// The destructor must clear all references since this UI object
	/// might have been deleted via the delete operator and not via
	/// the autoDeleteObject() method which should normaly be used for RefMaker derived classes.
	_subEditor = NULL;
	clearAllReferences();
	 	
	// Release GUI controls. 
	delete listWidget(); 
}

/******************************************************************************
* This method is called when a new editable object has been assigned to the properties owner this
* parameter UI belongs to. The parameter UI should react to this change appropriately and
* show the properties value for the new edit object in the UI.
******************************************************************************/
void RefTargetListParameterUI::resetUI()
{
	ParameterUI::resetUI();	
	
	if(listWidget()) {
		listWidget()->setEnabled(editObject() != NULL);
	
		_targets.clear();
		_targetToRow.clear();
		_rowToTarget.clear();
		
		if(editObject()) {
			// Create a local copy of the list of ref targets.
			const QVector<RefTarget*>& reflist = editObject()->getVectorReferenceField(referenceField());
			Q_FOREACH(RefTarget* t, reflist) {
				_targetToRow.push_back(_rowToTarget.size());
				if(t != NULL)
					_rowToTarget.push_back(_targets.size());
				_targets.push_back(t);
			}
		}
		
		_model->resetList();
	}
	openSubEditor();
}

/******************************************************************************
* Is called when the user has selected an item in the list view.
******************************************************************************/
void RefTargetListParameterUI::onListSelectionChanged()
{
	openSubEditor();
}

/******************************************************************************
* Opens a sub-editor for the object that is selected in the list view.
******************************************************************************/
void RefTargetListParameterUI::openSubEditor()
{
	try {
		RefTarget* selection = selectedObject();

		if(subEditor()) {
			// Close old editor if it is no longer needed.
			if(!selection || subEditor()->editObject() == NULL ||
					subEditor()->editObject()->pluginClassDescriptor() != selection->pluginClassDescriptor()) {

				if(selection || subEditor()->pluginClassDescriptor() != _defaultEditorClass)
					_subEditor = NULL;
			}
		}
		if(!subEditor()) {
			if(selection) {
				_subEditor = selection->createPropertiesEditor();
				if(_subEditor)
					_subEditor->initialize(editor()->container(), _rolloutParams);
			}
			else if(_defaultEditorClass) {
				_subEditor = dynamic_object_cast<PropertiesEditor>(_defaultEditorClass->createInstance());
				if(_subEditor)
					_subEditor->initialize(editor()->container(), _rolloutParams);
			}
			else return;
		}
		if(subEditor()) {
			subEditor()->setEditObject(selection);
		}
	}
	catch(const Exception& ex) {
		ex.showError();
	}
}

/******************************************************************************
* Returns the RefTarget that is currently selected in the UI.
******************************************************************************/
RefTarget* RefTargetListParameterUI::selectedObject() const
{
	if(!listWidget()) return NULL;
	QModelIndexList selection = listWidget()->selectionModel()->selectedRows();
	if(selection.empty()) return NULL;
	if(selection.front().row() >= _rowToTarget.size()) return NULL;
	int targetIndex = _rowToTarget[selection.front().row()];
	OVITO_ASSERT(targetIndex < _targets.size());
	CHECK_OBJECT_POINTER(_targets[targetIndex]);
	return _targets[targetIndex];
}

/******************************************************************************
* Selectes the given sub-object in the list.
******************************************************************************/
int RefTargetListParameterUI::setSelectedObject(RefTarget* selObj)
{
	if(!listWidget()) return -1;
	OVITO_ASSERT(_targetToRow.size() == _targets.size());
	if(selObj != NULL) {
		for(int i=0; i<_targets.size(); i++) {
			if(_targets[i] == selObj) {
				int rowIndex = _targetToRow[i];
				listWidget()->selectionModel()->select(_model->index(rowIndex), QItemSelectionModel::ClearAndSelect);
				return rowIndex;
			}
		}
	}
	listWidget()->selectionModel()->clear();
	return -1;
}

/******************************************************************************
* This method is called when a reference target changes.
******************************************************************************/
bool RefTargetListParameterUI::onRefTargetMessage(RefTarget* source, RefTargetMessage* msg)
{
	if(source == editObject()) {
		if(msg->type() == REFERENCE_FIELD_ADDED) {
			ReferenceFieldMessage* refmsg = static_cast<ReferenceFieldMessage*>(msg);
			if(refmsg->field() == referenceField()) {
				int rowIndex;
				if(refmsg->index() < _targetToRow.size())
					rowIndex = _targetToRow[refmsg->index()];
				else
					rowIndex = _rowToTarget.size();
				if(refmsg->newTarget() != NULL)
					_model->beginInsert(rowIndex);
				_targets.insert(refmsg->index(), refmsg->newTarget());
				_targetToRow.insert(refmsg->index(), rowIndex);
				for(int i=rowIndex; i<_rowToTarget.size(); i++)
					_rowToTarget[i]++;
				if(refmsg->newTarget() != NULL) {
					_rowToTarget.insert(rowIndex, refmsg->index());
					for(int i=refmsg->index()+1; i<_targetToRow.size(); i++)
						_targetToRow[i]++;
					_model->endInsert();
				}
#ifdef _DEBUG
				// Check internal list structures.
				int numRows = 0;
				int numTargets = 0;
				const QVector<RefTarget*>& reflist = editObject()->getVectorReferenceField(referenceField());
				Q_FOREACH(RefTarget* t, reflist) {
					OVITO_ASSERT(_targets[numTargets] == t);
					OVITO_ASSERT(_targetToRow[numTargets] == numRows);
					if(t != NULL) {
						OVITO_ASSERT(_rowToTarget[numRows] == numTargets);
						numRows++;
					}
					numTargets++;
				}
#endif
			}
		}
		else if(msg->type() == REFERENCE_FIELD_REMOVED) {
			ReferenceFieldMessage* refmsg = static_cast<ReferenceFieldMessage*>(msg);
			if(refmsg->field() == referenceField()) {
				int rowIndex = _targetToRow[refmsg->index()];
				if(refmsg->oldTarget())
					_model->beginRemove(rowIndex);
				OVITO_ASSERT(refmsg->oldTarget() == _targets[refmsg->index()]);
				_targets.remove(refmsg->index());
				_targetToRow.remove(refmsg->index());
				for(int i=rowIndex; i<_rowToTarget.size(); i++)
					_rowToTarget[i]--;
				if(refmsg->oldTarget()) {
					_rowToTarget.remove(rowIndex);
					for(int i=refmsg->index(); i<_targetToRow.size(); i++)
						_targetToRow[i]--;
					_model->endRemove();
				}
#ifdef _DEBUG
				// Check internal list structures.
				int numRows = 0;
				int numTargets = 0;
				const QVector<RefTarget*>& reflist = editObject()->getVectorReferenceField(referenceField());
				Q_FOREACH(RefTarget* t, reflist) {
					OVITO_ASSERT(_targets[numTargets] == t);
					OVITO_ASSERT(_targetToRow[numTargets] == numRows);
					if(t != NULL) {
						OVITO_ASSERT(_rowToTarget[numRows] == numTargets);
						numRows++;
					}
					numTargets++;
				}
#endif
			}
		}
	} 
	else if(msg->type() == SCHEMATIC_TITLE_CHANGED || msg->type() == REFTARGET_CHANGED) {
		OVITO_ASSERT(_targetToRow.size() == _targets.size());
		for(int i=0; i<_targets.size(); i++) {
			if(_targets[i] == source) {
				// Update a single item.
				_model->updateItem(_targetToRow[i]);
			}
		}
	}
	return ParameterUI::onRefTargetMessage(source, msg);
}

/******************************************************************************
* Returns the data stored under the given role for the given RefTarget.
* Calls the RefTargetListParameterUI::getItemData() virtual method to obtain
* the display data.
******************************************************************************/
QVariant RefTargetListParameterUI::ListViewModel::data(const QModelIndex& index, int role) const
{
	if(!index.isValid())
		return QVariant();

	if(index.row() >= getOwner()->_rowToTarget.size())
		return QVariant();

	int targetIndex = getOwner()->_rowToTarget[index.row()];
	OVITO_ASSERT(targetIndex < getOwner()->_targets.size());

	RefTarget* t = getOwner()->_targets[targetIndex];
	return getOwner()->getItemData(t, index, role);
}

/******************************************************************************
* Returns the data stored under the given role for the given RefTarget.
******************************************************************************/
QVariant RefTargetListParameterUI::getItemData(RefTarget* target, const QModelIndex& index, int role)
{
	if(role == Qt::DisplayRole) {
		if(target == NULL) return "";
		return target->schematicTitle();
	}
	else return QVariant();
}

};

