///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __OVITO_NAVIGATION_MODES_H
#define __OVITO_NAVIGATION_MODES_H

#include <core/Core.h>
#include <core/gui/ApplicationManager.h>
#include "ViewportInputHandler.h"

namespace Core {

/**
 * \brief Base class for viewport navigation modes likes zoom, pan and orbit.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT NavigationMode : public ViewportInputHandler
{
	Q_OBJECT
	
public:

	/// \brief Returns the activation behaviour of this input handler.
	///
	/// Viewport navigation modes are always temporary.
	virtual InputHandlerType handlerActivationType() { return TEMPORARY; }

	/// \brief Handles the mouse down event for the given viewport.
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse up event for the given viewport.
	virtual void onMouseUp(Viewport& vp, QMouseEvent* event);

	/// \brief Handles the mouse move event for the given viewport.
	virtual void onMouseMove(Viewport& vp, QMouseEvent* event);

protected:

	/// Protected constructor.
	NavigationMode() : viewport(NULL) {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos) {}

	/// Computes the new view zoom based on the new mouse position.
	virtual void modifyZoom(Viewport& vp, const QPoint& currentPos) {}

	/// \brief This is called by the system after the input handler is
	///        no longer the active handler.
	virtual void onDeactivated();
	
protected:

	/// Mouse position at first click.
	QPoint startPoint;

	/// The old view transformation matrix.
	AffineTransformation oldViewMatrix;

	/// The old zoom factor.
	FloatType oldZoom;
	
	/// The current viewport we are working in.
	Viewport* viewport;	
};

/******************************************************************************
* The pan viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT PanMode : public NavigationMode 
{
	Q_OBJECT
	
protected:
	
	/// \brief Protected constructor to prevent the creation of second instances.
	PanMode() : NavigationMode() {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_pan.png")); }

public:

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static PanMode* instance() { 
		static intrusive_ptr<PanMode> instance(new PanMode());
		return instance.get(); 
	}
};


/******************************************************************************
* The zoom viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT ZoomMode : public NavigationMode 
{
	Q_OBJECT

protected:

	/// \brief Protected constructor to prevent the creation of second instances.
	ZoomMode() : NavigationMode() {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Computes the new view zoom based on the new mouse position.
	virtual void modifyZoom(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_zoom.png")); }

public:

	/// Zooms the viewport in or out.
	void Zoom(Viewport& vp, FloatType steps);

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static ZoomMode* instance() { 
		static intrusive_ptr<ZoomMode> instance(new ZoomMode());
		return instance.get(); 
	}
};

/******************************************************************************
* The field of view input mode.
******************************************************************************/
class CORE_DLLEXPORT FOVMode : public NavigationMode
{
	Q_OBJECT

protected:

	/// \brief Protected constructor to prevent the creation of second instances.
	FOVMode() : NavigationMode() {}

	/// Computes the new view zoom based on the new mouse position.
	virtual void modifyZoom(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_fov.png")); }

public:

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static FOVMode* instance() {
		static intrusive_ptr<FOVMode> instance(new FOVMode());
		return instance.get();
	}
};

/******************************************************************************
* The orbit viewport input mode.
******************************************************************************/
class CORE_DLLEXPORT OrbitMode : public NavigationMode 
{
	Q_OBJECT
	
public:

	enum CenterMode {
		ORBIT_CONSTRUCTION_PLANE,	/// Take the intersection point of the viewport and the current consstruction plane as orbit center.
		ORBIT_SELECTION_CENTER,		/// The the center of mass fo the current selection as orbit center.
		ORBIT_USER_DEFINED			/// Use the orbit center set by the user.
	};

	/// Changes the way the center of rotation is chosen.
	void setCenterMode(CenterMode mode);

	/// Returns the current center of orbit mode.
	CenterMode centerMode() const { return _centerMode; }

	/// Sets the world space point around which the camera orbits.
	void setOrbitCenter(const Point3& center);

	/// Returns the world space point around which the camera orbits.
	const Point3& orbitCenter() const { return _orbitCenter; }

	/// \brief Lets the input mode render its overlay content in a viewport.
	virtual void renderOverlay(Viewport* vp, bool isActive);

	/// \brief Indicates whether this input mode renders into the viewports.
	virtual bool hasOverlay() { return true; }

protected:

	/// \brief Protected constructor to prevent the creation of second instances.
	OrbitMode() : NavigationMode(), _centerMode(ORBIT_SELECTION_CENTER), _orbitCenter(ORIGIN) {}

	/// Computes the new view matrix based on the new mouse position.
	virtual void modifyViewMatrix(Viewport& vp, const QPoint& currentPos);

	/// Gets the mouse cursor of this viewport mode.
	virtual QCursor getCursor() { return QCursor(QPixmap(":/core/viewport/cursor_orbit.png")); }

	/// \brief Handles the mouse down event for the given viewport.
	virtual void onMouseDown(Viewport& vp, QMouseEvent* event);

protected:

	/// Contains the current orbiting center.
	Point3 _orbitCenter;
	
	/// Indicates around which point the camera should orbit.
	CenterMode _centerMode;

public:

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static OrbitMode* instance() { 
		static intrusive_ptr<OrbitMode> instance(new OrbitMode());
		return instance.get(); 
	}
};

/******************************************************************************
* This input mode lets the user pick the center of rotation for the orbit mode.
******************************************************************************/
class CORE_DLLEXPORT PickOrbitCenterMode : public SimpleInputHandler
{
public:

	/// Constructor.
	PickOrbitCenterMode() : showCursor(false) {
		if(APPLICATION_MANAGER.guiMode())
			hoverCursor = QCursor(QPixmap(":/core/main/cursor_mode_select.png"));
	}

	/// Returns the activation behaviour of this input handler.
	virtual InputHandlerType handlerActivationType() { return ViewportInputHandler::NORMAL; }

	/// Handles the mouse click event for a Viewport.
	virtual void onMousePressed(QMouseEvent* event);

	/// Is called when the user moves the mouse without pressing the button.
	virtual void onMouseFreeMove(Viewport& vp, QMouseEvent* event);

	/// Gets the current cursor of this viewport mode.
	virtual QCursor getCursor() {
		if(showCursor)
			return hoverCursor;
		else
			return SimpleInputHandler::getCursor();
	}

	/// \brief Lets the input mode render its overlay content in a viewport.
	virtual void renderOverlay(Viewport* vp, bool isActive) { OrbitMode::instance()->renderOverlay(vp, isActive); }

	/// \brief Indicates whether this input mode renders into the viewports.
	virtual bool hasOverlay() { return true; }

	/// \brief Returns the instance of this class.
	/// \return A pointer to the global instance of this singleton class.
	static PickOrbitCenterMode* instance() {
		static intrusive_ptr<PickOrbitCenterMode> instance(new PickOrbitCenterMode());
		return instance.get();
	}

private:

	/// Finds the intersection point between a ray originating from the current mouse cursor position and the scene.
	bool findIntersection(Viewport* vp, const Point2I& mousePos, Point3& intersectionPoint);

	/// The mouse cursor that is shown when over an object.
	QCursor hoverCursor;

	/// Indicates that the mouse cursor is over an object.
	bool showCursor;
};


};

#endif // __OVITO_NAVIGATION_MODES_H
