/*	Selection

PIRL CVS ID: Selection.java,v 1.13 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.PVL;

import	java.util.Vector;
import	java.util.Iterator;
import	java.util.NoSuchElementException;
import	java.util.regex.PatternSyntaxException;

/**	A <I>Selection</I> implements a <I>Selector</I> for use with a
	<I>Parameter</I> or <I>Value</I> wherever selective comparison of
	Parameters and/or Values is needed.
<P>
	For example, both the Parameter and Value <CODE>Find</CODE> methods
	use a Selector when making object comparisons, and use this
	Selection class when no other Selector is specified. A Selection can
	be used to specify the criteria to be used in comparing all, or any
	part, of two Parameters or Values.
<P>
	The criteria are specified as the characteristics of the Parameter
	and/or Value to use in making a selection, and the combinatorial
	boolean logic to apply to the characteristics when comparing one
	object against another. The Selector interface defines the criteria
	symbols. The Selection class provides the definition of the criteria
	and comparison methods.
<P>
	@author		Bradford Castalia, UA/PIRL
	@version	1.13 
*/
public class Selection
	implements Selector
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.PVL.Selection (1.13 2012/04/16 06:14:23)";

private int
	_Criteria_					= ANY | SPECIFIC | EQUAL,
	_Last_Parameter_Criteria_	= 0,
	_Last_Value_Criteria_		= 0;

private static final String
	NL							= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_PARAMETERS	= 1 << 1,
	DEBUG_NAME			= 1 << 2,
	DEBUG_VALUES		= 1 << 3,
	DEBUG_CLASS			= 1 << 4,
	DEBUG_TYPES			= 1 << 5,
	DEBUG_BASES			= 1 << 6,
	DEBUG_UNITS			= 1 << 7,
	DEBUG_DATA			= 1 << 8,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Creates a Selection using <CODE>ANY</CODE> criteria.
<P>
	This Selection will match anything; i.e. any two Parameters or
	Values will always match with this Selection.
*/
public Selection () {}

/**	Creates a Selection using the specified criteria.
<P>
	The criteria code may be any combination of Selector criteria.
<P>
	@param	criteria	A Selector criteria code value.
	@see	Selector
*/
public Selection (int criteria)
{_Criteria_ = criteria;}

/**	Creates a Selection using the criteria from the specified Selector.
<P>
	@param	selector	The Selector from which to get the criteria to use.
*/
public Selection (Selector selector)
{this(selector.Criteria ());}

/*------------------------------------------------------------------------------
	Accessors
*/
/**	Sets Selection criteria to the specified code.
<P>
	@param	criteria	The Selector criteria to use.
	@return	This Selection.
	@see	Selector
*/
public Selector Criteria (int criteria)
{_Criteria_ = criteria; return this;}

/**	Gets the current Selection criteria code.
<P>
	@return	The current Selection criteria code.
*/
public int Criteria ()
{return _Criteria_;}

/**	Gets the current Parameter Selection criteria code.
<P>
	Only that portion of the current Selection criteria code for the
	<CODE>PARAMETER_SELECTION</CODE> is returned; all other criteria
	are masked out.
<P>
	@return	The <CODE>PARAMETER_SELECTION</CODE> section of the
		current Selection criteria.
	@see	Selector#PARAMETER_SELECTION
*/
public int Parameter_Criteria ()
{return _Criteria_ & PARAMETER_SELECTION;}

/**	Gets the current Value Selection criteria code.
<P>
	Only that portion of the current Selection criteria code for the
	<CODE>VALUE_SELECTION</CODE> is returned; all other criteria are
	masked out.
<P>
	@return	The <CODE>VALUE_SELECTION</CODE> section of the current
		Selection criteria.
	@see	Selector#VALUE_SELECTION
*/
public int Value_Criteria ()
{return _Criteria_ & VALUE_SELECTION;}

/**	Gets the modifiers of the current Selection criteria code.
<P>
	Only that portion of the current Selection criteria code for the
	<CODE>MODIFIERS</CODE> is returned; all other criteria are
	masked out.
<P>
	@return	The <CODE>MODIFIERS</CODE> section of the current
		Selection criteria.
	@see	Selector#MODIFIERS
*/
public int Modifiers ()
{return _Criteria_ & MODIFIERS;}

/**	Gets the logic settings of the current Selection criteria code.
<P>
	Only that portion of the current Selection criteria code for the
	<CODE>LOGIC</CODE> is returned; all other criteria are
	masked out.
<P>
	@return	The <CODE>LOGIC</CODE> section of the current
		Selection criteria.
	@see	Selector#LOGIC
*/
public int Logic ()
{return _Criteria_ & LOGIC;}

/**	Gets a description of the Selection criteria.
*/
public String toString ()
{
String
	description = "PIRL.PVL.Selection Criteria "
		+ Integer.toString (_Criteria_, 2);
description += NL
	+ "    Parameter (" + Integer.toString (Parameter_Criteria (), 2) + ") -";
if (Any_Parameter ())
	description += " Any";
if (Comments ())
	description += " Comments";
if (Name ())
	description += " Name";
if (Classification ())
	description += " Classification";
if (Value ())
	description += " Value";

description += NL
	+ "    Value (" + Integer.toString (Value_Criteria (), 2) + ") -";
if (Any_Value ())
	description += " Any";
if (Data ())
	description += " Data";
if (Type ())
	description += " Type";
if (Base ())
	description += " Base";
if (Units ())
	description += " Units";

description += NL
	+ "    Modifiers (" + Integer.toString (Modifiers (), 2) + ") -";
if (Pattern_Match ())
	description += " Pattern";
if (Specific ())
	description += " Specific";

description += NL
	+ "    Logic (" + Integer.toString (Logic (), 2) + ") -";
if (Equal ())
	description += " Equal";
if (Less_Than ())
	description += " Less-than";
if (Greater_Than ())
	description += " Greater-than";
if (And ())
	description += " And";

return description;
}

/*..............................................................................
	Parameter criteria:
*/
/**	Enables or disables the Parameter name criteria.
<P>
	@param	mode	true to enable the use of the Parameter name as a
		match criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#NAME
*/
public Selector Name (boolean mode)
{
if (mode) _Criteria_ |=  NAME;
else      _Criteria_ &= ~NAME;
return this;
}

/**	Tests if the Parameter name criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Name(boolean)
*/
public boolean Name ()
{return ((_Criteria_ & NAME) != 0);}

/**	Enables or disables the Parameter classification criteria.
<P>
	@param	mode	true to enable the use of the Parameter
		classification as a match criteria; false to disable this
		criteria.
	@return	This Selection.
	@see	Selector#CLASSIFICATION
*/
public Selector Classification (boolean mode)
{
if (mode) _Criteria_ |=  CLASSIFICATION;
else      _Criteria_ &= ~CLASSIFICATION;
return this;
}

/**	Tests if the Parameter classification criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Classification(boolean)
*/
public boolean Classification ()
{return ((_Criteria_ & CLASSIFICATION) != 0);}

/**	Enables or disables the Parameter Value criteria.
<P>
	@param	mode	true to enable the use of the Parameter Value as a
		match criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#VALUE
*/
public Selector Value (boolean mode)
{
if (mode) _Criteria_ |=  VALUE;
else      _Criteria_ &= ~VALUE;
return this;
}

/**	Tests if the Parameter Value criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Value(boolean)
*/
public boolean Value ()
{return ((_Criteria_ & VALUE) != 0);}

/**	Enables or disables the Parameter comments criteria.
<P>
	@param	mode	true to enable the use of the Parameter comments as
		a match criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#COMMENTS
*/
public Selector Comments (boolean mode)
{
if (mode) _Criteria_ |=  COMMENTS;
else      _Criteria_ &= ~COMMENTS;
return this;
}

/**	Tests if the Parameter comments criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Comments(boolean)
*/
public boolean Comments ()
{return ((_Criteria_ & COMMENTS) != 0);}

/**	Enables or disables using any Parameter characteristic to match.
<P>
	Enabling this mode will disable all Parameter criteria, which is a
	special case interpreted to mean that any Parameter characteristic
	matches with those of another Parameter. If, however,
	<CODE>AND</CODE> logic is enabled then the Value matching criteria
	must still be met. Disabling this mode has no effect; specific
	Parameter criteria must be enabled once <CODE>Any_Parameter</CODE>
	has been used.
<P>
	@param	mode	true to enable any Parameter characteristics
		to match; false leaves the current criteria unchanged.
	@return	This Selection.
	@see	Selector#PARAMETER_SELECTION
*/
public Selector Any_Parameter (boolean mode)
{
if (mode) _Criteria_ &= ~PARAMETER_SELECTION;
return this;
}

/**	Tests if any Parameter characteristic will match.
<P>
	@return	true if no Parameter criteria are enabled; false otherwise.
	@see	#Any_Parameter(boolean)
*/
public boolean Any_Parameter ()
{return ((_Criteria_ & PARAMETER_SELECTION) == ANY ||
         (_Criteria_ & ANY_PARAMETER) != 0);}

/*..............................................................................
	Value criteria:
*/
/**	Enables or disables using the data of a Value as a criteria.
<P>
	@param	mode	true to enable the use of Value data as a match
		criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#DATA
*/
public Selector Data (boolean mode)
{
if (mode) _Criteria_ |=  DATA;
else      _Criteria_ &= ~DATA;
return this;
}

/**	Tests if the data of a Value is a criteria.
<P>
	@return	true if the data of a Value is a criteria; false otherwise.
	@see	#Data(boolean)
*/
public boolean Data ()
{return ((_Criteria_ & DATA) != 0);}

/**	Enables or disables the Value type criteria.
<P>
	@param	mode	true to enable the use of the Value type as
		a match criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#TYPE
*/
public Selector Type (boolean mode)
{
if (mode) _Criteria_ |=  TYPE;
else      _Criteria_ &= ~TYPE;
return this;
}

/**	Tests if the Value type criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Type(boolean)
*/
public boolean Type ()
{return ((_Criteria_ & TYPE) != 0);}

/**	Enables or disables the Value integer base criteria.
<P>
	@param	mode	true to enable the use of the Value integer base as
		a match criteria; false to disable this criteria.
	@return	This Selection.
	@see	Selector#BASE
*/
public Selector Base (boolean mode)
{
if (mode) _Criteria_ |=  BASE;
else      _Criteria_ &= ~BASE;
return this;
}

/**	Tests if the Value integer base criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Base(boolean)
*/
public boolean Base ()
{return ((_Criteria_ & BASE) != 0);}

/**	Enables or disables the Value units description criteria.
<P>
	@param	mode	true to enable the use of the Value units
		description as a match criteria; false to disable this
		criteria.
	@return	This Selection.
	@see	Selector#UNITS
*/
public Selector Units (boolean mode)
{
if (mode) _Criteria_ |=  UNITS;
else      _Criteria_ &= ~UNITS;
return this;
}

/**	Tests if the Value units description criteria is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Units(boolean)
*/
public boolean Units ()
{return ((_Criteria_ & UNITS) != 0);}

/**	Enables or disables using any Value characteristic to match.
<P>
	Enabling this mode will disable all Value criteria, which is a
	special case interpreted to mean that any Value characteristic
	matches with those of another Value. Disabling this mode has no
	effect; specific Value criteria must be enabled once
	<CODE>Any_Value</CODE> has been used.
<P>
	@param	mode	true to enable any Value characteristics to match;
		false leaves the current criteria unchanged.
	@return	This Selection.
	@see	Selector#VALUE_SELECTION
*/
public Selector Any_Value (boolean mode)
{
if (mode)_Criteria_ &= ~VALUE_SELECTION;
return this;
}

/**	Tests if any Value characteristic will match.
<P>
	@return	true if no Value criteria are enabled; false otherwise.
	@see	#Any_Value(boolean)
*/
public boolean Any_Value ()
{return ((_Criteria_ & VALUE_SELECTION) == ANY ||
         (_Criteria_ & ANY_VALUE) != 0);}

/*..............................................................................
	Criteria modifiers:
*/
/**	Enables or disables using regular expression pattern matching for
	String comparisons.
<P>
	@param	mode	true to enable pattern matching; false otherwise.
	@return	This Selection.
	@see	Selector#PATTERN_MATCH
*/
public Selector Pattern_Match (boolean mode)
{
if (mode) _Criteria_ |=  PATTERN_MATCH;
else      _Criteria_ &= ~PATTERN_MATCH;
return this;
}

/**	Tests if regular expression pattern matching is enabled.
<P>
	@return	true if the criteria is enabled; false otherwise.
	@see	#Pattern_Match(boolean)
*/
public boolean Pattern_Match ()
{return ((_Criteria_ & PATTERN_MATCH) != 0);}

/**	Enables or disables specific criteria matching.
<P>
	When specific matching is enabled Parameter classifications and
	Value types must match exactly; when disabled only the general
	categories need to be the same for their criteria to be met. When
	specific matching is enabled Strings are compared using their
	<CODE>equals</CODE> methods; when disabled their
	<CODE>equalsIgnoreCase</CODE> methods are used. When specific
	matching is enabled aggregate Parameter lists are compared in their
	entirerty; when disabled <CODE>END</CODE> Parameters are recognized
	as the end of a list and an <CODE>END_PVL</CODE> Parameter marks
	the end of all nested lists.
<P>
	@param	mode	true to require criteria to match exactly; false
		will allow criteria to match when the characteristics are
		generally the same.
	@return	This Selection.
	@see	Selector#SPECIFIC
*/
public Selector Specific (boolean mode)
{
if (mode) _Criteria_ |=  SPECIFIC;
else      _Criteria_ &= ~SPECIFIC;
return this;
}

/**	Tests if specific criteria matching is enabled.
<P>
	@return	true if specific criteria matching is enabled; false
		otherwise.
	@see	#Specific(boolean)
*/
public boolean Specific ()
{return ((_Criteria_ & SPECIFIC) != 0);}

/*..............................................................................
	Criteria logic:
*/
/**	Enables or disables using equality logic for comparing Values with
	<CODE>NUMERIC</CODE> data.
<P>
	@param	mode	true to use equality logic when comparing numeric
		data values.
	@return	This Selection.
	@see	Selector#SPECIFIC
*/
public Selector Equal (boolean mode)
{
if (mode) _Criteria_ |=  EQUAL;
else      _Criteria_ &= ~EQUAL;
return this;
}

/**	Tests if equality matching logic is enabled.
<P>
	@return	true if equality matching logic is enabled; false
		otherwise.
	@see	#Specific(boolean)
*/
public boolean Equal ()
{return ((_Criteria_ & EQUAL) != 0);}

/**	Enables or disables using less-than logic for comparing Values
	with <CODE>NUMERIC</CODE> data.
<P>
	If <CODE>Equal</CODE> is also enabled, then <= logic will be used.
	<B>Note</B>: This method disables <CODE>Greater_Than</CODE> logic.
<P>
	@param	mode	true to use less-than logic when comparing numeric
		data values.
	@return	This Selection.
	@see	Selector#LESS_THAN
*/
public Selector Less_Than (boolean mode)
{
if (mode) _Criteria_ |=  LESS_THAN;
else      _Criteria_ &= ~LESS_THAN;
Greater_Than (false);
return this;
}

/**	Tests if less-than matching logic is enabled.
<P>
	@return	true if less-than matching logic is enabled; false
		otherwise.
	@see	#Less_Than(boolean)
*/
public boolean Less_Than ()
{return ((_Criteria_ & LESS_THAN) != 0);}

/**	Enables or disables using greater-than logic for comparing Values
	with <CODE>NUMERIC</CODE> data.
<P>
	If <CODE>Equal</CODE> is also enabled, then >= logic will be used;
	however, <CODE>Less_Than</CODE> takes precedence if they are both
	enabled. <B>Note</B>: This method disables <CODE>Less_Than</CODE>
	logic.
<P>
	@param	mode	true to use less-than logic when comparing numeric
		data values.
	@return	This Selection.
	@see	Selector#LESS_THAN
*/
public Selector Greater_Than (boolean mode)
{
if (mode) _Criteria_ |=  GREATER_THAN;
else      _Criteria_ &= ~GREATER_THAN;
Less_Than (false);
return this;
}

/**	Tests if greater-than matching logic is enabled.
<P>
	@return	true if greater-than matching logic is enabled; false
		otherwise.
	@see	#Greater_Than(boolean)
*/
public boolean Greater_Than ()
{return ((_Criteria_ & GREATER_THAN) != 0);}

/**	Enables or disables logical AND when matching with multiple
	criteria.
<P>
	<B>Note</B>: When disabled, logical OR is used when matching with
	multiple criteria.
<P>
	@param	mode	true to use logical AND when matching with multiple
		criteria; false to use logical OR.
	@return	This Selection.
	@see	Selector#AND
*/
public Selector And (boolean mode)
{
if (mode) _Criteria_ |= AND;
else      _Criteria_ &= ~AND;
return this;
}

/**	Tests if logical AND for multiple criteria is enabled.
<P>
	@return	true if AND logic is enabled; false if OR logic will be
		used.
	@see	#And(boolean)
*/
public boolean And ()
{return ((_Criteria_ & AND) != 0);}

/*------------------------------------------------------------------------------
	Parameter criteria match methods:
*/
/**	Tests if two Parameters match.
<P>
	If both arguments are null, they match; but if one or the
	other is null, they do not match.
<P>
	When <CODE>Any_Parameter</CODE> is enabled, a match occurs. However,
	if <CODE>And</CODE> is also enabled and <CODE>Any_Value</CODE> is
	not enabled, then a match only occurs if both Parameters are the
	<CODE>ASSIGNMENT</CODE> classification and their
	<CODE>Values_Match</CODE>.
<P>
	The following criteria are tested in the order listed:
<P>
<OL>
<LI><CODE>Classification</CODE> - <CODE>Classifications_Match</CODE>
<LI><CODE>Value</CODE> - <CODE>Parameter_Values_Match</CODE>
<LI><CODE>Comments</CODE> - <CODE>Comments_Match</CODE>
<LI><CODE>Name</CODE> - <CODE>Names_Match</CODE>
</OL>
<P>
	When <CODE>And</CODE> logic is enabled, all of these criteria tests
	must match for the Parameters to match (the first one that does not
	match stops further testing); otherwise any criteria test that
	matches produces a match for the Parameters (the first one that
	matches stops further testing).
<P>
	<B>N.B.</B>: When the associativity of comparison testing matters -
	e.g.  - the associativity is this_parameter op that_parameter. For
	example, when a Parameter name is being compared against a regular
	expression pattern the comparison is this_parameter.Name ().matches
	(that_parameter.Name ()); i.e. the second Parameter
	(that_parameter) is expected to contain the pattern to be matched.
<P>
	<B>N.B.</B>: Parameter matching of Aggregates is only recursive
	for <CODE>Value</CODE> matching, and in this case only the Values
	of the Parameter lists are compared.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	true if the Parameters match; false otherwise.
	@see	#And(boolean)
	@see	#Any_Parameter(boolean)
	@see	#Any_Value(boolean)
	@see	Parameter#ASSIGNMENT
	@see	#Values_Match(Value, Value)
	@see	#Classification(boolean)
	@see	#Classifications_Match(Parameter, Parameter)
	@see	#Value(boolean)
	@see	#Parameter_Values_Match(Parameter, Parameter)
	@see	#Comments(boolean)
	@see	#Comments_Match(Parameter, Parameter)
	@see	#Name(boolean)
	@see	#Names_Match(Parameter, Parameter)
*/
public boolean Parameters_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		(">>> Parameters_Match:\n"
		+ this_parameter + '\n'
		+ that_parameter + '\n'
		+ "    criteria = " + Integer.toString (Criteria (), 2) + "\n"
		+ "    Name           = " + Name () + "\n"
		+ "    Classification = " + Classification () + "\n"
		+ "    Comments       = " + Comments () + "\n"
		+ "    Value          = " + Value () + "\n"
		+ "    Data           = " + Data () + "\n"
		+ "    Type           = " + Type () + "\n"
		+ "    Base           = " + Base () + "\n"
		+ "    Units          = " + Units () + "\n"
		+ "    Specific       = " + Specific () + "\n"
		+ "    Pattern_Match  = " + Pattern_Match () + "\n"
		+ "    AND            = " + And ()
		);
boolean
	matched = false;

_Last_Parameter_Criteria_ = 0;

if (this_parameter == that_parameter)
	{
	_Last_Parameter_Criteria_ =
		ANY_PARAMETER | NAME | CLASSIFICATION | VALUE | COMMENTS;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    Same parameter\n"
			+"<<< Parameters_Match: true");
	return true;
	}
if (this_parameter == null ||
	that_parameter == null)
	{
	//	A Single null matches nothing.
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    One parameter is null\n"
			+"<<< Parameters_Match: false");
	return false;
	}

if (Any_Parameter ())
	{
	if (And () && ! Any_Value ())
		{
		//	A match against a Value is required.
		if (this_parameter.Is_Assignment () &&
			that_parameter.Is_Assignment ())
			{
			try
				{
				matched = Values_Match
					(this_parameter.Value (),
					 that_parameter.Value ());
				}
			catch (PVL_Exception exception)
				{
				//	Shouldn't happen since they're both assignments.
				matched = false;
				}
			}
		else
			//	Non-assignments can't match when a Value is required.
			matched = false;
		}
	else
		matched = true;

	_Last_Parameter_Criteria_ |= ANY_PARAMETER;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println
			("    Any_Parameter = " + matched + '\n'
			+"<<< Parameters_Match: " + matched);
	return matched;
	}

if (Classification ())
	{
	matched = Classifications_Match (this_parameter, that_parameter);
	if (matched)
		_Last_Parameter_Criteria_ |= CLASSIFICATION;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println ("    Classifications_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & DEBUG_PARAMETERS) != 0)
				System.out.println
					("<<< Parameters_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & DEBUG_PARAMETERS) != 0)
			System.out.println
				("<<< Parameters_Match: true");
		return true;		//	OR: Definate match.
		}
	}

if (Value ())
	{
	matched = Parameter_Values_Match
		(this_parameter,
		 that_parameter);
	if (matched)
		_Last_Parameter_Criteria_ |= VALUE;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println ("    Parameter_Values_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & DEBUG_PARAMETERS) != 0)
				System.out.println
					("<<< Parameters_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & DEBUG_PARAMETERS) != 0)
			System.out.println
				("<<< Parameters_Match: true");
		return true;		//	OR: Definate match.
		}
	}

if (Comments ())
	{
	matched = Comments_Match
		(this_parameter,
		 that_parameter);
	if (matched)
		_Last_Parameter_Criteria_ |= COMMENTS;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println ("    Comments_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & DEBUG_PARAMETERS) != 0)
				System.out.println
					("<<< Parameters_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & DEBUG_PARAMETERS) != 0)
			System.out.println
				("<<< Parameters_Match: true");
		return true;		//	OR: Definate match.
		}
	}

/*	Name criteria:

	The name match is done last so that "simpler" matches may come
	first for OR logic. In the Parameter.Find method pathname tracking
	requires name matches to succeed for the entire hierarchy of names
	that may presented to Find. If a match to another criteria is
	possible, then that should be given preference to avoid the
	unnecessary searching down into aggregate parameter lists that may
	be required for a name match.
*/
if (Name ())
	{
	matched = Names_Match
		(this_parameter,
		 that_parameter);
	if (matched)
		_Last_Parameter_Criteria_ |= NAME;
	if ((DEBUG & DEBUG_PARAMETERS) != 0)
		System.out.println ("    Names_Match = " + matched);
	}

if ((DEBUG & DEBUG_PARAMETERS) != 0)
	System.out.println
		("<<< Parameters_Match: " + matched);
return matched;
}

/**	Gets the selection criteria that resulted from the last
	<CODE>Parameters_Match</CODE> comparison of two Parameters.
<P>
	The Selector criteria code that is returned flags the criteria that
	were matched during the last use of the {@link
	#Parameters_Match(Parameter, Parameter)
	<CODE>Parameters_Match</CODE>} method with this Selection object. If
	no match occured the return value will be zero.
<P>
	@return	A Selector criteria code.
*/
public int Parameters_Match ()
{return _Last_Parameter_Criteria_;}

/**	Gets the matching criteria for two Parameters.
<P>
	The same matching logic as used in the <CODE>Parameters_Match</CODE>
	method is applied here, except that all criteria are tested. Thus
	the Selector criteria code that is returned indicates all matching
	criteria. When the <CODE>Value</CODE> criteria is enabled and the
	Parameters are the <CODE>ASSIGNMENT</CODE> classification, then the
	<CODE>Values_Criteria_Match</CODE> method is used to also get the
	matching criteria for the Parameters' Values. If no criteria match
	the return value will be zero.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	A Selector criteria code.
	@see	#Parameters_Match(Parameter, Parameter)
	@see	#Values_Criteria_Match(Value, Value)
*/
public int Parameters_Criteria_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
int
	selection = 0;

if (this_parameter == that_parameter)
	//	Same parameter.
	return
		ANY_PARAMETER | NAME | CLASSIFICATION | VALUE | COMMENTS;
if (this_parameter == null ||
	that_parameter == null)
	//	A Single null matches nothing.
	return 0;

if (Any_Parameter ())
	{
	selection |= ANY_PARAMETER;
	if (And () && ! Any_Value ())
		{
		//	A match against a Value is required.
		if (this_parameter.Is_Assignment () &&
			that_parameter.Is_Assignment ())
			{
			try
				{
				selection |= Values_Criteria_Match
					(this_parameter.Value (),
					 that_parameter.Value ());
				if (selection != ANY_PARAMETER)
					selection |= VALUE;
				}
			catch (PVL_Exception exception)
				{
				//	Shouldn't happen since they're both assignments.
				}
			}
		}
	return selection;
	}

if (Name () &&
	Names_Match
		(this_parameter,
		 that_parameter))
	selection |= NAME;

if (Classification () &&
	Classifications_Match
		(this_parameter,
		 that_parameter))
	selection |= CLASSIFICATION;

if (Value ())
	{
	if (this_parameter.Is_Assignment () &&
		that_parameter.Is_Assignment ())
		{
		try
			{
			int
				values_selection = Values_Criteria_Match
					(this_parameter.Value (),
					 that_parameter.Value ());
			if (values_selection != 0)
				selection |= values_selection | VALUE;
			}
		catch (PVL_Exception exception)
			{
			//	Shouldn't happen since they're both assignments.
			}
		}
	//	Possible aggregates.
	else if (Parameter_Values_Match
			(this_parameter,
			 that_parameter))
		selection |= VALUE;
	}

if (Comments () &&
	Comments_Match
		(this_parameter,
		 that_parameter))
	selection |= COMMENTS;

return selection;
}

/**	Tests if the names of two Parameters match.
<P>
	If <CODE>Pattern_Match</CODE>ing is enabled, then the Name of
	this_parameter is compared against the regular expression pattern
	in the Name of that_parameter. If the regular expression syntax is
	invalid a normal String comparision is done. If
	<CODE>Specific</CODE> matching is enabled, then the String
	<CODE>equals</CODE> method is used; otherwise the
	<CODE>equalsIgnoreCase</CODE> method is used.
<P>
	<B>N.B.</B>: If either Parameter is null the match is false.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	true if the Parameter names match; false otherwise.
	@see	String#matches(String)
*/
public boolean Names_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
if (this_parameter == null ||
	that_parameter == null)
	return false;
if (this_parameter.Name () == null ||
	that_parameter.Name () == null)
	return false;

try
	{
	if (Pattern_Match ())
		return this_parameter.Name ().matches
			  (that_parameter.Name ());
	}
catch (PatternSyntaxException exception)
	{/* Fall through to regular string matching */}

if (Specific ())
	return this_parameter.Name ().equals
		  (that_parameter.Name ());

return this_parameter.Name ().equalsIgnoreCase
	  (that_parameter.Name ());
}

/**	Tests if the classifications of two Parameters match.
<P>
	If <CODE>Specific</CODE> matching is enabled, then the classifications
	must be identical to match; otherwise general classifications
	(i.e. <CODE>AGGREGATE</CODE>) will also match.
<P>
	<B>N.B.</B>: If either Parameter is null the match is false.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	true if the Parameter classifications match; false otherwise.
*/
public boolean Classifications_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
if (this_parameter == null ||
	that_parameter == null)
	return false;
if (this_parameter.Classification () ==
	that_parameter.Classification ())
	return true;
if (! Specific () &&
	((this_parameter.Is_Begin_Aggregate () &&
	  that_parameter.Is_Begin_Aggregate ()) ||
	 (this_parameter.Is_End_Aggregate () &&
	  that_parameter.Is_End_Aggregate ())))
	return true;
return false;
}

/**	Tests if the data values of two Parameters match.
<P>
	If both Parameters have null data values, they match; but if one or
	the other is null, they do not match.
<P>
	For an <CODE>ASSIGNMENT</CODE>, <CODE>Values_Match</CODE> determines
	the match.
<P>
	For <CODE>AGGREGATE</CODE> Parameters, all of the Parameters in the
	aggregate lists must match, using the current Selection criteria,
	for a match to occur. However, if <CODE>Specific</CODE> matching is
	not enabled, then <CODE>END</CODE> Parameters are recognized as
	ending lists, and an <CODE>END_PVL</CODE> Parameter will stop any
	further comparisons for all nested aggregate lists. If the
	<CODE>Value</CODE> criteria applies then the match is recursive for
	all Aggregate Parameters. <CODE>TOKEN</CODE> and
	<CODE>UNKNOWN</CODE> Parameter Values always match when their
	classifications are the same.
<P>
	<B>N.B.</B>: If either Parameter is null the match is false.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	true if the Parameter data values match; false otherwise.
	@see	#Values_Match(Value, Value)
	@see	#Specific(boolean)
*/
public boolean Parameter_Values_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
boolean
	end[] = {false};
return parameter_values_match
	(this_parameter,
	 that_parameter, end);
}


private boolean parameter_values_match
	(
	Parameter	this_parameter,
	Parameter	that_parameter,
	boolean[]	end
	)
{
if (this_parameter == null ||
	that_parameter == null)
	return false;
if ((DEBUG & DEBUG_VALUES) != 0)
	System.out.println
		(">>> parameter_values_match:\n"
		+ this_parameter + '\n'
		+ that_parameter);
if (this_parameter.Data () == null)
	{
	if (that_parameter.Data () == null)
		{
		if ((DEBUG & DEBUG_VALUES) != 0)
			System.out.println
				("    Both null Data\n"
				+"<<< parameter_values_match: true");
		return true;
		}
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    One null Data\n"
			+"<<< parameter_values_match: false");
	return false;
	}
if (that_parameter.Data () == null)
	{
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    One null Data\n"
			+"<<< parameter_values_match: false");
	return false;
	}

boolean
	matches;
if (this_parameter.Is_Assignment () &&
	that_parameter.Is_Assignment ())
	{
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    ASSIGNMENT");
	try
		{
		//	N.B.: Values_Match sets _Last_Value_Criteria_.
		matches = Values_Match
			(this_parameter.Value (),
			 that_parameter.Value ());
		if ((DEBUG & DEBUG_VALUES) != 0)
			System.out.println
				("<<< parameter_values_match: " + matches);
		return matches;
		}
	catch (PVL_Exception exception)
		{
		//	Shouldn't happen since they're both assignments.
		return false;
		}
	}

if (this_parameter.Is_Aggregate () &&
	that_parameter.Is_Aggregate ())
	{
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    AGGREGATE");
	matches = true;
	Iterator
		these_parameters,
		those_parameters;

	if (Specific ())
		{
		//	Compare the entire lists.
		if ((DEBUG & DEBUG_VALUES) != 0)
			System.out.println ("    Specific");
		try
			{
			if (this_parameter.List_Size () !=
				that_parameter.List_Size ())
				{
				if ((DEBUG & DEBUG_VALUES) != 0)
					System.out.println
						("    Different list sizes\n"
						+"<<< parameter_values_match: false");
				return false;
				}
			Vector
				this_list = this_parameter.List (),
				that_list = that_parameter.List ();
			if (this_list == null &&
				that_list == null)
				{
				if ((DEBUG & DEBUG_VALUES) != 0)
					System.out.println
						("    Empty lists\n"
						+"<<< parameter_values_match: true");
				return true;
				}
			these_parameters = this_list.iterator ();
			those_parameters = that_list.iterator ();
			}
		catch (PVL_Exception exception)
			{
			//	Shouldn't happen since they're both valid aggregates.
			return false;
			}
		}
	else
		{
		/*	END_XXX parameters will end the lists.

			END_PVL parameters will be specially noticed and the end flag
			will be set so that no further comparisons will be done up the
			recursion line. 
		*/
		if ((DEBUG & DEBUG_VALUES) != 0)
			System.out.println ("    ! Specific");
		these_parameters = this_parameter.iterator ();
		those_parameters = that_parameter.iterator ();
		}
	while (these_parameters.hasNext () &&
		   those_parameters.hasNext ())
		{
		if (! parameter_values_match
				(
				(Parameter)these_parameters.next (),
				(Parameter)those_parameters.next (),
				end))
			{
			matches = false;
			break;
			}
		if (end[0])
			{
			//	END_PVL in sub-parameter.
			if ((DEBUG & DEBUG_VALUES) != 0)
				System.out.println ("    parameter_values_match: END");
			break;
			}
		}
	if (! end[0] && ! Specific ())
		{
		if (these_parameters.hasNext () ||
			those_parameters.hasNext ())
			//	The lists are not the same size.
			matches = false;
		//	Test for END_PVL in either list.
		try
			{
			these_parameters.next ();
			those_parameters.next ();
			}
		catch (NoSuchElementException exception)
			{
			if (exception.getMessage ().startsWith
				(Parameter.Classification_Name (Parameter.END_PVL) + "\n"))
				{
				//	END_PVL in list.
				if ((DEBUG & DEBUG_VALUES) != 0)
					System.out.println ("    parameter_values_match: END_PVL");
				end[0] = true;
				}
			}
		}
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("<<< Selection.parameter_values_match: " + matches);
	return matches;
	}

if (this_parameter.Classification () ==
	that_parameter.Classification ())
	matches = true;
else
	matches = false;
if ((DEBUG & DEBUG_VALUES) != 0)
	System.out.println
		("    Classification matches = " + matches + '\n'
		+"<<< parameter_values_match: " + matches);
return matches;
}

/**	Tests if the comments of two Parameters match.
<P>
	If both Parameters do not have comments, they match; but if one or
	the other does not have comments, they do not match. If
	<CODE>Specific</CODE> matching is enabled, then the String
	<CODE>equals</CODE> method is used; otherwise the
	<CODE>equalsIgnoreCase</CODE> method is used.
<P>
	<B>N.B.</B>: If either Parameter is null the match is false.
<P>
	@param	this_parameter	The first of a pair of Parameters to be matched.
	@param	that_parameter	The second of a pair of Parameters to be matched.
	@return	true if the Parameter comments match; false otherwise.
*/
public boolean Comments_Match
	(
	Parameter	this_parameter,
	Parameter	that_parameter
	)
{
if (this_parameter == null ||
	that_parameter == null)
	return false;
if (this_parameter.Comments () == null)
	{
	if (that_parameter.Comments () == null)
		return true;
	return false;
	}
if (that_parameter.Comments () == null)
	return false;

if (Specific ())
	return this_parameter.Comments ().equals
		  (that_parameter.Comments ());
return this_parameter.Comments ().equalsIgnoreCase
	  (that_parameter.Comments ());
}

/*------------------------------------------------------------------------------
	Value criteria match methods:
*/
/**	Tests if two Values match.
<P>
	When <CODE>Any_Value</CODE> is enabled, a match always occurs.
<P>
	If both arguments are null, they match; but if one or the
	other is null, they do not match.
<P>
	The following criteria are tested in the order listed:
<P>
<OL>
<LI><CODE>Type</CODE> - <CODE>Types_Match</CODE>
<LI><CODE>Base</CODE> - <CODE>Bases_Match</CODE>
<LI><CODE>Units</CODE> - <CODE>Units_Match</CODE>
<LI><CODE>Data</CODE> - <CODE>Data_Match</CODE>
</OL>
<P>
	When <CODE>And</CODE> logic is enabled, all of the enabled criteria
	tests must match for the Values to match (the first one that does
	not match stops further testing); otherwise any criteria test that
	matches produces a match for the Values (the first one that matches
	stops further testing).
<P>
	<B>N.B.</B>: Value matching of Arrays is only recursive for
	<CODE>Data</CODE> matching, and in this case only the data values of
	the Array elements are compared.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	true if the Values match; false otherwise.
	@see	#And(boolean)
	@see	#Type(boolean)
	@see	#Types_Match(Value, Value)
	@see	#Base(boolean)
	@see	#Bases_Match(Value, Value)
	@see	#Units(boolean)
	@see	#Units_Match(Value, Value)
	@see	#Data(boolean)
	@see	#Data_Match(Value, Value)
*/
public boolean Values_Match
	(
	Value		this_value,
	Value		that_value
	)
{
if ((DEBUG & DEBUG_VALUES) != 0)
	System.out.println
		(">>> Values_Match:\n"
		+ "    criteria = " + Integer.toString (Criteria (), 2) + "\n"
		+ "    Data     = " + Data () + "\n"
		+ "    Type     = " + Type () + "\n"
		+ "    Base     = " + Base () + "\n"
		+ "    Units    = " + Units () + "\n"
		+ "    Specific = " + Specific () + "\n"
		+ "    AND      = " + And ()
		);
_Last_Value_Criteria_ = 0;
if (Any_Value ())
	{
	_Last_Value_Criteria_ = ANY_VALUE;
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    Any_Value = true" + '\n'
			+"<<< Values_Match: true");
	return true;
	}

if (this_value == that_value)
	{
	//	The same Values.
	_Last_Value_Criteria_ = ANY_VALUE | DATA | TYPE | BASE | UNITS;
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    Same value\n"
			+"<<< Values_Match: true");
	return true;
	}
if (this_value == null ||
	that_value == null)
	{
	if ((DEBUG & DEBUG_VALUES) != 0)
		System.out.println
			("    One value is null\n"
			+"<<< Values_Match: false");
	return false;
	}

boolean
	matched = false;

if (Type ())
	{
	matched = Types_Match (this_value, that_value);
	if (matched)
		_Last_Value_Criteria_ |= TYPE;
	if ((DEBUG & (DEBUG_VALUES | DEBUG_TYPES)) != 0)
		System.out.println ("    Types_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & (DEBUG_VALUES | DEBUG_TYPES)) != 0)
				System.out.println
					("<<< Values_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & (DEBUG_VALUES | DEBUG_TYPES)) != 0)
			System.out.println
				("<<< Values_Match: true");
		return true;		//	OR: Definate match.
		}
	}

if (Base ())
	{
	matched = Bases_Match (this_value, that_value);
	if (matched)
		_Last_Value_Criteria_ |= BASE;
	if ((DEBUG & (DEBUG_VALUES | DEBUG_BASES)) != 0)
		System.out.println ("    Bases_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & (DEBUG_VALUES | DEBUG_BASES)) != 0)
				System.out.println
					("<<< Values_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & (DEBUG_VALUES | DEBUG_BASES)) != 0)
			System.out.println
				("<<< Values_Match: true");
		return true;		//	OR: Definate match.
		}
	}

if (Units ())
	{
	matched = Units_Match (this_value, that_value);
	if (matched)
		_Last_Value_Criteria_ |= UNITS;
	if ((DEBUG & (DEBUG_VALUES | DEBUG_UNITS)) != 0)
		System.out.println ("    Units_Match = " + matched);

	if (And ())
		{
		if (! matched)
			{
			if ((DEBUG & (DEBUG_VALUES | DEBUG_UNITS)) != 0)
				System.out.println
					("<<< Values_Match: false");
			return false;	//	AND: Definate no match.
			}
		}
	else if (matched)
		{
		if ((DEBUG & (DEBUG_VALUES | DEBUG_UNITS)) != 0)
			System.out.println
				("<<< Values_Match: true");
		return true;		//	OR: Definate match.
		}
	}

if (Data ())
	{
	matched = Data_Match (this_value, that_value);
	if (matched)
		//	Data_Match may match other criteria.
		_Last_Value_Criteria_ |= DATA;
	if ((DEBUG & (DEBUG_VALUES | DEBUG_DATA)) != 0)
		System.out.println ("    Data_Match = " + matched);
	}

if ((DEBUG & DEBUG_VALUES) != 0)
	System.out.println
		("<<< Values_Match: " + matched);
return matched;
}

/**	Gets the selection criteria that resulted from the last
	<CODE>Values_Match</CODE> comparison of two Values.
<P>
	The Selector criteria code that is returned flags the criteria that
	were matched during the last use of the
	{@link #Values_Match(Value, Value) <CODE>Values_Match</CODE>} method
	with this Selection object. If no match occured the return value
	will be zero.
<P>
	@return	A Selector criteria code.
*/
public int Values_Match ()
{return _Last_Value_Criteria_;}

/**	Gets the matching criteria for two Values.
<P>
	The same matching logic as used in the <CODE>Values_Match</CODE>
	method is applied here, except that all criteria are tested. Thus
	the Selector criteria code that is returned indicates all matching
	criteria. If no criteria match the return value will be zero.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	A Selector criteria code.
	@see	#Values_Match(Value, Value)
*/
public int Values_Criteria_Match
	(
	Value		this_value,
	Value		that_value
	)
{
int
	selection = 0;

if (Any_Value ())
	return ANY_VALUE;

if (this_value == that_value)
	//	Same value.
	return ANY_VALUE | DATA | TYPE | BASE | UNITS;
if (this_value == null ||
	that_value == null)
	//	A Single null matches nothing.
	return 0;

if (Type () &&
	Types_Match
		(this_value,
		 that_value))
	selection |= TYPE;

if (Base () &&
	Bases_Match
		(this_value,
		 that_value))
	selection |= BASE;

if (Units () &&
	Units_Match
		(this_value,
		 that_value))
	selection |= UNITS;

if (Data () &&
	Data_Match
		(this_value,
		 that_value))
	selection |= DATA;

return selection;
}

/**	Tests if the types of two Values match.
<P>
	If <CODE>Specific</CODE> matching is enabled, then the types must
	be identical to match; otherwise gneral classifications (i.e.
	<CODE>NUMERIC</CODE>,<CODE>STRING</CODE>, or <CODE>ARRAY</CODE>)
	will also match.
<P>
	<B>N.B.</B>: If either Value is null the match is false.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	true if the Value types match; false otherwise.
*/
public boolean Types_Match
	(
	Value		this_value,
	Value		that_value
	)
{
if (this_value == null ||
	that_value == null)
	return false;
if (this_value.Type () ==
	that_value.Type ())
	return true;
if (! Specific () &&
	((this_value.Is_Numeric () &&
	  that_value.Is_Numeric ()) ||
	 (this_value.Is_String () &&
	  that_value.Is_String ()) ||
	 (this_value.Is_Array () &&
	  that_value.Is_Array ())))
	return true;
return false;
}

/**	Tests if the integer radix base values of two Values match.
<P>
	When the two Values do not have <CODE>INTEGER</CODE> data, a match
	always occurs. Otherwise the radix base values are compared
	numerically using the <CODE>Equal</CODE> and/or
	<CODE>Less_Than</CODE> or <CODE>Greater_Than</CODE> logic of this
	Selection.
<P>
	<B>N.B.</B>: If either Value is null the match is false.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	true if the Value radix base values match; false otherwise.
	@see	#Equal(boolean)
	@see	#Less_Than(boolean)
	@see	#Greater_Than(boolean)
*/
public boolean Bases_Match
	(
	Value		this_value,
	Value		that_value
	)
{
if (this_value == null ||
	that_value == null)
	return false;
if (this_value.Is_Integer () &&
	that_value.Is_Integer ())
	{
	int
		this_number = this_value.Base (),
		that_number = that_value.Base ();
	if (Equal ())
		{
		if (Less_Than ())
			return (this_number <= that_number);
		else if (Greater_Than ())
			return (this_number >= that_number);
		else
			return (this_number == that_number);
		}
	else if (Less_Than ())
		return (this_number < that_number);
	else if (Greater_Than ())
		return (this_number > that_number);
	else
		return (this_number != that_number);
	}
return true;
}

/**	Tests if the units descriptions of two Values match.
<P>
	If both Values do not have units descriptions, they match; but if
	one or the other does not have a units description, they do not
	match. If <CODE>Pattern_Match</CODE>ing is enabled, the the units
	String of this_value is matched against the regular expression
	pattern of that_value's units String. If the regular expression
	syntax is invalid a normal String comparision is done. If
	<CODE>Specific</CODE> matching is enabled, then the String
	<CODE>equals</CODE> method is used; otherwise the
	<CODE>equalsIgnoreCase</CODE> method is used.
<P>
	<B>N.B.</B>: If either Value is null the match is false.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	true if the Value units descriptions match; false otherwise.
	@see	String#matches(String)
*/
public boolean Units_Match
	(
	Value		this_value,
	Value		that_value
	)
{
if (this_value == null ||
	that_value == null)
	return false;
String
	these_units = this_value.Units (),
	those_units = that_value.Units ();
if (these_units != null)
	{
	if (those_units != null)
		{
		try
			{
			if (Pattern_Match ())
				return these_units.matches
					  (those_units);
			}
		catch (PatternSyntaxException exception)
			{/* Fall through to regular string matching */}
		if (Specific ())
			return these_units.equals
				  (those_units);
		return these_units.equalsIgnoreCase
			  (those_units);
		}
	}
else if (those_units == null)
	return true;
return false;
}

/**	Tests if the data of two Values match.
<P>
	If both Values have null data, they match; but if one or the other
	is null, they do not match.
<P>
	If the Values are <CODE>NUMERIC</CODE> type the data are compared
	numerically using the <CODE>Equal</CODE> and/or
	<CODE>Less_Than</CODE> or <CODE>Greater_Than</CODE> logic of this
	Selection.
<P>
	For <CODE>STRING</CODE> types, if <CODE>Pattern_Match</CODE>ing is
	enabled the String data of this_value is matched against the regular
	expression pattern of that_value's String data. If the regular
	expression syntax is invalid a normal String comparision is done. If
	<CODE>Specific</CODE> matching is enabled, then the String
	<CODE>equals</CODE> method is used; otherwise the
	<CODE>equalsIgnoreCase</CODE> method is used.
<P>
	For <CODE>ARRAY</CODE> types the data of all Values in the data
	Vector must match, using the current Selection criteria, for a match
	to occur. If the <CODE>Data</CODE> criteria applies then the match
	is recursive for all Array Values.
<P>
	When both Values are of <CODE>UNKNONW</CODE> type, they match.
<P>
	<B>N.B.</B>: If either Value is null the match is false.
<P>
	@param	this_value	The first of a pair of Values to be matched.
	@param	that_value	The second of a pair of Values to be matched.
	@return	true if the data of the Values match; false otherwise.
	@see	String#matches(String)
*/
public boolean Data_Match
	(
	Value		this_value,
	Value		that_value
	)
{
if (this_value == null ||
	that_value == null)
	return false;
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		(">>> Data_Match:\n"
		+ this_value + '\n'
		+ that_value);
if (this_value.Data () == null)
	{
	if (that_value.Data () == null)
		{
		if ((DEBUG & DEBUG_DATA) != 0)
			System.out.println
				("    Both null Data\n"
				+"<<< Data_Match: true");
		return true;
		}
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    One null Data\n"
			+"<<< Data_Match: false");
	return false;
	}
if (that_value.Data () == null)
	{
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    One null Data\n"
			+"<<< Data_Match: false");
	return false;
	}

boolean
	matches;
if (this_value.Is_Numeric () &&
	that_value.Is_Numeric ())
	{
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    NUMERIC");
	double
		this_double,
		that_double;
	try
		{
		this_double = this_value.double_Data ();
		that_double = that_value.double_Data ();
		}
	catch (PVL_Exception exception)
		{return false; /* Shouldn't happen */}
	if (Equal ())
		{
		if (Less_Than ())
			matches = (this_double <= that_double);
		else if (Greater_Than ())
			matches = (this_double >= that_double);
		else
			matches = (this_double == that_double);
		}
	else if (Less_Than ())
		matches = (this_double < that_double);
	else if (Greater_Than ())
		matches = (this_double > that_double);
	else
		matches = (this_double != that_double);
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("<<< Data_Match: " + matches);
	return matches;
	}

if (this_value.Is_String () &&
	that_value.Is_String ())
	{
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    STRING");
	try
		{
		try
			{
			if (Pattern_Match ())
				matches =
					this_value.String_Data ().matches
				   (that_value.String_Data ());
			}
		catch (PatternSyntaxException exception)
			{/* Fall through to regular string matching */}
		if (Specific ())
			matches = this_value.String_Data ().equals
				  (that_value.String_Data ());
		else
			matches = this_value.String_Data ().equalsIgnoreCase
				  (that_value.String_Data ());
		}
	catch (PVL_Exception exception)
		{return false; /* Shouldn't happen */}
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("<<< Data_Match: " + matches);
	return matches;
	}

if (this_value.Is_Array () &&
	that_value.Is_Array ())
	{
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("    ARRAY");
	try
		{
		if (this_value.Vector_Data ().size () !=
			that_value.Vector_Data ().size ())
			{
			if ((DEBUG & DEBUG_DATA) != 0)
				System.out.println
					("    Different vector sizes\n"
					+"<<< Data_Match: false");
			return false;
			}
		}
	catch (PVL_Exception exception)
		{
		//	Shouldn't happen since they're both valid arrays.
		return false;
		}
	Iterator
		these_values = this_value.iterator (),
		those_values = that_value.iterator ();
	while (these_values.hasNext () &&
		   those_values.hasNext ())
		{
		if (! Data_Match
				(
				(Value)these_values.next (),
				(Value)those_values.next ()
				))
			{
			if ((DEBUG & DEBUG_DATA) != 0)
				System.out.println
					("<<< Data_Match: false");
			return false;
			}
		}
	if ((DEBUG & DEBUG_DATA) != 0)
		System.out.println
			("<<< Data_Match: true");
	return true;
	}

if (this_value.Is_Unknown () &&
	that_value.Is_Unknown ())
	matches = true;
else
	matches = false;
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("    Both Unknown\n"
		+"<<< Data_Match: true");
return matches;
}



}	//	End of class
