/*
 * Copyright 2022-2023 Soren Stoutner <soren@stoutner.com>.
 *
 * This file is part of Privacy Browser PC <https://www.stoutner.com/privacy-browser-pc>.
 *
 * Privacy Browser PC is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Privacy Browser PC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Privacy Browser PC.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BROWSERWINDOW_H
#define BROWSERWINDOW_H

// Application headers.
#include "widgets/TabWidget.h"

// KDE Frameworks headers.
#include <KConfigDialog>
#include <KToggleFullScreenAction>
#include <KToolBar>
#include <KXmlGuiWindow>

// Qt toolkit headers.
#include <QComboBox>
#include <QLabel>
#include <QProgressBar>

class BrowserWindow : public KXmlGuiWindow
{
    // Include the Q_OBJECT macro.
    Q_OBJECT

public:
    // The default constructor.
    BrowserWindow(bool firstWindow=true, QString *initialUrlStringPointer = nullptr);

    // The public functions.
    QSize sizeHint() const override;

    // The public variables.
    TabWidget *tabWidgetPointer;

private Q_SLOTS:
    // The private slots.
    void addOrEditDomainSettings() const;
    void back() const;
    void clearUrlLineEditFocus() const;
    void decrementZoom();
    void editBookmarks() const;
    void escape() const;
    void findNext() const;
    void findPrevious() const;
    void forward() const;
    void fullScreenRequested(const bool toggleOn);
    void getZoomFactorFromUser();
    void hideFindTextActions() const;
    void home() const;
    void incrementZoom();
    void loadUrlFromLineEdit(const QString &url) const;
    void newWindow() const;
    void populateBookmarks();
    void refresh() const;
    void reloadAndBypassCache() const;
    void showBookmarkContextMenu(const QPoint &point);
    void showCookiesDialog();
    void showDownloadLocationBrowseDialog() const;
    void showDomainSettingsDialog() const;
    void showFindTextActions() const;
    void showProgressBar(const int &progress) const;
    void showSettingsDialog();
    void toggleBookmark();
    void toggleDomStorage() const;
    void toggleFindCaseSensitive() const;
    void toggleJavaScript() const;
    void toggleLocalStorage() const;
    void toggleFullScreen();
    void toggleViewBookmarksToolBar();
    void toggleViewSource() const;
    void toggleViewSourceInNewTab() const;
    void updateCookiesAction(const int numberOfCookies) const;
    void updateDefaultZoomFactor(const double newDefaultZoomFactor);
    void updateDomStorageAction(const bool &isEnabled) const;
    void updateDomainSettingsIndicator(const bool status);
    void updateFindText(const QString &text, const bool findCaseSensitive) const;
    void updateFindTextResults(const QWebEngineFindTextResult &findTextResult) const;
    void updateJavaScriptAction(const bool &isEnabled);
    void updateLocalStorageAction(const bool &isEnabled);
    void updateSearchEngineActions(const QString &searchEngine, const bool &updateCustomSearchEngineStatus);
    void updateUserAgentActions(const QString &userAgent, const bool &updateCustomUserAgentStatus);
    void updateZoomActions(const double &zoomFactor);
    void updateSearchEngineLabel(const QString &searchEngineString) const;
    void updateUrlLineEdit(const QUrl &newUrl);
    void updateUserAgentLabel(const QString &userAgentDatabaseName) const;
    void updateViewBookmarksToolBarCheckbox(const bool visible);
    void updateWindowTitle(const QString &title);
    void zoomDefault();

private:
    // The private variables.
    KActionCollection *actionCollectionPointer;
    QAction *bookmarkedActionPointer;
    QList<QPair<QMenu *, QAction *> *> bookmarksMenuActionList;
    QMenu *bookmarksMenuPointer;
    QList<QPair<QMenu *, QMenu *> *> bookmarksMenuSubmenuList;
    QList<QAction*> bookmarksToolBarActionList;
    QList<QPair<QMenu *, const double> *> bookmarksToolBarMenuList;
    KToolBar *bookmarksToolBarPointer;
    QList<QPair<QMenu *, QAction *> *> bookmarksToolBarSubfolderActionList;
    bool bookmarksToolBarIsVisible = false;
    bool bookmarksToolBarUninitialized = true;
    KConfigDialog *configDialogPointer;
    QAction *cookiesActionPointer;
    QUrl currentUrl;
    QPushButton *currentZoomButtonPointer;
    double currentZoomFactor;
    bool customSearchEngineEnabled;
    bool customUserAgentEnabled;
    double defaultZoomFactor;
    QAction *domStorageActionPointer;
    QComboBox *downloadLocationComboBoxPointer;
    QList<QPair<QMenu *, QAction *> *> finalBookmarkFolderMenuActionList;
    QAction *findCaseSensitiveActionPointer;
    QAction *findNextActionPointer;
    QAction *findPreviousActionPointer;
    QAction *findTextLabelActionPointer;
    QLabel *findTextLabelPointer;
    QAction *findTextLineEditActionPointer;
    KLineEdit *findTextLineEditPointer;
    KToggleFullScreenAction *fullScreenActionPointer;
    QAction *hideFindTextActionPointer;
    QAction *javaScriptActionPointer;
    bool javaScriptEnabled;
    QAction *localStorageActionPointer;
    bool localStorageEnabled;
    KToolBar *navigationToolBarPointer;
    QPalette negativeBackgroundPalette;
    QPalette normalBackgroundPalette;
    QPalette positiveBackgroundPalette;
    QProgressBar *progressBarPointer;
    QLabel *searchEngineLabelPointer;
    QAction *searchEngineMenuActionPointer;
    QAction *searchEngineMojeekActionPointer;
    QAction *searchEngineMonoclesActionPointer;
    QAction *searchEngineMetagerActionPointer;
    QAction *searchEngineGoogleActionPointer;
    QAction *searchEngineBingActionPointer;
    QAction *searchEngineYahooActionPointer;
    QAction *searchEngineCustomActionPointer;
    QLabel *userAgentLabelPointer;
    QAction *userAgentMenuActionPointer;
    QAction *userAgentPrivacyBrowserActionPointer;
    QAction *userAgentWebEngineDefaultActionPointer;
    QAction *userAgentFirefoxLinuxActionPointer;
    QAction *userAgentChromiumLinuxActionPointer;
    QAction *userAgentFirefoxWindowsActionPointer;
    QAction *userAgentChromeWindowsActionPointer;
    QAction *userAgentEdgeWindowsActionPointer;
    QAction *userAgentSafariMacosActionPointer;
    QAction *userAgentCustomActionPointer;
    KLineEdit *urlLineEditPointer;
    KToolBar *urlToolBarPointer;
    QAction *viewBookmarksToolBarActionPointer;
    QAction *viewSourceActionPointer;
    QAction *viewSourceInNewTabActionPointer;
    QAction *zoomDefaultActionPointer;
    QAction *zoomFactorActionPointer;
    QAction *zoomInActionPointer;
    QPushButton *zoomMinusButtonPointer;
    QAction *zoomOutActionPointer;
    QPushButton *zoomPlusButtonPointer;

    // The private functions.
    void addFinalBookmarkFolderMenuActions(QMenu *menuPointer, double folderId);
    void populateBookmarksMenuSubfolders(const double folderId, QMenu *menuPointer);
    void populateBookmarksToolBar();
    void populateBookmarksToolBarSubfolders(const double folderId, QMenu *menuPointer);
    void updateBookmarkedAction() const;
};
#endif
