/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef OWQUTECOMACCOUNTPARSER_H
#define OWQUTECOMACCOUNTPARSER_H

#include "QuteComAccount.h"

#include <string>

class StringList;

/**
 * Parses the XML datas from the Single Sign On authentication on the QuteCom platform (SSO).
 *
 * This a separated class rather than to be inside the class QuteComAccount,
 * this allows more modularity and encapsulates the use of TinyXML.
 *
 * XML scheme
 *
 * http://ws.qutecom.fr/softphone-sso/sso.php?login=LOGIN&password=PASSWORD
 *
 * Replace LOGIN and PASSWORD by the login/password provided by QuteCom.
 *
 * If the login/password are wrong it will return:
 * <pre>
 * <?xml version="1.0" ?>
 * <sso vapi="1">
 *     <status code="401">Unauthorized</status>
 * </sso>
 * </pre>
 *
 * If the login/password are correct it will return:
 * <pre>
 * <?xml version="1.0" encoding="UTF-8" ?>
 * <sso mib="1">
 * 	<notice><![CDATA[DEPRECATED on 12/01/2006]]></notice>
 * 	<status code="200"><![CDATA[OK]]></status>
 * 	<d k="sip.auth.userid" v="toto"/>
 * 	<d k="sip.auth.password" v="secret"/>
 * 	<d k="sip.auth.realm" v="192.168.1.1"/>
 * 	<d k="sip.address.name" v="toto"/>
 * 	<d k="sip.address.displayname" v="pseudo"/>
 * 	<d k="sip.address.server.host" v="192.168.1.1"/>
 * 	<d k="sip.address.server.port" v="5060"/>
 * 	<d k="sip.outbound" v="1"/>
 * 	<d k="sip.outbound.proxy.host" v="192.168.1.1"/>
 * 	<d k="sip.outbound.proxy.port" v="5060"/>
 * 	<d k="netlib.stun.host" v="192.168.1.1"/>
 * 	<d k="netlib.stun.port" v="5060"/>
 *	<d k="netlib.tunnel.http">
 *		<l v="80.118.99.31"/>
 *		<l v="80.118.99.32"/>
 *	</d>
 *	<d k="netlib.tunnel.https">
 *		<l v="80.118.99.51"/>
 *		<l v="80.118.99.52"/>
 *		<l v="80.118.99.53"/>
 *		<l v="80.118.99.54"/>
 *	</d>
 * </sso>
 * </pre>
 *
 * @ingroup model
 * @author Mathieu Stute
 * @author Tanguy Krotoff
 */
class QuteComAccountParser {
public:

	QuteComAccountParser(QuteComAccount & account, const std::string & data);

	/**
	 * Check if password is correct
	 *
	 * @return true if login ok
	 */
	bool isLoginPasswordOk() const {
		return _loginPasswordOk;
	}

private:

	/**
	 * Choose a random http tunnel
	 *
	 * @param httpTunnels a list of http tunnel ip
	 * @return a http tunnel ip
	 */
	std::string chooseHttpTunnel(const StringList & httpTunnels);

	bool _loginPasswordOk;
};

#endif	//OWQUTECOMACCOUNTPARSER_H
