% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h5write.R
\name{h5_write}
\alias{h5_write}
\alias{h5write}
\alias{h5write.default}
\alias{h5writeDataset}
\alias{h5writeDataset.data.frame}
\alias{h5writeDataset.array}
\title{Write object to an HDF5 file.}
\usage{
h5write(obj, file, name, ...)

\method{h5write}{default}(
  obj,
  file,
  name,
  createnewfile = TRUE,
  write.attributes = FALSE,
  ...,
  native = FALSE
)

h5writeDataset(obj, h5loc, name, ...)

\method{h5writeDataset}{data.frame}(obj, h5loc, name, level = 6, chunk, DataFrameAsCompound = TRUE)

\method{h5writeDataset}{array}(
  obj,
  h5loc,
  name,
  index = NULL,
  start = NULL,
  stride = NULL,
  block = NULL,
  count = NULL,
  size = NULL,
  variableLengthString = FALSE,
  encoding = c("ASCII", "UTF-8"),
  level = 6
)
}
\arguments{
\item{obj}{The R object to be written.}

\item{file}{The filename (character) of the file in which the dataset will be
located. For advanced programmers it is possible to provide an object of
class \linkS4class{H5IdComponent} representing a H5 location identifier (file or
group). See \code{\link{H5Fcreate}}, \code{\link{H5Fopen}},
\code{\link{H5Gcreate}}, \code{\link{H5Gopen}} to create an object of this
kind.}

\item{name}{The name of the dataset in the HDF5 file.}

\item{\dots}{Further arguments passed to \code{\link{H5Dwrite}}.}

\item{createnewfile}{If \code{TRUE}, a new file will be created if necessary.}

\item{write.attributes}{(logical) If TRUE, all R-attributes attached to the
object \code{obj} are written to the HDF5 file.}

\item{native}{An object of class \code{logical}. If TRUE, array-like objects
are treated as stored in HDF5 row-major rather than R column-major
orientation. Using \code{native = TRUE} increases HDF5 file portability
between programming languages. A file written with \code{native = TRUE}
should also be read with \code{native = TRUE}}

\item{h5loc}{An object of class \linkS4class{H5IdComponent} representing a H5
location identifier (file or group). See \code{\link{H5Fcreate}},
\code{\link{H5Fopen}}, \code{\link{H5Gcreate}}, \code{\link{H5Gopen}} to
create an object of this kind.}

\item{level}{The compression level. An integer value between 0 (no
compression) and 9 (highest and slowest compression). Only used, if the
dataset does not yet exist. See \code{\link[=h5createDataset]{h5createDataset()}} to create an dataset.}

\item{chunk}{Specifies the number of items to be include in an HDF5 chunk. If
left unspecified the defaults is the smaller of: the total number of
elements or the number of elements that fit within 4GB of memory. If
\code{DataFrameAsCompound=FALSE} each row of the \code{data.frame} can be consider an
"element".}

\item{DataFrameAsCompound}{If true, a \code{data.frame} will be saved as a
compound data type. Otherwise it is saved like a list. The advantage of
saving a data.frame as a compound data type is that it can be read as a
table from python or with a struct-type from C. The disadvantage is that
the data has to be rearranged on disk and thus can slow down I/O. If fast
reading is required, \code{DataFrameAsCompound=FALSE} is recommended.}

\item{index}{List of indices for subsetting. The length of the list has to
agree with the dimensional extension of the HDF5 array. Each list element
is an integer vector of indices. A list element equal to \code{NULL} chooses all
indices in this dimension. Counting is R-style 1-based.}

\item{start}{The start coordinate of a hyperslab (similar to subsetting in
R). Counting is R-style 1-based. This argument is ignored, if index is not
NULL.}

\item{stride}{The stride of the hypercube. Read the introduction
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html} before using this
argument. R behaves like Fortran in this example. This argument is ignored,
if index is not NULL.}

\item{block}{The block size of the hyperslab. Read the introduction
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html} before using this
argument. R behaves like Fortran in this example. This argument is ignored,
if index is not NULL.}

\item{count}{The number of blocks to be written. This argument is ignored, if
index is not NULL.}

\item{size}{The length of the fixed-width string data type, when \code{obj} is a
character vector. If \code{NULL}, this is set to the length of the largest
string.}

\item{variableLengthString}{Whether character vectors should be written as
variable-length strings into the attributes. If \code{TRUE}, \code{size} is ignored.}

\item{encoding}{The encoding of the string data type.  Valid options are
"ASCII" or "UTF-8".}
}
\value{
\code{h5write} returns 0 if successful.
}
\description{
Writes an R object to an HDF5 file. This function can be used to write either
full arrays/vectors or subarrays (hyperslabs) within an existing dataset.
}
\details{
Writes an R object to an HDF5 file. If none of the arguments \code{start,
stride, block, count} is specified, the dataset has the same dimension in the
HDF5 file and in memory. If the dataset already exists in the HDF5 file, one
can write subarrays, (so called hyperslabs) to the HDF5 file. The arguments
\code{start, stride, block, count} define the subset of the dataset in the
HDF5 file that is to be written to. See these introductions to hyperslabs:
\url{https://support.hdfgroup.org/HDF5/Tutor/selectsimple.html},
\url{https://support.hdfgroup.org/HDF5/Tutor/select.html} and
\url{http://ftp.hdfgroup.org/HDF5/Tutor/phypecont.html}. Please note that in
R the first dimension is the fastest changing dimension.

When viewing the HDF5 datasets with any C-program (e.g. HDFView), the order
of dimensions is inverted. In the R interface counting starts with 1, whereas
in the C-programs (e.g. HDFView) counting starts with 0.
}
\examples{

h5File <- tempfile(fileext = ".h5")
h5createFile( h5File )

# write a matrix
B = array(seq(0.1,2.0,by=0.1),dim=c(5,2,2))
attr(B, "scale") <- "liter"
h5write(B, h5File,"B")

# write a submatrix
h5createDataset(h5File, "S", c(5,8), storage.mode = "integer", chunk=c(5,1), level=7)
h5write(matrix(1:5,nr=5,nc=1), file=h5File, name="S", index=list(NULL,1))

}
\references{
\url{https://portal.hdfgroup.org/display/HDF5}
}
\seealso{
\code{\link{h5ls}}, \code{\link{h5createFile}},
\code{\link{h5createDataset}}, \link{rhdf5}
}
\author{
Bernd Fischer, Mike Smith
}
\keyword{IO}
\keyword{file}
\keyword{interface}
\keyword{programming}
