-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

with E_Strings;
with SPARK_IO;

--# inherit ExaminerConstants,
--#         E_Strings,
--#         SPARK_IO;

package FileSystem is

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to reading
   -- command-line arguments
   -----------------------------------------------------------------

   function Argument_Separator return Character;

   function Is_An_Argument_Terminator (Ch : Character) return Boolean;

   function Use_Windows_Command_Line return Boolean;

   function Use_Unix_Command_Line return Boolean;

   procedure Read_Cmd_Line (Cmd_Line_Found : out Boolean;
                            Cmd_Line       : out E_Strings.T);
   --# global in out SPARK_IO.File_Sys;
   --# derives Cmd_Line,
   --#         Cmd_Line_Found,
   --#         SPARK_IO.File_Sys from SPARK_IO.File_Sys;

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to the casing
   -- of filenames
   -----------------------------------------------------------------

   -- Select default filename casing when Opening files
   function Case_Of_Files_For_Open (E_Str : E_Strings.T) return E_Strings.T;

   -- Select default filename casing when Creating files
   function Case_Of_Files_For_Create (E_Str : E_Strings.T) return E_Strings.T;

   -----------------------------------------------------------------
   -- Constants, types, and functions relating to Source files
   -----------------------------------------------------------------

   -- Opens a source file for reading, but using the Form required
   -- by the host compiler to enable sharing of the file.
   procedure Open_Source_File (File   : in out SPARK_IO.File_Type;
                               Name   : in     E_Strings.T;
                               Status :    out SPARK_IO.File_Status);
   --# global in out SPARK_IO.File_Sys;
   --# derives File,
   --#         SPARK_IO.File_Sys,
   --#         Status            from File,
   --#                                Name,
   --#                                SPARK_IO.File_Sys;

   -----------------------------------------------------------------
   -- Constants, types, and subprograms relating to filenames and
   -- extensions.
   -----------------------------------------------------------------

   type Typ_File_Spec_Status is (File_Found, File_Missing, File_Invalid);

   -- Returns File_Name fully normalized relative to the second argument.
   -- If the second argument is an empty string, then the filename
   -- is interpretated relative to current working directory
   function Interpret_Relative (File_Name, Relative_To_Directory : E_Strings.T) return E_Strings.T;

   -- Find the fully normalized pathname for File_Spec relative
   -- to current working directory and returns it in Full_File_Name.
   -- File_Status is set to File_Found if this pathname denotes
   -- an existing regular file, or File_Missing otherwise.
   procedure Find_Full_File_Name
     (File_Spec      : in     E_Strings.T;
      File_Status    :    out Typ_File_Spec_Status;
      Full_File_Name :    out E_Strings.T);
   --# global in SPARK_IO.File_Sys;
   --# derives File_Status,
   --#         Full_File_Name from File_Spec,
   --#                             SPARK_IO.File_Sys;

   -- Returns the basename of Fn, with or without the extension. For example:
   -- Just_File ("a/b/c/f.dat", True) = "f.dat"
   -- Just_File ("a/b/c/g.exe", False) = "g"
   function Just_File (Fn  : E_Strings.T;
                       Ext : Boolean) return E_Strings.T;

   --  Any directory prefix is removed. If Suffix is non-empty and is a
   --  suffix of Fn (1 .. Stop), it is removed.  This implements the
   --  same semantics as GNAT.Directory_Operations.Base_Name on systems
   --  where this library is available, but is legal SPARK.
   function Base_Name (Path   : E_Strings.T;
                       Suffix : String) return E_Strings.T;

   procedure Check_Extension (Fn  : in out E_Strings.T;
                              Ext : in     E_Strings.T);
   --# derives Fn from *,
   --#                 Ext;

   --780--special version of Check_Extension just for listing files
   procedure Check_Listing_Extension (Source_Name : in     E_Strings.T;
                                      Fn          : in out E_Strings.T;
                                      Ext         : in     E_Strings.T);
   --# derives Fn from *,
   --#                 Ext,
   --#                 Source_Name;

   function Is_Readable_File (F : E_Strings.T) return Boolean;
   --# global in SPARK_IO.File_Sys;

   function Same_File (F1, F2 : E_Strings.T) return Boolean;

   function Is_Directory (F : E_Strings.T) return Boolean;
   --# global in SPARK_IO.File_Sys;

   function Start_Of_Directory return  E_Strings.T;
   function Directory_Separator return  E_Strings.T;

   -- if directory name D ends with an End_Of_Path string, then
   -- no change, otherwise append End_Of_Path
   procedure Append_End_Of_Path_If_Needed (D : in out E_Strings.T);
   --# derives D from *;

   -- if directory name D ends with an End_Of_Path string, then
   -- remove it, otherwise no change
   procedure Remove_End_Of_Path_If_Present (D : in out E_Strings.T);
   --# derives D from *;

   procedure Idempotent_Create_Subdirectory (Path : in     E_Strings.T;
                                             Ok   :    out Boolean);
   --# global in out SPARK_IO.File_Sys;
   --# derives Ok,
   --#         SPARK_IO.File_Sys from Path,
   --#                                SPARK_IO.File_Sys;

   -- For Unix/NT does not return a trailing / or \
   function Working_Directory return  E_Strings.T;
   --# global in SPARK_IO.File_Sys;

   -- For Unix/NT does not return a trailing / or \
   function Examiner_Lib_Directory return  E_Strings.T;
   --# global in SPARK_IO.File_Sys;

   -- For Unix/NT does not return a trailing / or \
   function Examiner_SPARK_Lib_Directory return  E_Strings.T;
   --# global in SPARK_IO.File_Sys;

   function Get_Environment_Variable (Env_Var_Name : String) return E_Strings.T;
   --# global in SPARK_IO.File_Sys;

   function Get_Relative_Path (Full_Path : E_Strings.T;
                               Prefix    : E_Strings.T) return E_Strings.T;

   function End_Of_Line return  E_Strings.T;

end FileSystem;
