// Copyright (C) 2006-2009 Kent-Andre Mardal and Simula Research Laboratory
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.

#ifndef DOF_IS_INCLUDED
#define DOF_IS_INCLUDED

#include "FE.h"

#include <map>
#include <vector>
#include <iterator>
#include <iostream>
#include <stdexcept>

namespace SyFi
{

	typedef std::pair<unsigned int, unsigned int> pair_ii;
	typedef std::vector< std::pair<unsigned int, unsigned int> > vector_ii;

	class Dof
	{
		protected:
			// running counter for inserted global indices
			unsigned int counter;

			// highest element number inserted
			unsigned int emax;

			// highest local index inserted
			unsigned int imax;

			// the structures loc2dof, dof2index, and doc2loc  are completely dynamic
			// they are all initialized and updated by insert_dof(int e, int i, ex Li)

			// (int e, int i) -> int j
			std::map<std::pair<unsigned int,unsigned int>, unsigned int>          loc2glob;

			// (ex j)                -> vector< pair<e1, i1>, ..  pair<en, in> >
			bool create_glob2loc;
			std::map< unsigned int, vector_ii >  glob2loc_map;

			// (ex Lj)               ->  int j
			std::map<GiNaC::ex , unsigned int, GiNaC::ex_is_less >       dof2glob;

			// (int j)               -> ex Lj
			bool create_glob2dof;
			std::map< unsigned int, GiNaC::ex >          glob2dof;

		public:
			Dof(bool create_glob2dof = false, bool create_glob2loc = false):
			counter(0),
				emax(0),
				imax(0),
				create_glob2loc(create_glob2loc),
				create_glob2dof(create_glob2dof)
			{
			}

			~Dof() {}

			void clear();

			// Update the internal structures with a new dof.
			unsigned int insert_dof(unsigned int e, unsigned int i, GiNaC::ex Li);

			// --- Helper functions to be used after all the dofs have been inserted.
			// These do not modify the internal structure. ---

			// Return the number of global dofs inserted.
			unsigned int size() const
				{ return counter; }

			// Return the number of elements inserted.
			unsigned int num_elements() const
				{ return emax+1; }

			// Return the number of dofs per elements.
			unsigned int max_dofs_per_element() const
				{ return imax+1; }

			// Return the global index for local dof i in element e.
			unsigned int glob_dof(unsigned int e, unsigned int i);

			// Return the global index for dof Lj represented with GiNaC::ex.
			unsigned int glob_dof(GiNaC::ex Lj);

			// Return the dof GiNaC::ex for global index j.
			GiNaC::ex glob_dof(unsigned int j);

			// Return a vector of all the (element,index) pairs this global index is equivalent with.
			vector_ii glob2loc(unsigned int j);

	};

}
#endif
