# -*- coding: utf-8 -*-
"""
web2ldapcnf/hosts.py - Host-related options
(c) by Michael Stroeder <michael@stroeder.com>
"""

# Leave this alone
import os,web2ldapcnf

class Web2LDAPConfig:
  """
  Base class for a web2ldap host-/backend configuration section.

  Leave untouched!!!
  """
  def __init__(self,**kwargs):
    self.__dict__.update(kwargs)
  def get(self,name,default=None):
    self.__dict__.get(name,default)


########################################################################
# List of all LDAP hosts to use. These hosts will appear in the
# default select list of the login form.
# A list containing only one host results in a normal input field
# with the host set as default.
########################################################################

ldap_uri_list = [
  'ldap://localhost',
  (
    'ldap://localhost:1390',
    u'Local OpenLDAP 2.x server'
  ),
  (
    'ldapi://%2Ftmp%2Fopenldap-socket',
    u'Local OpenLDAP 2.x server (Unix domain socket)'
  ),
  ('ldap://ldap.uninett.no/dc=uninett,dc=no','UNINETT'),
  ('ldap://magnesio.rediris.es:1389/dc%3Drediris%2Cdc%3Des??one?%28objectClass%3D%2A%29','Red.es/RedIRIS'),
  ('ldap://ldap.openosi.org/??one?%28objectClass%3D%2A%29','OpenOSI'),
  ('ldap://db.debian.org/dc=debian,dc=org??one',u'debian.org Developers LDAP Search'),
  ('ldap://x500.bund.de/o=Bund,c=DE??one',u'X.500 Directory Informationsverbund Berlin-Bonn'),
  ('ldap://www.trustcenter.de/dc=trustcenter,dc=de',u'TC Trustcenter'),
  ('ldap://pki-ext-vzd1.bayern.de/DC%3Dpki%2CDC%3Dbayern%2CDC%3Dde??one',u'Bayerische Verwaltungs-PKI'),
  ('ldap://keyserver.pgp.com',u"PGP Global Directory"),
  ('ldap://ds1.us.freeldap.org/??one?',u"Free LDAP (needs registration)"),
  ('ldap://mailcert.t-online.de/ou%3D4901%2Cou%3Dmailcert%2Cou%3Dservices%2Cdc%3DT-Online%2Cdc%3Dcom',u'T-Online Mail Certs'),
  ('ldap://pks-ldap.telesec.de/c=DE',u'T-TeleSec PKS'),
  ('ldap://ldap.nrca-ds.de/dc=ldap,dc=nrca-ds,dc=de??one',u"Qualified certificates of german BNetzA"),
  ('ldap://ldap.globaltrustpoint.com/dc%3Dzertificon%2Cdc%3Dcom',u"Z1 Global TrustPoint (www.globaltrustpoint.com)"),
  'ldap://ldap.itd.umich.edu',
  ('ldap://ldap.a-trust.at/c=AT??one?',u'A-Trust PKI Verzeichnisdienst'),
  ('ldap://directory.s-trust.de',u'S-Trust'),
  ('ldap://ldap1.pca.dfn.de/ou%3DDFN-PKI%2Co%3DDFN-Verein%2Cc%3Dde??one',u'DFN-PKI'),
  ('ldap://ldap.t-mailpass.de/c=de??one?',u'T-Systems Mailpass'),
  ('ldap://ldap.sbca.telesec.de/c=de??one',u'T-Systems Shared Business'),
  ('ldap://ldap.crl.esecure.datev.de/??one',u'Datev'),
  ('ldap://directory.swisssign.net/o=SwissSign,c=CH??one',u'SwissSign AG'),
]

# Set to True (or 1) if LDAP access should be restricted to the LDAP servers
# defined in ldap_uri_list (default if absent is 0 - restriction disabled)
restricted_ldap_uri_list = 1

########################################################################
# LDAP host(s) with their defaults can be pre-defined as dictionary
# ldap_def = {'host:port':{paramdictionary}}
########################################################################

ldap_def = {

  # Overall default values.
  # There's most times no need to adjust the values in this section
  # since they can be overridden by host specific sections (see below).
  '_': Web2LDAPConfig(

    # Search filter template for smart login
    # Use indexed attributes here!
    binddnsearch=ur'(|(cn=%s)(uid=%s)(sAMAccountName=%s)(userPrincipalName=%s))',

    # HTML template strings used to bind name in the status section
    # on top of page depending on the object class of an entry.
    boundas_template= {
      'inetOrgPerson':r'<strong>%(cn)s</strong> &lt;%(mail)s&gt;(%(uid)s/%(employeeNumber)s)',
      'account':r'<strong>%(uid)s</strong>',
    },

    # Timeout for LDAP operations (seconds)
    timeout=300,

    # Use StartTLS on connect. It's recommended saying 0 as default
    # since not many servers can handle an unknown extended operation properly
    starttls=0,

    # Directory containing all the trusted root CA certs (symbolic hash links required!)
#    tls_cacertdir=os.path.sep.join([web2ldapcnf.etc_dir,'web2ldap', 'ssl', 'crt']),
    # File containing all the trusted root CA certs
    tls_cacertfile=os.path.sep.join([web2ldapcnf.etc_dir,'web2ldap', 'ssl', 'crt','trusted-certs.crt']),

    # regular expression of subject DNs in client certs
    ssl_valid_dn=r'/.*',
    # regular expression of subject DNs in issuer certs
    ssl_valid_idn=r'/.*',
    sec_reqlevel=0,

    # Send session track control
    session_track_control=0,

    # Attributes explicitly requested while doing read
    # and modify operations
    requested_attrs=[
      'structuralObjectClass','governingStructureRule','subschemaSubentry',
      # password policy attributes
      'passwordExpirationTime','passwordExpWarned',
      'passwordRetryCount','retryCountResetTime','accountUnlockTime',
      'passwordHistory','passwordAllowChangeTime',
      # Account specifics
      'nsAccountLock','nsLookthroughLimit',
      # ACL attributes
      'aci','aclentry',
      # counters
      'hasSubordinates','numSubordinates','subordinateCount',
      # Dynamic Entries (see RFC 2589)
      'entryTTL',
      # Proxy Authz attributes
      'authzTo','authzFrom',
      # password policy attributes
      'pwdPolicySubentry',
      # Sun Directory Server
      'isMemberOf',
      # MS Active Directory
      'nTSecurityDescriptor',
      'tokenGroups','tokenGroupsGlobalAndUniversal','tokenGroupsNoGCAcceptable',
      'mS-DS-CreatorSID','primaryGroupToken','canonicalName','fromEntry',
      'sDRightsEffective','msDS-Approx-Immed-Subordinates','msDS-KeyVersionNumber',
#      'msDS-ReplAttributeMetaData',
#      'lastLogonTimestamp','lockoutTime',
      'allowedAttributes','allowedAttributesEffective','allowedChildClasses','allowedChildClassesEffective',
      # X.500 DSAs
      'administrativeRole',
    ],

    # List of attribute type names which are supposed to be constant during
    # editing an entry.
    modify_constant_attrs=[
      # Mostly OpenLDAP
      'entryCSN','entryDN','entryUUID',
      # see RFC
      'createTimestamp','modifyTimestamp','creatorsName','modifiersName',
      # MS AD
      'uSNChanged','uSNCreated','whenChanged','whenCreated',
      # Netscape/Sun DS
      'nsUniqueId',
      # 389 Directory Server also known as Fedora DS
      'entryUSN',
      # eDirectory
      'localEntryID','GUID',
    ],

    # vCard template files
    vcard_template={
      # 'object class':'pathname of vCard template file'
      'person':os.path.join(web2ldapcnf.templates_dir,'vcard_person.txt'),
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'vcard_person.txt'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'vcard_organization.txt'),
    },

    # HTML template files for printing table entries
    print_template={
      # 'object class':'pathname of printable HTML template file'
      'person':os.path.join(web2ldapcnf.templates_dir,'print_person.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'print_organization.html'),
      'organizationalUnit':os.path.join(web2ldapcnf.templates_dir,'print_organizationalUnit.html'),
    },

    # HTML template file for search options fieldset in search forms
    searchoptions_template=os.path.join(web2ldapcnf.templates_dir,'searchform_search_options.html'),

    # HTML template file for Basic Search form
    searchform_template= {
      u'_':os.path.join(web2ldapcnf.templates_dir,'searchform_Base.html'),
      u'Users':os.path.join(web2ldapcnf.templates_dir,'searchform_users.html'),
      u'Orga':os.path.join(web2ldapcnf.templates_dir,'searchform_orga.html'),
      u'Persons':os.path.join(web2ldapcnf.templates_dir,'searchform_persons.html'),
    },

    # HTML template file for rename form
    rename_template= os.path.join(web2ldapcnf.templates_dir,'rename.html'),

    # HTML template file for rename form
    status_template= os.path.join(web2ldapcnf.templates_dir,'status.html'),

    # HTML template file for password change  form
    passwd_template= os.path.join(web2ldapcnf.templates_dir,'passwd.html'),

    # HTML template file for Login form
    login_template= os.path.join(web2ldapcnf.templates_dir,'login.html'),
    # Default mechanism for BindRequest (empty string for simple bind)
    login_default_mech='',

    # Attributes which should be present in attribute select list of advanced search form
    search_attrs=[],

    # There are some situations where this client just wants to get the
    # attributes of an entry and not the data itself for saving bandwidth.
    # However some LDAP hosts (e.g. Notes Domino 4.61) have problems with
    # such an attribute-only request, they won't return any matches for a search.
    # If you experience these problems (no matching entry) set this to 0.
    search_attrsonly=1,

    search_tablistattrs=['cn','mail','homephone','telephonenumber','mobile'],

    # HTML template strings used to display entries in the table
    # of search results
    search_tdtemplate= {
      'inetOrgPerson':r'%(cn)s &lt;%(mail)s&gt;<br>Office: %(telephoneNumber)s, Home: %(homePhone)s, Mobile: %(mobile)s',
#      'organization':r'&quot;%(o)s&quot Tel.: %(telephoneNumber)s',
#      'organizationalUnit':r'Org. Unit &quot;%(ou)s&quot;',
      'rfc822MailGroup':r'Mailing list %(cn)s &lt;%(mail)s&gt;, see %(labeledurl)s',
      'account':r'Account <strong>%(uid)s</strong>',
      'groupOfNames':r'Group <strong>%(cn)s</strong>',
      'groupOfUniqueNames':r'Group <strong>%(cn)s</strong>',
      'posixGroup':r'Posix group <strong>%(cn)s</strong> (%(gidNumber)s)',
      'sambaDomain':r'Samba domain <strong>%(sambaDomainName)s</strong> (%(sambaSID)s)',
    },

    # Default for number of results shown per page
    search_resultsperpage=10,

    # HTML template file used for displaying entries of specific object class
    read_template={
      # 'object class':'pathname of HTML template file'
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'account':os.path.join(web2ldapcnf.templates_dir,'read_account.html'),
      'organizationalPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'msPerson':os.path.join(web2ldapcnf.templates_dir,'read_msPerson.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'read_organization.html'),
      'msOrganization':os.path.join(web2ldapcnf.templates_dir,'read_msOrganization.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'read_posixAccount.html'),
      'eduPerson':os.path.join(web2ldapcnf.templates_dir,'read_eduPerson.html'),
      'pwdPolicy':os.path.join(web2ldapcnf.templates_dir,'read_pwdPolicy.html'),
      'sambaDomain':os.path.join(web2ldapcnf.templates_dir,'read_sambaDomain.html'),
      'sambaSamAccount':os.path.join(web2ldapcnf.templates_dir,'read_sambaSamAccount.html'),
      'sambaGroupMapping':os.path.join(web2ldapcnf.templates_dir,'read_sambaGroupMapping.html'),
    },

    # Maximum count of attribute values displayed when displaying a single entry
    # without attribute values being expanded
    read_tablemaxcount={
      'member':20,
      'uniqueMember':20,
      'memberUID':20,
      'roleOccupant':20,
      'memberOf':2,
      'allowedAttributes':2,
      'allowedAttributesEffective':2,
      'allowedChildClasses':2,
      'allowedChildClassesEffective':2,
      'allowedChildClassesEffective':2,
      'tokenGroups':4,
      'tokenGroupsGlobalAndUniversal':4,
      'tokenGroupsNoGCAcceptable':4,
    },

    # HTML template file used for displaying input forms for entries
    # of specific object class
    input_template={
      # 'object class':'pathname of HTML template file'
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_inetOrgPerson.html'),
      'account':os.path.join(web2ldapcnf.templates_dir,'inputform_account.html'),
      'msPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_msPerson.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'inputform_posixAccount.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'inputform_organization.html'),
      'msOrganization':os.path.join(web2ldapcnf.templates_dir,'inputform_msOrganization.html'),
      'eduPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_eduPerson.html'),
      'pwdPolicy':os.path.join(web2ldapcnf.templates_dir,'inputform_pwdPolicy.html'),
      'sambaDomain':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaDomain.html'),
      'sambaSamAccount':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaSamAccount.html'),
      'sambaGroupMapping':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaGroupMapping.html'),
    },

    # HTML template file used for displaying operational attributes
    # in a footer when displaying a single entry
    read_operationalattrstemplate=os.path.join(web2ldapcnf.templates_dir,'read_operationalattrs.html'),

    # Number of columns for print output
    print_cols=4,

    # Parameters for password generation
    # Unicode string containing all valid characters used when generating
    # password values
#    passwd_genchars=u'abcd',
    # Length of generated password
    passwd_genlength=12,

    # Dictionary { description : LDIF file name } for displaying
    # link for quickly choosing LDIF templates for new entries
    addform_entry_templates={
      u'Person':os.path.join(web2ldapcnf.templates_dir,'add_person.ldif'),
      u'Organization':os.path.join(web2ldapcnf.templates_dir,'add_organization.ldif'),
      u'User':os.path.join(web2ldapcnf.templates_dir,'add_user.ldif'),
      u'Org. Unit':os.path.join(web2ldapcnf.templates_dir,'add_orgunit.ldif'),
      u'Group':os.path.join(web2ldapcnf.templates_dir,'add_group.ldif'),
    },

    # HTML template strings used to display the superior entry
    # in the input form when adding/modifying entries
    inputform_supentrytemplate={
      'organization':r'Organization <strong>%(o)s</strong>',
      'organizationalUnit':r'Organizational Unit <strong>%(ou)s</strong>',
    },

    # Named set of LDAP URLs for searching new superior DN when renaming an entry
    rename_supsearchurl={
      u'Search current naming context for organizationalUnit':'ldap:///_??sub?(objectClass=organizationalUnit)',
      u'Search current naming context for organization':'ldap:///_??sub?(objectClass=organization)',
      u'Search current naming context for locality':'ldap:///_??sub?(objectClass=locality)',
      u'Search current naming context for domain':'ldap:///_??sub?(objectClass=domain)',
      u'Search below superior entry for organizationalUnit':'ldap:///..??sub?(objectClass=organizationalUnit)',
      u'Search below superior entry for organization':'ldap:///..??sub?(objectClass=organization)',
      u'Search below superior entry for locality':'ldap:///..??sub?(objectClass=locality)',
      u'Search below superior entry for domain':'ldap:///..??sub?(objectClass=domain)',
    },

    # LDIF file used as locally stored pseudo LDAPv3 schema
    schema_uri='file:'+os.path.join(web2ldapcnf.etc_dir,os.path.join('web2ldap','localschema.ldif')),

    # The definitions for group entry administration
    groupadm_filterstr_template=r'(|%s)',
    groupadm_defs={
      'groupOfNames':       ('member',None),
      'groupOfUniqueNames': ('uniqueMember',None),
      'organizationalRole': ('roleOccupant',None),
      'rfc822MailGroup':    ('mail','mail'),
      'nisMailAlias':       ('rfc822MailMember','mail'),
      'mailGroup':          ('mgrprfc822mailmember','mail'),
      # Found on IBM SecureWay Directory
      'accessGroup':        ('member',None),
      # RFC2370
      'posixGroup':         ('memberUid','uid'),
      'nisNetgroup':        ('memberNisNetgroup','uid'),
      # Samba 3.0
      'sambaGroupMapping':  ('sambaSIDList','sambaSID'),
      # Active Directory
      'group':              ('member',None),
      # draft-findlay-ldap-groupofentries
      'groupOfEntries':     ('member',None),
      # Apple MAC OS X
      'apple-group':        ('apple-group-memberguid','apple-generateduid'),
    },

    # <link> sections to be displayed in <head> section of HTML output
    # Mainly needed for specifying the CSS files to be used.
    link_css="""
    <link rel="stylesheet" title="default" type="text/css" media="screen" href="/css/web2ldap/default.css">
    <!--link rel="stylesheet" title="Red and Blue" type="text/css" media="screen" href="/css/web2ldap/redandblue.css">
    <link rel="stylesheet" title="Retro terminal" type="text/css" media="screen" href="/css/web2ldap/terminal.css">
    <link rel="stylesheet" title="Hippie" type="text/css" media="screen" href="/css/web2ldap/hippie.css"-->
    <link rel="stylesheet" type="text/css" media="print, embossed" href="/css/web2ldap/print.css">
    """,
  ),

  'ldap://localhost': Web2LDAPConfig(
    description=u'My poorly configured LDAP host',
  ),


  'ldap:///dc=stroeder,dc=de': Web2LDAPConfig(
#    schema_uri='file:'+os.path.join(web2ldapcnf.etc_dir,os.path.join('web2ldap','localschema.ldif')),
  ),

  # This meant as an more complex example section
  # Adjust the settings to reflect your local LDAP installation
  'ldapi://%2Ftmp%2Fopenldap-socket/dc=stroeder,dc=de': Web2LDAPConfig(
    description=u'My Address Book',
    ssl_valid_dn=r'^/S=Stroeder/G=Michael/CN=Michael Stroeder/Email=michael@stroeder.com$',
    ssl_valid_idn=r'^/C=ZA/ST=Western Cape/L=Durbanville/O=Thawte/OU=Certificate Services/CN=Personal Freemail RSA 1999\.9\.16$',
    session_track_control=1,
    searchform_search_root_url=u'ldap:///dc=stroeder,dc=de??sub?(&(|(ou:dn:=Bizness)(ou:dn:=Kultur))(objectClass=organization)(hasSubordinates=TRUE))',
    search_attrs=[
      'cn','mail','sn','givenName','personalTitle','businessTitle',
      'o','ou','departmentNumber','employeeNumber',
      'telephoneNumber','homePhone','mobile',
      'c','st','l','streetAddress','roomNumber',
      'uid','uidNumber','description',
      'objectClass','entryUUID','organizationalStatus',
    ],
    read_template={
      # 'object class':'pathname of HTML template file'
      'germanBankArrangement':os.path.join(web2ldapcnf.templates_dir,'read_germanBankArrangement.html'),
      'musician':os.path.join(web2ldapcnf.templates_dir,'read_musician.html'),
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'account':os.path.join(web2ldapcnf.templates_dir,'read_account.html'),
      'organizationalPerson':os.path.join(web2ldapcnf.templates_dir,'read_inetOrgPerson.html'),
      'msPerson':os.path.join(web2ldapcnf.templates_dir,'read_msPerson.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'read_organization.html'),
      'msOrganization':os.path.join(web2ldapcnf.templates_dir,'read_msOrganization.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'read_posixAccount.html'),
      'eduPerson':os.path.join(web2ldapcnf.templates_dir,'read_eduPerson.html'),
      'pwdPolicy':os.path.join(web2ldapcnf.templates_dir,'read_pwdPolicy.html'),
      'sambaDomain':os.path.join(web2ldapcnf.templates_dir,'read_sambaDomain.html'),
      'sambaSamAccount':os.path.join(web2ldapcnf.templates_dir,'read_sambaSamAccount.html'),
      'sambaGroupMapping':os.path.join(web2ldapcnf.templates_dir,'read_sambaGroupMapping.html'),
    },
    # HTML template file used for displaying input forms for entries
    # of specific object class
    input_template={
      # 'object class':'pathname of HTML template file'
      'germanBankArrangement':os.path.join(web2ldapcnf.templates_dir,'inputform_germanBankArrangement.html'),
      'musician':os.path.join(web2ldapcnf.templates_dir,'inputform_musician.html'),
      'inetOrgPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_inetOrgPerson.html'),
      'account':os.path.join(web2ldapcnf.templates_dir,'inputform_account.html'),
      'msPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_msPerson.html'),
      'posixAccount':os.path.join(web2ldapcnf.templates_dir,'inputform_posixAccount.html'),
      'organization':os.path.join(web2ldapcnf.templates_dir,'inputform_organization.html'),
      'msOrganization':os.path.join(web2ldapcnf.templates_dir,'inputform_msOrganization.html'),
      'eduPerson':os.path.join(web2ldapcnf.templates_dir,'inputform_eduPerson.html'),
      'pwdPolicy':os.path.join(web2ldapcnf.templates_dir,'inputform_pwdPolicy.html'),
      'sambaDomain':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaDomain.html'),
      'sambaSamAccount':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaSamAccount.html'),
      'sambaGroupMapping':os.path.join(web2ldapcnf.templates_dir,'inputform_sambaGroupMapping.html'),
    },
    addform_entry_templates={
      u'stroeder.com Person':os.path.join(web2ldapcnf.templates_dir,'add_msperson.ldif'),
      u'stroeder.com Organization':os.path.join(web2ldapcnf.templates_dir,'add_msorganization.ldif'),
      u'User':os.path.join(web2ldapcnf.templates_dir,'add_user.ldif'),
      u'Org. Unit':os.path.join(web2ldapcnf.templates_dir,'add_orgunit.ldif'),
      u'Group':os.path.join(web2ldapcnf.templates_dir,'add_group.ldif'),
    },
#    login_default_mech='EXTERNAL',
#    passwd_hashtypes=['ssha',''],
#    schema_uri='ldapi://%2Ftmp%2Fopenldap-socket/cn=Subschema',
#    schema_uri='file:'+os.path.join(web2ldapcnf.etc_dir,os.path.join('web2ldap','localschema.ldif')),
  ),

  # Example for OpenLDAP's accesslog database
  'ldap:///cn=accesslog': Web2LDAPConfig(
    description=u'OpenLDAP accesslog',
    search_tdtemplate= {
      'auditsearch':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Search %(reqscope)s: %(reqdn)s<br>%(reqfilter)s',
      'auditbind':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s: %(reqsession)s<br>%(reqdn)s',
      'auditadd':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Entry %(reqdn)s added<br>by %(reqauthzid)s',
      'auditmodify':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Entry %(reqdn)s modified<br>by %(reqauthzid)s',
      'auditmodrdn':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Entry %(reqdn)s renamed to %(reqnewrdn)s,%(reqnewsuperior)s<br>by %(reqauthzid)s',
      'auditobject':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>by %(reqauthzid)s',
      'auditdelete':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Entry %(reqdn)s deleted<br>by %(reqauthzid)s',
      'auditabandon':r'<strong>%(reqtype)s</strong> %(reqstart)s session %(reqSession)s <br>&rArr; %(reqresult)s: %(reqMessage)s<br>Abandonded %(reqId)s by %(reqauthzid)s',
    },
    searchform_template= {
      u'_':os.path.join(web2ldapcnf.templates_dir,'searchform_accesslog.html'),
    },
    search_attrs=[
      'reqType','reqDN','reqAuthzID','reqStart','reqEnd','reqSession','reqResult',
    ],
  ),

  # Example for changelog database
  'ldap:///cn=changelog': Web2LDAPConfig(
    description=u'changelog',
    search_tdtemplate= {
      'changelogentry':r'No. %(changenumber)s at %(changetime)s<br>%(changetype)s %(targetdn)s (%(targetentryuuid)s)<br>by %(changeInitiatorsName)s',
    },
    searchform_template= {
      u'_':os.path.join(web2ldapcnf.templates_dir,'searchform_changelog.html'),
    },
    search_attrs=[
      'changeType','targetDN','targetEntryUUID','changeTime',
      'changeLogCookie','replicaIdentifier','replicationCSN','changeInitiatorsName',
    ],
    read_template={
      'changeLogEntry':os.path.join(web2ldapcnf.templates_dir,'read_changeLogEntry.html'),
    },
  ),

  # Example for cn=Monitor covering various server implementations
  'ldap:///cn=Monitor': Web2LDAPConfig(
    # Important workaround for OpenLDAP bug ITS#7053 !!!
    search_resultsperpage=0,
    search_tdtemplate= {
      # for OpenLDAP
      'monitorOperation':r'%(cn)s operations: %(monitorOpCompleted)s of %(monitorOpInitiated)s completed',
      'monitoredObject':r"""%(cn)s: %(monitoredInfo)s<br>
         %(monitorTimestamp)s %(namingContexts)s %(labeledURI)s<br>
         %(seeAlso)s
        """,
      'monitorContainer':r'%(cn)s - %(description)s',
      'monitorCounterObject':r'%(cn)s: %(monitorCounter)s',
      'monitorConnection':r"""Connection %(monitorConnectionNumber)s (LDAPv%(monitorConnectionProtocol)s):<br>
        %(monitorConnectionPeerAddress)s &raquo; %(monitorConnectionListener)s<br>
        %(monitorConnectionAuthzDN)s<br>
        (ops r: %(monitorConnectionOpsReceived)s, e: %(monitorConnectionOpsExecuting)s, p: %(monitorConnectionOpsPending)s, c: %(monitorConnectionOpsCompleted)s)
        """,
      # for OpenDS/OpenDJ
    },
    read_template={
      # for OpenLDAP
      'monitorConnection':os.path.join(web2ldapcnf.templates_dir,'read_monitorConnection.html'),
      # for OpenDS/OpenDJ
    },
  ),

  # Special example for IBM Directory Server 5.1
  'ldap://localhost:7389': Web2LDAPConfig(
    requested_attrs=[
      'ibm-audit','ibm-auditAdd','ibm-auditBind','ibm-auditDelete','ibm-auditExtOpEvent','ibm-auditFailedOpOnly','ibm-auditLog','ibm-auditModify','ibm-auditModifyDN','ibm-auditSearch','ibm-auditUnbind'
    ],
  ),

  # For dynamic configuration via cn=config
  'ldap:///cn=config': Web2LDAPConfig(
    description=u"Configuration backend",
    search_tdtemplate={
      # for OpenLDAP
      'olcModuleList':r'Modules from %(olcModulePath)s',
      'olcDatabaseConfig':r'Database <em>%(olcDatabase)s</em> suffix %(olcSuffix)s',
      'olcSchemaConfig':r'Schema <em>%(cn)s</em>',
      'olcOverlayConfig':r'Overlay <em>%(olcOverlay)s</em>',
      # for OpenDS/OpenDJ
      'ds-cfg-backend':r'Backend <em>%(ds-cfg-backend-id)s</em>, suffix <em>%(ds-cfg-base-dn)s</em><br>Class: %(ds-cfg-java-class)s',
      'ds-cfg-local-db-index':r'Index for <em>%(ds-cfg-attribute)s</em>: <em>%(ds-cfg-index-type)s</em>',
      'ds-cfg-replication-server':r"""
        Replication server-id <em>%(ds-cfg-replication-server-id)s</em>:<br>
        Replication server(s): %(ds-cfg-replication-server)s
      """,
      'ds-cfg-replication-domain':r"""
        Replication domain <em>%(cn)s</em>:<br>
        Server-ID: %(ds-cfg-server-id)s<br>
        Base-DN: %(ds-cfg-base-dn)s<br>
        Replication server(s): %(ds-cfg-replication-server)s
      """,
    },
    read_template={
      # for OpenLDAP
      'olcGlobal':os.path.join(web2ldapcnf.templates_dir,'inputform_olcGlobal.html'),
      'olcDatabaseConfig':os.path.join(web2ldapcnf.templates_dir,'inputform_olcDatabaseConfig.html'),
    },
    input_template={
      # for OpenLDAP
      'olcGlobal':os.path.join(web2ldapcnf.templates_dir,'inputform_olcGlobal.html'),
      'olcDatabaseConfig':os.path.join(web2ldapcnf.templates_dir,'inputform_olcDatabaseConfig.html'),
    },
    # HTML template strings used to display the superior entry
    # in the input form when adding/modifying entries
    inputform_supentrytemplate={
      # for OpenLDAP
      'olcDatabaseConfig':r"""<table>
          <tr><td>Database name:</td><td>%(olcDatabase)s</td></tr>
          <tr><td>Suffix:</td><td>%(olcSuffix)s</td></tr>
        </table>
      """,
    },
    addform_entry_templates={
      # for OpenLDAP
      u'Database - Adress book (back-hdb)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_AddressBook.ldif'),
      u'Database - Unix Users (back-hdb)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_UnixUsers.ldif'),
      u'Database - Accesslog (back-hdb)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_accesslog.ldif'),
      u'Database - LDAP-Proxy (back-ldap)':os.path.join(web2ldapcnf.templates_dir,'add_olcLdapConfig.ldif'),
      u'Overlay - Syncrepl-Provider (slapo-syncprov)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_accesslog.ldif'),
      u'Overlay - Accesslog (slapo-accesslog)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_accesslog.ldif'),
      u'Overlay - Accesslog (slapo-accesslog)':os.path.join(web2ldapcnf.templates_dir,'add_olcHdbConfig_accesslog.ldif'),
      u'Schema config':os.path.join(web2ldapcnf.templates_dir,'add_olcSchemaConfig.ldif'),
    },
  ),

}
