/* This file is part of the YAZ toolkit.
 * Copyright (C) 1995-2012 Index Data
 * See the file LICENSE for details.
 */
/**
 * \file
 * \brief Implements RPN to CQL conversion
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <yaz/rpn2cql.h>
#include <yaz/xmalloc.h>
#include <yaz/diagbib1.h>
#include <yaz/z-core.h>
#include <yaz/wrbuf.h>
#include <yaz/logrpn.h> /* For yaz_prox_unit_name() */

static void wrbuf_vputs(const char *buf, void *client_data)
{
    wrbuf_write((WRBUF) client_data, buf, strlen(buf));
}

static const char *lookup_index_from_string_attr(Z_AttributeList *attributes)
{
    int j;
    int server_choice = 1;
    for (j = 0; j < attributes->num_attributes; j++)
    {
        Z_AttributeElement *ae = attributes->attributes[j];
        if (*ae->attributeType == 1) /* use attribute */
        {
            if (ae->which == Z_AttributeValue_complex)
            {
                Z_ComplexAttribute *ca = ae->value.complex;
                int i;
                for (i = 0; i < ca->num_list; i++)
                {
                    Z_StringOrNumeric *son = ca->list[i];
                    if (son->which == Z_StringOrNumeric_string)
                        return son->u.string;
                }
            }
            server_choice = 0; /* not serverChoice because we have use attr */
        }
    }
    if (server_choice)
        return "cql.serverChoice";
    return 0;
}

static const char *lookup_relation_index_from_attr(Z_AttributeList *attributes)
{
    int j;
    for (j = 0; j < attributes->num_attributes; j++)
    {
        Z_AttributeElement *ae = attributes->attributes[j];
        if (*ae->attributeType == 2) /* relation attribute */
        {
            if (ae->which == Z_AttributeValue_numeric)
            {
                /* Only support for numeric relation */
                Odr_int *relation = ae->value.numeric;
                /* map this numeric to representation in CQL */
                switch (*relation)
                {
                    /* Unsure on whether this is the relation attribute constants? */
                case Z_ProximityOperator_Prox_lessThan: 
                    return "<";
                case Z_ProximityOperator_Prox_lessThanOrEqual: 
                    return "<="; 
                case Z_ProximityOperator_Prox_equal: 
                    return "="; 
                case Z_ProximityOperator_Prox_greaterThanOrEqual: 
                    return ">="; 
                case Z_ProximityOperator_Prox_greaterThan: 
                    return ">"; 
                case Z_ProximityOperator_Prox_notEqual: 
                    return "<>"; 
                case 100: 
                    /* phonetic is not supported in CQL */
                    return 0; 
                case 101: 
                    /* stem is not supported in CQL */
                    return 0; 
                case 102: 
                    /* relevance is supported in CQL, but not implemented yet */
                    return 0; 
                default:
                    /* Invalid relation */
                    return 0;
                }
            }
            else {
                /*  Can we have a complex relation value?
                    Should we implement something?
                */
            }
        }
    }
    return "=";
}

static int rpn2cql_attr(cql_transform_t ct,
                        Z_AttributeList *attributes, WRBUF w)
{
    const char *relation = cql_lookup_reverse(ct, "relation.", attributes);
    const char *index = cql_lookup_reverse(ct, "index.", attributes);
    const char *structure = cql_lookup_reverse(ct, "structure.", attributes);

    /* if transform (properties) do not match, we'll just use a USE string attribute (bug #2978) */
    if (!index)
        index = lookup_index_from_string_attr(attributes);

    /* Attempt to fix bug #2978: Look for a relation attribute */
    if (!relation) 
        relation = lookup_relation_index_from_attr(attributes);

    if (!index)
    {
        cql_transform_set_error(ct,
                                YAZ_BIB1_UNSUPP_USE_ATTRIBUTE, 0);
        return -1;
    }
    /* for serverChoice we omit index+relation+structure */
    if (strcmp(index, "cql.serverChoice"))
    {
        wrbuf_puts(w, index);
        if (relation)
        {
            if (!strcmp(relation, "exact"))
                relation = "==";
            else if (!strcmp(relation, "eq"))
                relation = "=";
            else if (!strcmp(relation, "le"))
                relation = "<=";
            else if (!strcmp(relation, "ge"))
                relation = ">=";
            /* Missing mapping of not equal, phonetic, stem and relevance */
            wrbuf_puts(w, relation);
        }
        else
            wrbuf_puts(w, "=");

        if (structure)
        {
            if (strcmp(structure, "*"))
            {
                wrbuf_puts(w, "/");
                wrbuf_puts(w, structure);
                wrbuf_puts(w, " ");
            }
        }
    }
    return 0;
}

static Odr_int lookup_truncation(Z_AttributeList *attributes)
{
    int j;
    for (j = 0; j < attributes->num_attributes; j++)
    {
        Z_AttributeElement *ae = attributes->attributes[j];
        if (*ae->attributeType == 5) /* truncation attribute */
        {
            if (ae->which == Z_AttributeValue_numeric)
                return *(ae->value.numeric);
        }
    }
    /* No truncation specified */
    return 0;
};

static int rpn2cql_simple(cql_transform_t ct,
                          void (*pr)(const char *buf, void *client_data),
                          void *client_data,
                          Z_Operand *q, WRBUF w)
{
    int ret = 0;
    if (q->which != Z_Operand_APT)
    {
        ret = -1;
        cql_transform_set_error(ct, YAZ_BIB1_RESULT_SET_UNSUPP_AS_A_SEARCH_TERM, 0);
    }
    else
    {
        Z_AttributesPlusTerm *apt = q->u.attributesPlusTerm;
        Z_Term *term = apt->term;
        const char *sterm = 0;
        size_t lterm = 0;
        Odr_int trunc = lookup_truncation(apt->attributes);
        size_t i;

        wrbuf_rewind(w);
        ret = rpn2cql_attr(ct, apt->attributes, w);

        switch (term->which)
        {
        case Z_Term_general:
            lterm = term->u.general->len;
            sterm = (const char *) term->u.general->buf;
            break;
        case Z_Term_numeric:
            wrbuf_printf(w, ODR_INT_PRINTF, *term->u.numeric);
            break;
        case Z_Term_characterString:
            sterm = term->u.characterString;
            lterm = strlen(sterm);
            break;
        default:
            cql_transform_set_error(ct, YAZ_BIB1_TERM_TYPE_UNSUPP, 0);
            return -1;
        }

        if (trunc <= 3 || trunc == 100 || trunc == 102 || trunc == 104)
        {
            for (i = 0 ; i < lterm; i++)
                if (strchr(" ()=></", sterm[i]))
                    break;
            wrbuf_puts(w, "\"");
            if (trunc == 2 || trunc == 3)
                wrbuf_puts(w, "*");
            for (i = 0; i < lterm; i++)
            {
                if (sterm[i] == '\\' && i < lterm - 1)
                {
                    i++;
                    if (strchr("*?\"\\", sterm[i]))
                        wrbuf_putc(w, '\\');
                    wrbuf_putc(w, sterm[i]);
                }
                else if (trunc == 102 && sterm[i] == '.' && sterm[i+1] == '*')
                {
                    wrbuf_putc(w, '*');
                    i++;
                }
                else if (trunc == 102 && sterm[i] == '.')
                    wrbuf_putc(w, '?');
                else if (trunc == 104 && sterm[i] == '?')
                    wrbuf_putc(w, '*');
                else if (trunc == 104 && sterm[i] == '#')
                    wrbuf_putc(w, '?');
                else if (strchr("*?\"", sterm[i]))
                {
                    wrbuf_putc(w, '\\');
                    wrbuf_putc(w, sterm[i]);
                }
                else
                    wrbuf_putc(w, sterm[i]);
            }
            if (trunc == 1 || trunc == 3)
                wrbuf_puts(w, "*");
            wrbuf_puts(w, "\"");
        }
        else
        {
            cql_transform_set_error(
                ct, YAZ_BIB1_UNSUPP_TRUNCATION_ATTRIBUTE, 0);
            ret = -1;
        }
        if (ret == 0)
            pr(wrbuf_cstr(w), client_data);
    }
    return ret;
}


static int rpn2cql_structure(cql_transform_t ct,
                             void (*pr)(const char *buf, void *client_data),
                             void *client_data,
                             Z_RPNStructure *q, int nested,
                             WRBUF w)
{
    if (q->which == Z_RPNStructure_simple)
        return rpn2cql_simple(ct, pr, client_data, q->u.simple, w);
    else
    {
        Z_Operator *op = q->u.complex->roperator;
        Z_ProximityOperator *prox;
        int r;

        if (nested)
            pr("(", client_data);

        r = rpn2cql_structure(ct, pr, client_data, q->u.complex->s1, 1, w);
        if (r)
            return r;
        switch(op->which)
        {
        case  Z_Operator_and:
            pr(" and ", client_data);
            break;
        case  Z_Operator_or:
            pr(" or ", client_data);
            break;
        case  Z_Operator_and_not:
            pr(" not ", client_data);
            break;
        case  Z_Operator_prox: {
            pr(" prox", client_data);
            prox = op->u.prox;
            /* No way to express Odr_bool *exclusion -- ignore it */
            if (prox->distance) {
                char buf[21]; /* Enough for any 64-bit int */
                char *op2name[6] = { "<", "<=", "=", ">=", ">","<>" };
                pr("/distance", client_data);
                if (!prox->relationType ||
                    *prox->relationType < Z_ProximityOperator_Prox_lessThan ||
                    *prox->relationType > Z_ProximityOperator_Prox_notEqual) {
                    cql_transform_set_error(ct, YAZ_BIB1_UNSUPP_SEARCH,
                        "unrecognised proximity relationType");
                    return -1;
                }
                pr(op2name[*prox->relationType-1], client_data);
                sprintf(buf, "%ld", (long) *prox->distance);
                pr(buf, client_data);
            }
            if (prox->ordered) {
                if (*prox->ordered) {
                    pr("/ordered", client_data);
                } else {
                    pr("/unordered", client_data);
                }
            }
            if (prox->which != Z_ProximityOperator_known ||
                *prox->u.known != Z_ProxUnit_word) {
                    pr("/unit=", client_data);
                    pr(yaz_prox_unit_name(prox), client_data);
            }
            pr(" ", client_data);
            break;
        }
        }
        r = rpn2cql_structure(ct, pr, client_data, q->u.complex->s2, 1, w);
        if (nested)
            pr(")", client_data);
        return r;
    }
}

int cql_transform_rpn2cql_stream(cql_transform_t ct,
                                 void (*pr)(const char *buf, void *client_data),
                                 void *client_data,
                                 Z_RPNQuery *q)
{
    int r;
    WRBUF w = wrbuf_alloc();
    cql_transform_set_error(ct, 0, 0);
    r = rpn2cql_structure(ct, pr, client_data, q->RPNStructure, 0, w);
    wrbuf_destroy(w);
    return r;
}


int cql_transform_rpn2cql_wrbuf(cql_transform_t ct,
                                WRBUF w,
                                Z_RPNQuery *q)
{
    return cql_transform_rpn2cql_stream(ct, wrbuf_vputs, w, q);
}

/*
 * Local variables:
 * c-basic-offset: 4
 * c-file-style: "Stroustrup"
 * indent-tabs-mode: nil
 * End:
 * vim: shiftwidth=4 tabstop=8 expandtab
 */

